use core:lang;

/**
 * Explicit barriers.
 *
 * We support three types of barriers:
 * - acquire
 * - release
 * - full (acquire + release, sequentially consistent)
 *
 * The runtime takes them into account from two places:
 * 1. From SrcBarrier instances inserted as metadata in the listing.
 * 2. From the CodeHints class. When a function call without a Barrier instance is encountered,
 *    the function 'functionBarrier' is called to determine the semantics. This allows an
 *    implementation to hide calls to internal helper functions, and properly treat library- or
 *    external functions as hidden. This will not be done for functions that are patched
 *    (as decided by 'patchExternal'). These barriers are inserted right before the function
 *    call. Thus, an implementation that inserts barriers itself should return "none"
 *    for the relevant functions in CodeHints.
 */
enum Barrier : bitmask {
	none = 0,
	acquire = 1,
	release = 2,
	full = 3,

	// Is this a function call where the call itself should be shown?
	showCall = 32,
}

/**
 * A barrier located in source code. Also provides location information.
 */
class SrcBarrier {
	init(Barrier type, SrcPos pos) {
		init {
			type = type;
			pos = pos;
		}
	}

	Barrier type;

	SrcPos pos;

	void toS(StrBuf to) : override {
		to << "Barrier: " << type << "@" << pos;
	}
}
