#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2025 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Diablo 2
# send your bug reports to vv221@dotslashplay.it
###

script_version=20250628.5

PLAYIT_COMPATIBILITY_LEVEL='2.32'

GAME_ID='diablo-2'
GAME_NAME='Diablo II'

ARCHIVE_BASE_EN_0_NAME='Installer Tome.mpq'
ARCHIVE_BASE_EN_0_MD5='f5cf9b3685eacd506e9ddb521751cdfc'
ARCHIVE_BASE_EN_0_SIZE='1526589'
ARCHIVE_BASE_EN_0_VERSION='1.14b-blizzard'
ARCHIVE_BASE_EN_0_URL='https://eu.battle.net/download/getLegacy?os=win&locale=enUS&product=D2DV'

ARCHIVE_BASE_FR_0_NAME='Installer Tome.mpq'
ARCHIVE_BASE_FR_0_MD5='04a89652d44e7acf126d4d108bdcff87'
ARCHIVE_BASE_FR_0_SIZE='1524530'
ARCHIVE_BASE_FR_0_VERSION='1.14b-blizzard'
ARCHIVE_BASE_FR_0_URL='https://eu.battle.net/download/getLegacy?os=win&locale=frFR&product=D2DV'

CONTENT_PATH_DEFAULT='.'
CONTENT_GAME_BIN_PATH='PC'
## Diablo II.exe differs between the English and the French build of the game,
## for now we assume the differences are not critical and one can be used in place of the other.
CONTENT_GAME_BIN_FILES='
binkw32.dll
ijl11.dll
SmackW32.dll
Diablo II.exe
Game.exe
D2.LNG'
CONTENT_GAME_DATA_PATH='Common'
CONTENT_GAME_DATA_FILES='
d2data.mpq
d2sfx.mpq
d2speech.mpq
d2video.mpq'
CONTENT_GAME_DATA_MUSIC_PATH='Common'
CONTENT_GAME_DATA_MUSIC_FILES='
d2music.mpq'
CONTENT_GAME_DATA_SHARED_PATH='Common'
CONTENT_GAME_DATA_SHARED_FILES='
d2char.mpq
patch_d2.mpq'

WINE_PERSISTENT_DIRECTORIES='
users/${USER}/Saved Games/Diablo II'

APP_MAIN_EXE='Game.exe'
APP_MAIN_ICON='PC/Diablo II.exe'

PACKAGES_LIST='
PKG_BIN
PKG_DATA
PKG_DATA_MUSIC
PKG_DATA_SHARED'

PKG_BIN_ARCH='32'
PKG_BIN_DEPENDENCIES_SIBLINGS='
PKG_DATA_BASE'
PKG_BIN_DEPENDENCIES_GSTREAMER_PLUGINS='
video/x-msvideo'

PKG_DATA_BASE_ID="${GAME_ID}-data"
PKG_DATA_ID_EN="${PKG_DATA_BASE_ID}-en"
PKG_DATA_ID_FR="${PKG_DATA_BASE_ID}-fr"
PKG_DATA_PROVIDES="
$PKG_DATA_BASE_ID"
PKG_DATA_DESCRIPTION='data'
PKG_DATA_DEPENDENCIES_SIBLINGS='
PKG_DATA_MUSIC
PKG_DATA_SHARED'

PKG_DATA_MUSIC_ID="${PKG_DATA_BASE_ID}-music"
PKG_DATA_MUSIC_DESCRIPTION='music'

PKG_DATA_SHARED_ID="${PKG_DATA_BASE_ID}-shared"
PKG_DATA_SHARED_DESCRIPTION='data shared between the base game and the expansion'

# Set the list of requirements to extract the archive contents.

REQUIREMENTS_LIST="${REQUIREMENTS_LIST:-}
smpq"

# Load common functions

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run the default initialization actions

initialization_default "$@"

# Check for the presence of required CD key files.

## During a regular install of Diablo 2, the CD key is stored encrypted into d2data.mpq, in the files under the "data/global/sfx/cursor" directory.
## In order to build a package that includes the CD key (required, or the game will not launch),
## you need to provide the following files in addition to the installer .mpq:
## - curindx.wav
## - wavindx.wav
##
## They can be extracted from a regular Diablo 2 install with:
## smpq --extract d2data.mpq data/global/sfx/cursor/curindx.wav data/global/sfx/cursor/wavindx.wav
ARCHIVE_REQUIRED_CDKEY_1_NAME='curindx.wav'
ARCHIVE_REQUIRED_CDKEY_2_NAME='wavindx.wav'
archive_initialize_required \
	'ARCHIVE_CDKEY_1' \
	'ARCHIVE_REQUIRED_CDKEY_1'
archive_initialize_required \
	'ARCHIVE_CDKEY_2' \
	'ARCHIVE_REQUIRED_CDKEY_2'

# Extract game data

archive_path=$(archive_path "$(current_archive)")
archive_path_cdkey_1=$(archive_path 'ARCHIVE_CDKEY_1')
archive_path_cdkey_2=$(archive_path 'ARCHIVE_CDKEY_2')
mkdir --parents "${PLAYIT_WORKDIR}/gamedata"
information_archive_data_extraction "$archive_path"
(
	cd "${PLAYIT_WORKDIR}/gamedata"
	smpq --extract "$archive_path"

	# Update the .mpq file
	(
		cd 'PC-100'
		smpq --append ../Common/d2data.mpq ./*
	)
	(
		cd 'Files'
		mkdir --parents 'data/global/sfx/cursor'
		cp --dereference \
			"$archive_path_cdkey_1" \
			"$archive_path_cdkey_2" \
			'data/global/sfx/cursor'
		smpq --append ../Common/d2data.mpq \
			data/global/sfx/cursor/curindx.wav \
			data/global/sfx/cursor/wavindx.wav
		rm --force --recursive 'data/global/sfx'
	)
)

# Include game data

content_inclusion_icons 'PKG_DATA_SHARED'
content_inclusion_default

# Write launchers

launchers_generation 'PKG_BIN'

# Build packages

packages_generation
print_instructions

# Clean up

working_directory_cleanup

exit 0
