"""ACWF-benchmark.

Do equation-of-state calculation for the 10 reference systems from
the AiiDA common workflows (ACWF) benchmark:
DIAMOND, FCC, SC, BCC, XO3, XO, X4O6, XO2, X4O10, X2O.
"""
from __future__ import annotations
import json
from pathlib import Path
from typing import Any

import numpy as np
from ase import Atoms
from ase.data import atomic_numbers
from gpaw.atom.check import all_names
from gpaw.mpi import world
from gpaw.new.ase_interface import GPAW
from gpaw.dft import Parameters


def eos(atoms: Atoms,
        npoints: int = 7,
        maxstrain: float = 0.02) -> dict:
    """Do EOS calculation and fit result."""
    cell_cv = atoms.get_cell()

    energies = []
    strains = np.linspace(-maxstrain, maxstrain, npoints)
    for i, strain in enumerate(strains):
        atoms.set_cell((1.0 + strain) * cell_cv, scale_atoms=True)
        e = atoms.get_potential_energy()
        energies.append(e)

    strain0, e0, d2eds2 = fit(strains, energies)
    v0 = abs(np.linalg.det(cell_cv))
    volume = v0 * (1 + strain0)**3

    return dict(
        volume=volume,
        strains=strains.tolist(),
        energies=energies,
        energy=e0,
        strain=strain0,
        d2eds2=d2eds2)


def fit(strains: np.ndarray,
        energies: np.ndarray | list[float]) -> tuple[float, float, float]:
    """Fit energies.

    ::

                    3
                   ---   n        1
     f(s) = p(x) = >  c x,  x = -----,
                   --- n        1 + s
                   n=0

     df/ds = dp/dx dx/ds,

      2    2   2    2       2
     d f/ds = d p/dx (dx/ds).

    """

    x = 1 / (1 + strains)
    p0 = np.poly1d(np.polyfit(x, energies, 3))
    p1 = np.polyder(p0, 1)
    p2 = np.polyder(p1, 1)
    for x0 in np.roots(p1):
        if isinstance(x0, float) and x0 > 0 and p2(x0) > 0:
            break
    else:  # no break
        raise ValueError
    strain = 1 / x0 - 1
    e0 = p0(x0)
    d2eds2 = p2(x0) * x0**2
    return strain, e0, d2eds2


def run_eos_calculation(structure: str, symbol: str, **params) -> dict:
    """Create GPAW calculator and do EOS calculation."""
    atoms = reference_structure(symbol, structure)
    builder = Parameters(**params).dft_component_builder(atoms, log=None)
    params['kpts'] = builder.ibz.bz.size_c  # type: ignore
    params['gpts'] = builder.grid.size
    params.pop('h', None)
    atoms.calc = GPAW(txt=f'{builder.mode}-{structure}.txt', **params)
    data = eos(atoms)
    return data


def workflow() -> None:
    """Submit PW and LCAO calculations.

    101 PAW-potentials, 10 crystals, PW and LCAO: 2020 calculations.
    """
    from myqueue.workflow import run
    symbol, _, setup_name = Path().absolute().name.partition('.')
    for x in acwf_structures:
        run(function=work, args=[x, symbol, setup_name, 'pw'],
            cores=24, tmax='1h', name=f'pw-{x}')
        run(function=work, args=[x, symbol, setup_name, 'lcao'],
            cores=24, tmax='5h', name=f'lcao-{x}', restart=2)


def work(structure: str,
         symbol: str,
         setup_name: str = '',
         mode: str = 'pw'):
    """Do single EOS calculations with PBE."""
    params: dict[str, Any] = dict(
        xc='PBE',
        occupations=dict(name='fermi-dirac', width=0.0612),
        kpts=dict(density=6.0),
        convergence=dict(density=1e-6))
    if mode == 'pw':
        params['mode'] = dict(name='pw', ecut=1000)
    else:
        params['mode'] = 'lcao'
        params['basis'] = 'dzp'
        params['h'] = 0.12
    if setup_name:
        params['setups'] = {symbol: setup_name}
    data = run_eos_calculation(
        structure,
        symbol,
        **params)
    if world.rank == 0:
        Path(f'{mode}-{structure}.json').write_text(
            json.dumps(data, indent=1))


def create_folders() -> None:
    """Create folder for all 101 PAW-potentials."""
    for name in all_names:
        Path(name).mkdir(exist_ok=True)


def collect_data() -> None:
    """Collect everything need for a table on the web-page."""
    results = {}
    for name in all_names:
        result: dict[str, list[tuple[str, float]]] = {'pw': [], 'lcao': []}
        for mode in ['pw', 'lcao']:
            for path in Path(name).glob(f'{mode}-*.json'):
                structure = path.stem.split('-')[1]
                strain = json.loads(path.read_text())['strain']
                result[mode].append((structure, strain))
        results[name] = result
    Path('acwf-results.json').write_text(json.dumps(results))


def reference_structure(symbol: str,
                        name: str) -> Atoms:
    """Create one of the ACWF structures with the WIEN2K volume."""
    atoms = acwf_structures[name].copy()
    x = (volumes[symbol][name] / atoms.get_volume())**(1 / 3)
    if symbol == 'O' and name in ['X4O10', 'XO']:
        x *= 2**(1 / 3)
    atoms.set_cell(atoms.cell * x, scale_atoms=True)
    atoms.numbers[atoms.numbers == 0] = atomic_numbers[symbol]
    return atoms


acwf_structures = {
    'XO3': Atoms(
        'XO3',
        cell=[[1.0, 0.0, 0.0], [0.0, 1.0, 0.0], [0.0, 0.0, 1.0]],
        scaled_positions=[[0.0, 0.0, 0.0], [0.5, 0.0, 0.0],
                          [0.0, 0.5, 0.0], [0.0, 0.0, 0.5]],
        pbc=True),
    'XO': Atoms(
        'XO',
        cell=[[0.0, 1.0, 1.0], [1.0, 0.0, 1.0], [1.0, 1.0, 0.0]],
        scaled_positions=[[0.0, 0.0, -0.0], [0.5, -0.5, 0.5]],
        pbc=True),
    'X4O6': Atoms(
        'X4O6',
        cell=[[1.0, 0.0, 0.0], [0.0, 1.0, 0.0], [0.0, 0.0, 1.0]],
        scaled_positions=[[0.25, 0.25, 0.25], [0.75, 0.75, 0.25],
                          [0.75, 0.25, 0.75], [0.25, 0.75, 0.75],
                          [0.5, 0.0, 0.0], [0.0, 0.5, 0.0],
                          [0.0, 0.0, 0.5], [0.5, 0.5, 0.0],
                          [0.5, 0.0, 0.5], [0.0, 0.5, 0.5]],
        pbc=True),
    'XO2': Atoms(
        'XO2',
        cell=[[0.0, 1.0, 1.0], [1.0, 0.0, 1.0], [1.0, 1.0, 0.0]],
        scaled_positions=[[0.0, 0.0, -0.0],
                          [0.25, 1.25, 0.25],
                          [-0.25, 0.75, 0.75]],
        pbc=True),
    'X4O10': Atoms(
        'X4O10',
        cell=[[1.0, 0.0, 0.0], [0.0, 1.0, 0.0], [0.0, 0.0, 1.0]],
        scaled_positions=[[0.75, 0.75, 0.75], [0.25, 0.25, 0.75],
                          [0.25, 0.75, 0.25], [0.75, 0.25, 0.25],
                          [0.25, 0.75, 0.75], [0.75, 0.25, 0.75],
                          [0.75, 0.75, 0.25], [0.25, 0.25, 0.25],
                          [0.5, 0.5, 0.0], [0.5, 0.0, 0.5],
                          [0.0, 0.5, 0.5], [0.5, 0.0, 0.0],
                          [0.0, 0.5, 0.0], [0.0, 0.0, 0.5]],
        pbc=True),
    'X2O': Atoms(
        'X2O',
        cell=[[0.0, 1.0, 1.0], [1.0, 0.0, 1.0], [1.0, 1.0, 0.0]],
        scaled_positions=[[0.25, 1.25, 0.25],
                          [-0.25, 0.75, 0.75],
                          [0.0, 0.0, -0.0]],
        pbc=True),
    'Diamond': Atoms(
        'X2',
        cell=[[0.0, 1.0, 1.0], [1.0, 0.0, 1.0], [1.0, 1.0, 0.0]],
        scaled_positions=[[0.0, 0.0, -0.0], [0.25, 0.25, 0.25]],
        pbc=True),
    'FCC': Atoms(
        'X',
        cell=[[0.0, 1.0, 1.0], [1.0, 0.0, 1.0], [1.0, 1.0, 0.0]],
        scaled_positions=[[0.0, 0.0, -0.0]],
        pbc=True),
    'SC': Atoms(
        'X',
        cell=[[1.0, 0.0, 0.0], [0.0, 1.0, 0.0], [0.0, 0.0, 1.0]],
        scaled_positions=[[0.0, 0.0, 0.0]],
        pbc=True),
    'BCC': Atoms(
        'X',
        cell=[[-1.0, 1.0, 1.0], [1.0, -1.0, 1.0], [1.0, 1.0, -1.0]],
        scaled_positions=[[-0.0, 0.0, 0.0]],
        pbc=True)}


volumes: dict[str, dict[str, float]] = {
    'Ac': {
        'BCC': 45.943689065180855,
        'Diamond': 129.76193763607355,
        'FCC': 45.55065392302246,
        'SC': 49.82906214069011,
        'X2O': 80.3766777041316,
        'X4O6': 203.326117907875,
        'X4O10': 207.76797985303918,
        'XO': 38.992350681601565,
        'XO2': 52.96179138555256,
        'XO3': 91.44195279647013
    },
    'Ag': {
        'BCC': 17.98159943412582,
        'Diamond': 60.138447753538145,
        'FCC': 17.83856038211966,
        'SC': 20.807816653747608,
        'X2O': 48.38808819777706,
        'X4O6': 131.2721383029882,
        'X4O10': 149.92799548230082,
        'XO': 25.50923480620743,
        'XO2': 34.38335555693807,
        'XO3': 65.23416320682753
    },
    'Al': {
        'BCC': 16.925656872192757,
        'Diamond': 55.25812791788162,
        'FCC': 16.49535905981626,
        'SC': 20.155139416208016,
        'X2O': 46.21663447693852,
        'X4O6': 98.4792078105453,
        'X4O10': 126.14348555770839,
        'XO': 22.456524937543783,
        'XO2': 26.300916999476847,
        'XO3': 49.74390043966775
    },
    'Am': {
        'BCC': 16.191064699425894,
        'Diamond': 38.92606088376118,
        'FCC': 17.363741775723295,
        'SC': 16.11661629141215,
        'X2O': 45.419636119060456,
        'X4O6': 144.59436691493505,
        'X4O10': 148.57639001361287,
        'XO': 26.66553580766982,
        'XO2': 36.96323694007846,
        'XO3': 67.85355676723577
    },
    'Ar': {
        'BCC': 53.35453138568846,
        'Diamond': 197.20372578528713,
        'FCC': 52.276383334536746,
        'SC': 65.24068819085674,
        'X2O': 113.25843565850016,
        'X4O6': 206.36298900234982,
        'X4O10': 185.38977453731096,
        'XO': 39.43160913365185,
        'XO2': 51.33857264225451,
        'XO3': 68.32594446494795
    },
    'As': {
        'BCC': 19.052151090336626,
        'Diamond': 57.03345638956617,
        'FCC': 19.317577518973586,
        'SC': 20.367749732901732,
        'X2O': 47.81293031010519,
        'X4O6': 130.57910836008148,
        'X4O10': 144.03087762987815,
        'XO': 27.191332237135178,
        'XO2': 31.49881160785104,
        'XO3': 47.319197509869724
    },
    'At': {
        'BCC': 40.007256561713156,
        'Diamond': 133.82441316731246,
        'FCC': 39.030666540798215,
        'SC': 46.14406680206717,
        'X2O': 77.79112798187565,
        'X4O6': 177.486626152432,
        'X4O10': 169.0902164530073,
        'XO': 40.774272420246774,
        'XO2': 43.83458777704999,
        'XO3': 71.71609632447499
    },
    'Au': {
        'BCC': 18.042083697768916,
        'Diamond': 58.541509746305394,
        'FCC': 17.97894940070536,
        'SC': 20.769602790294257,
        'X2O': 49.45298956153372,
        'X4O6': 134.26817861354488,
        'X4O10': 150.36198987822257,
        'XO': 26.881873150878082,
        'XO2': 35.00356102786805,
        'XO3': 63.961952949423605
    },
    'B': {
        'BCC': 6.139355211328122,
        'Diamond': 16.626316908817323,
        'FCC': 5.891550054407846,
        'SC': 6.7004880836453475,
        'X2O': 30.02852147513494,
        'X4O6': 71.97017487676789,
        'X4O10': 108.7466438012355,
        'XO': 14.710476771224293,
        'XO2': 20.373538745231247,
        'XO3': 33.628829190051064
    },
    'Ba': {
        'BCC': 63.30541055553059,
        'Diamond': 113.16998237960502,
        'FCC': 64.11404100611098,
        'SC': 61.60561802807037,
        'X2O': 79.89102772639173,
        'X4O6': 233.84319295075028,
        'X4O10': 227.20804659070285,
        'XO': 43.30590522313118,
        'XO2': 60.342062817349294,
        'XO3': 91.21675807848992
    },
    'Be': {
        'BCC': 7.815756001821475,
        'Diamond': 29.37398729097643,
        'FCC': 7.8716206738707015,
        'SC': 10.26723560161619,
        'X2O': 26.833686313645984,
        'X4O6': 77.59929916270295,
        'X4O10': 114.64908157675103,
        'XO': 12.124123532413858,
        'XO2': 22.18537049044565,
        'XO3': 38.99206686558694
    },
    'Bi': {
        'BCC': 31.634667577446024,
        'Diamond': 96.90223944114533,
        'FCC': 31.810468127145906,
        'SC': 35.20505822838321,
        'X2O': 69.24164021274835,
        'X4O6': 174.87946592197846,
        'X4O10': 177.94746787410855,
        'XO': 36.02504757390335,
        'XO2': 42.07312443213904,
        'XO3': 74.04342790643858
    },
    'Br': {
        'BCC': 26.784212478063857,
        'Diamond': 86.17112439244427,
        'FCC': 26.417536335741566,
        'SC': 29.8278511781182,
        'X2O': 61.628473750026465,
        'X4O6': 147.33738792781116,
        'X4O10': 155.97272615705282,
        'XO': 31.106626065259217,
        'XO2': 37.01371722315619,
        'XO3': 58.256274759446185
    },
    'C': {
        'BCC': 6.685722025660118,
        'Diamond': 11.391522946768983,
        'FCC': 7.321634771212951,
        'SC': 5.582211299597029,
        'X2O': 28.554339698563062,
        'X4O6': 84.53807260317583,
        'X4O10': 117.2300823463996,
        'XO': 15.70175180655379,
        'XO2': 22.756394307135132,
        'XO3': 32.919218490588555
    },
    'Ca': {
        'BCC': 42.15047346676994,
        'Diamond': 159.96939544107403,
        'FCC': 42.19429685750488,
        'SC': 43.584913194130294,
        'X2O': 56.093056556486744,
        'X4O6': 159.87070574443365,
        'X4O10': 168.84121978259935,
        'XO': 28.190280638134226,
        'XO2': 42.30717797818566,
        'XO3': 94.28216713499612
    },
    'Cd': {
        'BCC': 23.419572636611186,
        'Diamond': 74.9080478398366,
        'FCC': 22.84127817506259,
        'SC': 26.923600245740374,
        'X2O': 53.08274544890885,
        'X4O6': 146.50420325772748,
        'X4O10': 161.23783826032457,
        'XO': 27.093952692307997,
        'XO2': 38.53660786052451,
        'XO3': 73.15212767492528
    },
    'Ce': {
        'BCC': 27.323964391559745,
        'Diamond': 60.36962802029463,
        'FCC': 26.52235577369828,
        'SC': 24.921013988065226,
        'X2O': 56.10337207266545,
        'X4O6': 164.25418463410853,
        'X4O10': 168.09408301376885,
        'XO': 31.03931067267397,
        'XO2': 40.79349906605082,
        'XO3': 81.65266123145898
    },
    'Cl': {
        'BCC': 21.45474447156215,
        'Diamond': 67.51485775600094,
        'FCC': 21.28818952390059,
        'SC': 23.464029289304705,
        'X2O': 54.559995010249025,
        'X4O6': 135.2505227617596,
        'X4O10': 150.35467864680692,
        'XO': 26.958660419342714,
        'XO2': 34.69762129543773,
        'XO3': 52.00560808001245
    },
    'Cm': {
        'BCC': 16.520782253696993,
        'Diamond': 38.65236260941347,
        'FCC': 17.49246649598698,
        'SC': 16.399002476823348,
        'X2O': 45.8647310715384,
        'X4O6': 144.5823454157204,
        'X4O10': 149.2067378771397,
        'XO': 26.794240945658323,
        'XO2': 36.65760496498912,
        'XO3': 67.29565446076828
    },
    'Co': {
        'BCC': 10.544792682061145,
        'Diamond': 29.76943354558406,
        'FCC': 10.30839278272375,
        'SC': 11.89326679933492,
        'X2O': 29.835635132747164,
        'X4O6': 90.62563815518891,
        'X4O10': 117.91158291691508,
        'XO': 17.2707951704295,
        'XO2': 25.033682445053497,
        'XO3': 47.13776514479808
    },
    'Cr': {
        'BCC': 11.548199339585498,
        'Diamond': 33.07091175174273,
        'FCC': 11.88591646485507,
        'SC': 12.806958012708076,
        'X2O': 30.441620462552265,
        'X4O6': 96.55853606378332,
        'X4O10': 116.26560686420035,
        'XO': 17.664208258348324,
        'XO2': 25.425018327785768,
        'XO3': 46.87447267496256
    },
    'Cs': {
        'BCC': 116.84172285179223,
        'Diamond': 377.51195120825054,
        'FCC': 117.36057894318543,
        'SC': 128.35905522569587,
        'X2O': 96.49631431852791,
        'X4O6': 283.36831128941446,
        'X4O10': 244.2227704566822,
        'XO': 53.296471975757754,
        'XO2': 63.55584224078437,
        'XO3': 76.20321967007675
    },
    'Cu': {
        'BCC': 12.004516294003697,
        'Diamond': 38.35431876920354,
        'FCC': 11.952224652322,
        'SC': 13.93497312424489,
        'X2O': 34.538824246939555,
        'X4O6': 103.80050324183955,
        'X4O10': 128.49045023552725,
        'XO': 19.107179475952414,
        'XO2': 28.233135239309007,
        'XO3': 52.01329300394603
    },
    'Dy': {
        'BCC': 32.288569204367704,
        'Diamond': 46.02037644692374,
        'FCC': 32.47663596342035,
        'SC': 31.857193083077554,
        'X2O': 51.791906354845,
        'X4O6': 148.57156262807263,
        'X4O10': 155.3517188345445,
        'XO': 28.17588955662974,
        'XO2': 36.94493400809891,
        'XO3': 71.77986544273953
    },
    'Er': {
        'BCC': 33.930706759113306,
        'Diamond': 160.71821137423967,
        'FCC': 34.82288485206389,
        'SC': 35.94303590731403,
        'X2O': 53.81459081043589,
        'X4O6': 147.37472200530317,
        'X4O10': 156.02502208600697,
        'XO': 28.024032432884447,
        'XO2': 36.90354261761303,
        'XO3': 73.1595986199416
    },
    'Eu': {
        'BCC': 26.134083220040758,
        'Diamond': 41.36507024860562,
        'FCC': 24.991976993289317,
        'SC': 17.795148296296187,
        'X2O': 50.001667417550664,
        'X4O6': 152.32836518004666,
        'X4O10': 156.76478360639607,
        'XO': 28.504144398060703,
        'XO2': 37.69178097711213,
        'XO3': 71.09394505182284
    },
    'F': {
        'BCC': 10.084127330253507,
        'Diamond': 29.00480135135658,
        'FCC': 10.146987257565563,
        'SC': 10.520941986117641,
        'X2O': 30.965700436027305,
        'X4O6': 112.64541976601427,
        'X4O10': 121.58537901423888,
        'XO': 18.85666827649016,
        'XO2': 30.8381241671117,
        'XO3': 53.32461728600295
    },
    'Fe': {
        'BCC': 10.500484196384424,
        'Diamond': 28.92444480806717,
        'FCC': 10.260211776246214,
        'SC': 11.651347771934487,
        'X2O': 29.393596433158027,
        'X4O6': 90.8708646557079,
        'X4O10': 116.01663708535033,
        'XO': 17.13602302251458,
        'XO2': 24.193568321607785,
        'XO3': 45.93568036284212
    },
    'Fr': {
        'BCC': 116.49231248676648,
        'Diamond': 384.032159640575,
        'FCC': 117.16295360986017,
        'SC': 132.17408585059354,
        'X2O': 106.7081347574964,
        'X4O6': 262.19120504163266,
        'X4O10': 229.91803402594206,
        'XO': 55.39235301269504,
        'XO2': 58.91457051825229,
        'XO3': 79.51922007961696
    },
    'Ga': {
        'BCC': 19.205587610065592,
        'Diamond': 50.83804519831408,
        'FCC': 18.94650008463183,
        'SC': 20.117312153714767,
        'X2O': 53.04283169078497,
        'X4O6': 110.9910906198663,
        'X4O10': 135.6777983405775,
        'XO': 24.3519260047855,
        'XO2': 29.261674771756013,
        'XO3': 57.47020127815971
    },
    'Gd': {
        'BCC': 28.947403790341237,
        'Diamond': 41.89427015507523,
        'FCC': 27.993870497573496,
        'SC': 20.809598933525876,
        'X2O': 50.44857514811846,
        'X4O6': 151.01960637372918,
        'X4O10': 156.1541161775155,
        'XO': 28.361967230056884,
        'XO2': 37.36132396877068,
        'XO3': 71.11221246370377
    },
    'Ge': {
        'BCC': 19.269495309541668,
        'Diamond': 47.82658983637545,
        'FCC': 19.582450801516337,
        'SC': 19.941748311438904,
        'X2O': 49.682664655338414,
        'X4O6': 121.20682589314904,
        'X4O10': 139.51685630367012,
        'XO': 27.122474603810435,
        'XO2': 28.190777443226906,
        'XO3': 51.07396911183831
    },
    'H': {
        'BCC': 2.9667742929491983,
        'Diamond': 6.831111656170396,
        'FCC': 2.964809112631019,
        'SC': 3.0866686938954198,
        'X2O': 11.96347891754409,
        'X4O6': 61.84619062082082,
        'X4O10': 103.34760585605244,
        'XO': 10.027848555766598,
        'XO2': 19.10459642792833,
        'XO3': 31.08928579616201
    },
    'He': {
        'BCC': 18.03038805667906,
        'Diamond': 64.21487893655794,
        'FCC': 17.77258072097029,
        'SC': 21.485271132346007,
        'X2O': 92.1784561523737,
        'X4O6': 94.28925599148977,
        'X4O10': 112.79696925975088,
        'XO': 31.528419933938714,
        'XO2': 24.776988443791783,
        'XO3': 43.05953909874314
    },
    'Hf': {
        'BCC': 22.304719470255996,
        'Diamond': 70.15774064881346,
        'FCC': 22.56760341919224,
        'SC': 24.781139308615224,
        'X2O': 48.646400121650814,
        'X4O6': 128.35931068254416,
        'X4O10': 142.6630326850143,
        'XO': 24.23287799836855,
        'XO2': 33.11711229980498,
        'XO3': 70.82641284222711
    },
    'Hg': {
        'BCC': 29.237160212480383,
        'Diamond': 112.58234465071779,
        'FCC': 32.347797693222105,
        'SC': 29.854323349822568,
        'X2O': 56.18140344248944,
        'X4O6': 152.67994259762438,
        'X4O10': 166.5429926971196,
        'XO': 29.815337246318833,
        'XO2': 39.30320403679202,
        'XO3': 72.63086986279107
    },
    'Ho': {
        'BCC': 33.267285286514536,
        'Diamond': 50.850112182697885,
        'FCC': 33.89173938695238,
        'SC': 34.29406972261268,
        'X2O': 52.720584538970684,
        'X4O6': 147.77237454797466,
        'X4O10': 155.498638197736,
        'XO': 28.09751593211625,
        'XO2': 36.88059139506703,
        'XO3': 72.36955554798217
    },
    'I': {
        'BCC': 35.98676703422804,
        'Diamond': 121.14527750115774,
        'FCC': 35.10488450303778,
        'SC': 41.563154436885426,
        'X2O': 72.18837519600658,
        'X4O6': 167.70157634361635,
        'X4O10': 163.00329788775545,
        'XO': 38.012187732710515,
        'XO2': 41.44308732251686,
        'XO3': 60.8593853544075
    },
    'In': {
        'BCC': 27.78058028977925,
        'Diamond': 76.43205868239434,
        'FCC': 27.510098861849173,
        'SC': 29.550210982224606,
        'X2O': 65.93128247399382,
        'X4O6': 140.37972957567774,
        'X4O10': 156.84754366426756,
        'XO': 30.493020046452415,
        'XO2': 36.4728530560971,
        'XO3': 74.76771621358446
    },
    'Ir': {
        'BCC': 15.055641509624866,
        'Diamond': 43.19355479899218,
        'FCC': 14.504994129130639,
        'SC': 16.994504125715814,
        'X2O': 40.335773558548205,
        'X4O6': 111.03802506574334,
        'X4O10': 130.89009942137127,
        'XO': 23.407091787120702,
        'XO2': 30.412061386006737,
        'XO3': 53.86205787750703
    },
    'K': {
        'BCC': 73.77951798925906,
        'Diamond': 223.8044852857056,
        'FCC': 74.0044361285296,
        'SC': 79.35441292282184,
        'X2O': 68.00231952386676,
        'X4O6': 227.58585562272344,
        'X4O10': 198.7688451835494,
        'XO': 42.74542841231468,
        'XO2': 58.364291316549775,
        'XO3': 136.53458069146527
    },
    'Kr': {
        'BCC': 67.46343111231373,
        'Diamond': 249.70798358307468,
        'FCC': 66.04176534194272,
        'SC': 82.66454249647283,
        'X2O': 118.9585508090093,
        'X4O6': 194.32198413188252,
        'X4O10': 192.3183698028015,
        'XO': 40.074708805542606,
        'XO2': 47.54241675063361,
        'XO3': 68.05698604893871
    },
    'La': {
        'BCC': 37.81757025987239,
        'Diamond': 74.6380534647987,
        'FCC': 36.94689152187624,
        'SC': 36.74506294698075,
        'X2O': 65.1355386334241,
        'X4O6': 182.51664661894716,
        'X4O10': 189.9565090452286,
        'XO': 34.431148430436664,
        'XO2': 47.879356819541435,
        'XO3': 90.87226882578761
    },
    'Li': {
        'BCC': 20.26746521326914,
        'Diamond': 51.386718035200765,
        'FCC': 20.224478592944862,
        'SC': 20.408801974058356,
        'X2O': 24.721960544273422,
        'X4O6': 89.47029219795057,
        'X4O10': 123.69216300739882,
        'XO': 16.825820190091974,
        'XO2': 24.922284658946587,
        'XO3': 46.08915629110866
    },
    'Lu': {
        'BCC': 29.625706403168756,
        'Diamond': 101.21824904812954,
        'FCC': 28.971404244986218,
        'SC': 32.94002697403406,
        'X2O': 54.07084024181938,
        'X4O6': 144.42641168708099,
        'X4O10': 157.87453325697595,
        'XO': 26.56021013564612,
        'XO2': 37.2635986858799,
        'XO3': 78.08376717694293
    },
    'Mg': {
        'BCC': 22.917256964511758,
        'Diamond': 80.85167299497583,
        'FCC': 23.125246144035756,
        'SC': 27.580585068743265,
        'X2O': 44.86183000035521,
        'X4O6': 113.54760895185626,
        'X4O10': 138.89600330910122,
        'XO': 19.249038606577802,
        'XO2': 30.59038256739605,
        'XO3': 62.138771825273366
    },
    'Mn': {
        'BCC': 10.780826839746656,
        'Diamond': 30.343273943677755,
        'FCC': 10.74714431323143,
        'SC': 11.898840334365449,
        'X2O': 29.56938020768425,
        'X4O6': 92.93093174774003,
        'X4O10': 115.32569138735641,
        'XO': 17.29322893846185,
        'XO2': 24.65355351858858,
        'XO3': 45.60581266709946
    },
    'Mo': {
        'BCC': 15.79258104697389,
        'Diamond': 46.0075591929068,
        'FCC': 16.03513025909276,
        'SC': 17.596261003631483,
        'X2O': 39.71697359069131,
        'X4O6': 113.5385735057236,
        'X4O10': 125.80601908145752,
        'XO': 21.517172018665377,
        'XO2': 29.728405877601176,
        'XO3': 55.619143262267016
    },
    'N': {
        'BCC': 7.234713384391083,
        'Diamond': 18.353307328655553,
        'FCC': 7.601290987311529,
        'SC': 6.4799301718134945,
        'X2O': 26.668195825122606,
        'X4O6': 91.34940280257845,
        'X4O10': 114.46774472474029,
        'XO': 15.338064037074709,
        'XO2': 25.510433208662196,
        'XO3': 39.709446138578244
    },
    'Na': {
        'BCC': 37.0150641099501,
        'Diamond': 109.14307889062314,
        'FCC': 37.09898609533899,
        'SC': 39.75136024526744,
        'X2O': 43.71121002820873,
        'X4O6': 141.71681543078222,
        'X4O10': 148.71649268928093,
        'XO': 27.75973762355678,
        'XO2': 36.67384223422674,
        'XO3': 82.14899104753098
    },
    'Nb': {
        'BCC': 18.141501149703938,
        'Diamond': 51.561238078557054,
        'FCC': 18.767798961263477,
        'SC': 20.11491091476522,
        'X2O': 43.28278775475433,
        'X4O6': 120.16533336097291,
        'X4O10': 131.56127186478793,
        'XO': 22.37583525598806,
        'XO2': 31.245484010498544,
        'XO3': 61.62342809972614
    },
    'Nd': {
        'BCC': 21.067820980974645,
        'Diamond': 47.09910554296685,
        'FCC': 22.76475614736112,
        'SC': 18.076519773292212,
        'X2O': 50.55913963478756,
        'X4O6': 157.2750957000189,
        'X4O10': 159.43404205819974,
        'XO': 29.462626031877953,
        'XO2': 39.09652888678157,
        'XO3': 72.646900859646
    },
    'Ne': {
        'BCC': 24.71124681788727,
        'Diamond': 89.14741870388465,
        'FCC': 24.302821945776444,
        'SC': 29.745196214720195,
        'X2O': 95.68692748957515,
        'X4O6': 163.85525050963838,
        'X4O10': 139.05329523577188,
        'XO': 47.035178282601564,
        'XO2': 38.907280067896835,
        'XO3': 80.74050072734626
    },
    'Ni': {
        'BCC': 10.895093061761445,
        'Diamond': 33.01213803450207,
        'FCC': 10.834896920712186,
        'SC': 12.559018562774645,
        'X2O': 31.63920052019124,
        'X4O6': 96.00067736850777,
        'X4O10': 121.43398022227859,
        'XO': 17.995025574679634,
        'XO2': 26.201216458836914,
        'XO3': 49.20510555899307
    },
    'Np': {
        'BCC': 17.807922782490138,
        'Diamond': 42.95330163212968,
        'FCC': 19.29454685971419,
        'SC': 17.27341608157796,
        'X2O': 47.24868290388707,
        'X4O6': 147.2543455647368,
        'X4O10': 149.62945441680233,
        'XO': 27.332561407471673,
        'XO2': 38.068057796935754,
        'XO3': 70.28795397638314
    },
    'O': {
        'BCC': 7.78628266525722,
        'Diamond': 21.362896477617028,
        'FCC': 7.998751998386165,
        'SC': 7.9535032234012455,
        'X2O': 27.113414270760558,
        'X4O6': 95.75038198469915,
        'X4O10': 57.93310344724358,
        'XO': 7.953538741823925,
        'XO2': 27.113413250460304,
        'XO3': 44.77956083125996
    },
    'Os': {
        'BCC': 14.78084322507656,
        'Diamond': 42.9003271344621,
        'FCC': 14.340859045509657,
        'SC': 16.730379012809376,
        'X2O': 39.55396843553531,
        'X4O6': 109.41163467284458,
        'X4O10': 126.3678623634867,
        'XO': 22.534802096302215,
        'XO2': 28.79510233771812,
        'XO3': 53.087375075866504
    },
    'P': {
        'BCC': 14.230164321304509,
        'Diamond': 41.31830893726011,
        'FCC': 14.563581331499318,
        'SC': 14.656449377963373,
        'X2O': 40.02771510485819,
        'X4O6': 112.87273517265223,
        'X4O10': 130.5423350374474,
        'XO': 24.301025668451572,
        'XO2': 27.202425333619257,
        'XO3': 37.00993065934867
    },
    'Pa': {
        'BCC': 24.79711503036195,
        'Diamond': 61.01937641455636,
        'FCC': 25.297938295713884,
        'SC': 24.020708104944944,
        'X2O': 56.2923013639154,
        'X4O6': 157.60248641887026,
        'X4O10': 159.66427397432875,
        'XO': 30.08936686358132,
        'XO2': 40.646251505228875,
        'XO3': 77.70400767520664
    },
    'Pb': {
        'BCC': 31.970428409394437,
        'Diamond': 88.08996646012986,
        'FCC': 32.03306938203389,
        'SC': 34.475841929694184,
        'X2O': 70.31981813270207,
        'X4O6': 172.60825336207543,
        'X4O10': 176.45864055935584,
        'XO': 36.512023686270126,
        'XO2': 39.57820403727723,
        'XO3': 79.27127800108997
    },
    'Pd': {
        'BCC': 15.444289916443642,
        'Diamond': 49.01252559747134,
        'FCC': 15.325422077832961,
        'SC': 17.86134273356192,
        'X2O': 42.21329783824948,
        'X4O6': 118.05280817732637,
        'X4O10': 137.92391497458223,
        'XO': 23.309291604400354,
        'XO2': 31.39171980940216,
        'XO3': 58.588091256016085
    },
    'Pm': {
        'BCC': 20.357867142704176,
        'Diamond': 43.304676330483446,
        'FCC': 22.245498247271406,
        'SC': 17.298203854027527,
        'X2O': 49.83379328748851,
        'X4O6': 155.20449547214102,
        'X4O10': 158.30224161117826,
        'XO': 29.0223921974359,
        'XO2': 38.54048420728664,
        'XO3': 71.72250240290344
    },
    'Po': {
        'BCC': 32.85387443277191,
        'Diamond': 104.96348144641622,
        'FCC': 32.56333044469028,
        'SC': 37.59547097593211,
        'X2O': 70.32868290650609,
        'X4O6': 169.25786655507474,
        'X4O10': 170.67339820930886,
        'XO': 37.33735195514773,
        'XO2': 41.78482550799856,
        'XO3': 70.8861203509619
    },
    'Pr': {
        'BCC': 23.14136632760563,
        'Diamond': 52.44010307513777,
        'FCC': 24.094101059154127,
        'SC': 20.150812769506164,
        'X2O': 52.30909193445264,
        'X4O6': 159.98230063418686,
        'X4O10': 161.1448735980028,
        'XO': 30.079647021966668,
        'XO2': 39.8130143486941,
        'XO3': 76.68360703622963
    },
    'Pt': {
        'BCC': 15.83897506117951,
        'Diamond': 48.22996585367427,
        'FCC': 15.655951376029684,
        'SC': 18.08608810066213,
        'X2O': 43.22427839127414,
        'X4O6': 121.23220297706072,
        'X4O10': 138.52531619520533,
        'XO': 24.65126174899141,
        'XO2': 32.28610877596024,
        'XO3': 57.33606186529146
    },
    'Pu': {
        'BCC': 16.564337263638887,
        'Diamond': 40.40756807632782,
        'FCC': 17.802132741670547,
        'SC': 16.367182603361524,
        'X2O': 45.856916438975105,
        'X4O6': 145.34713832221837,
        'X4O10': 148.4904181466331,
        'XO': 26.843555957636305,
        'XO2': 37.404653487398264,
        'XO3': 68.83896241530358
    },
    'Ra': {
        'BCC': 70.96686726813562,
        'Diamond': 339.33715568177746,
        'FCC': 71.62698453639577,
        'SC': 75.34559972942921,
        'X2O': 93.85222231306872,
        'X4O6': 253.14665217290093,
        'X4O10': 243.08914406560956,
        'XO': 47.84672983416793,
        'XO2': 63.85690933808056,
        'XO3': 87.72887126431581
    },
    'Rb': {
        'BCC': 91.14407104919322,
        'Diamond': 282.75532076118924,
        'FCC': 91.42746794058951,
        'SC': 98.97943978164447,
        'X2O': 81.07805755649316,
        'X4O6': 269.0813732808233,
        'X4O10': 228.691418292538,
        'XO': 48.78608722464497,
        'XO2': 69.07223386788792,
        'XO3': 109.6419543319426
    },
    'Re': {
        'BCC': 15.104459817094252,
        'Diamond': 45.148809236588164,
        'FCC': 15.016349807290217,
        'SC': 17.161126852487122,
        'X2O': 40.19220628587344,
        'X4O6': 110.69361853100324,
        'X4O10': 124.47879492363927,
        'XO': 22.114932739891916,
        'XO2': 29.303668265498118,
        'XO3': 53.801956348577164
    },
    'Rh': {
        'BCC': 14.474214566259567,
        'Diamond': 41.90045770658525,
        'FCC': 14.050455087789901,
        'SC': 16.305292548470145,
        'X2O': 38.864125383904906,
        'X4O6': 108.86550922824448,
        'X4O10': 130.83222315437524,
        'XO': 22.007590948747932,
        'XO2': 29.59704033883366,
        'XO3': 54.262883750534
    },
    'Rn': {
        'BCC': 95.44712307816137,
        'Diamond': 353.9049285432958,
        'FCC': 93.15642568099791,
        'SC': 117.66721154187272,
        'X2O': 136.68062106103142,
        'X4O6': 203.14298424179267,
        'X4O10': 182.19732283519943,
        'XO': 47.53716008769388,
        'XO2': 48.684569917388075,
        'XO3': 72.11672976809862
    },
    'Ru': {
        'BCC': 14.2356061963808,
        'Diamond': 40.58636439793814,
        'FCC': 13.836615162302435,
        'SC': 15.838129195478007,
        'X2O': 37.88465524044423,
        'X4O6': 107.30705982161992,
        'X4O10': 126.58242850379463,
        'XO': 21.407374709227543,
        'XO2': 28.197060854448544,
        'XO3': 52.50000766450978
    },
    'S': {
        'BCC': 15.761846240045852,
        'Diamond': 48.56211942685064,
        'FCC': 15.880688535760665,
        'SC': 17.21972354529015,
        'X2O': 42.69615177372292,
        'X4O6': 117.26546327164885,
        'X4O10': 134.15692800852705,
        'XO': 24.68237090281372,
        'XO2': 29.74064057982278,
        'XO3': 39.144876124659845
    },
    'Sb': {
        'BCC': 27.22578892025933,
        'Diamond': 85.3650954131397,
        'FCC': 27.489629174053118,
        'SC': 30.063412199047434,
        'X2O': 61.58040236167005,
        'X4O6': 157.40430221967745,
        'X4O10': 157.7596931951854,
        'XO': 33.620445141559216,
        'XO2': 37.56851654211421,
        'XO3': 60.200363163061155
    },
    'Sc': {
        'BCC': 24.885851875806644,
        'Diamond': 68.91513800081006,
        'FCC': 24.68685206618529,
        'SC': 26.148054896880076,
        'X2O': 43.1201936529914,
        'X4O6': 124.43596210045614,
        'X4O10': 142.7497334543844,
        'XO': 22.33245591294768,
        'XO2': 32.98816253118319,
        'XO3': 70.49803013141532
    },
    'Se': {
        'BCC': 20.36002862058445,
        'Diamond': 63.51075543280724,
        'FCC': 20.377873243171113,
        'SC': 22.683410900018394,
        'X2O': 49.916540443645076,
        'X4O6': 132.28191329377364,
        'X4O10': 143.8666792862057,
        'XO': 28.353249176098515,
        'XO2': 33.07339199042741,
        'XO3': 49.835371160285675
    },
    'Si': {
        'BCC': 14.645167329852903,
        'Diamond': 40.914946909495136,
        'FCC': 14.482203156537594,
        'SC': 16.229264266618493,
        'X2O': 42.67943300969328,
        'X4O6': 97.64181551405363,
        'X4O10': 122.0201687316496,
        'XO': 24.591702654494867,
        'XO2': 24.054540201794367,
        'XO3': 41.77917674264566
    },
    'Sm': {
        'BCC': 21.645872487778497,
        'Diamond': 41.8054697729813,
        'FCC': 22.828410176942523,
        'SC': 17.19651251000493,
        'X2O': 49.728958873912646,
        'X4O6': 153.57555936269387,
        'X4O10': 157.45162227100155,
        'XO': 28.714206116400945,
        'XO2': 38.08248873604046,
        'XO3': 71.2926208460172
    },
    'Sn': {
        'BCC': 27.647292816555392,
        'Diamond': 73.68775929245572,
        'FCC': 28.008842565317664,
        'SC': 29.45303068635492,
        'X2O': 63.517454564272384,
        'X4O6': 145.0015751259266,
        'X4O10': 153.57255640399904,
        'XO': 33.53646695463933,
        'XO2': 34.005522463068516,
        'XO3': 67.66760180204915
    },
    'Sr': {
        'BCC': 54.01288070681168,
        'Diamond': 223.78212467284516,
        'FCC': 54.89226505461252,
        'SC': 57.12901437076641,
        'X2O': 70.15460135057535,
        'X4O6': 195.28799627773287,
        'X4O10': 195.56435016669093,
        'XO': 35.054726744617696,
        'XO2': 51.288214374025756,
        'XO3': 113.83558488790722
    },
    'Ta': {
        'BCC': 18.29199212051548,
        'Diamond': 56.81857153866048,
        'FCC': 18.839371123678717,
        'SC': 20.666915196502814,
        'X2O': 45.02366173703048,
        'X4O6': 119.92560072854101,
        'X4O10': 131.27425244267795,
        'XO': 22.857933915362626,
        'XO2': 31.393323034842506,
        'XO3': 61.74859411091955
    },
    'Tb': {
        'BCC': 30.900833790207777,
        'Diamond': 43.3929318627123,
        'FCC': 30.5520741878348,
        'SC': 27.819943230633633,
        'X2O': 51.03398079922181,
        'X4O6': 149.70034490399354,
        'X4O10': 155.63241924194543,
        'XO': 28.259571151217607,
        'XO2': 37.10490933052478,
        'XO3': 71.34886575115553
    },
    'Tc': {
        'BCC': 14.619575706232261,
        'Diamond': 42.49186074297397,
        'FCC': 14.512674098516719,
        'SC': 16.23500677922041,
        'X2O': 38.36873836025427,
        'X4O6': 109.22831208729696,
        'X4O10': 124.60383293240093,
        'XO': 21.215085714825058,
        'XO2': 28.72262824974431,
        'XO3': 53.10185902183261
    },
    'Te': {
        'BCC': 28.515291441922543,
        'Diamond': 92.80026299313013,
        'FCC': 28.278725764214045,
        'SC': 32.79348787788877,
        'X2O': 62.902862135347036,
        'X4O6': 157.41471574623864,
        'X4O10': 159.5213569309115,
        'XO': 34.886756554005444,
        'XO2': 39.05998140464958,
        'XO3': 56.87028551704623
    },
    'Th': {
        'BCC': 32.56774695907916,
        'Diamond': 91.46531559551607,
        'FCC': 32.18393193608481,
        'SC': 35.332143116981726,
        'X2O': 68.78943704444279,
        'X4O6': 174.9462606147876,
        'X4O10': 178.1984812274667,
        'XO': 33.14910649617435,
        'XO2': 44.316722036950985,
        'XO3': 84.96337662280618
    },
    'Ti': {
        'BCC': 17.266838705452646,
        'Diamond': 45.84215346743876,
        'FCC': 17.39458723307156,
        'SC': 18.41317508859035,
        'X2O': 36.316198404813036,
        'X4O6': 108.53055392333177,
        'X4O10': 127.4653908398403,
        'XO': 19.618577422797088,
        'XO2': 28.180821050777354,
        'XO3': 57.34969526362194
    },
    'Tl': {
        'BCC': 31.414378186238142,
        'Diamond': 90.28791740945354,
        'FCC': 31.14020499214428,
        'SC': 34.39406929999538,
        'X2O': 72.2312455718899,
        'X4O6': 159.29312629246806,
        'X4O10': 171.9695385864756,
        'XO': 33.95979805446588,
        'XO2': 40.712338658884974,
        'XO3': 79.03970760997683
    },
    'Tm': {
        'BCC': 34.35838197510023,
        'Diamond': 163.35125337993082,
        'FCC': 35.3320901937893,
        'SC': 37.20775751590177,
        'X2O': 55.15416506052816,
        'X4O6': 147.37391185959382,
        'X4O10': 156.9139177633689,
        'XO': 27.97404633502329,
        'XO2': 37.031684990572316,
        'XO3': 74.23192335305474
    },
    'U': {
        'BCC': 20.26617788045501,
        'Diamond': 49.519975357164356,
        'FCC': 21.71333551347746,
        'SC': 19.122378866945898,
        'X2O': 50.13448442439475,
        'X4O6': 150.81719404599662,
        'X4O10': 152.5408494440474,
        'XO': 28.358175935910488,
        'XO2': 39.024831046588396,
        'XO3': 72.22981860208787
    },
    'V': {
        'BCC': 13.460765481651748,
        'Diamond': 37.24006161339059,
        'FCC': 13.90495481845793,
        'SC': 14.680101780445174,
        'X2O': 32.71565060122779,
        'X4O6': 101.56973840415188,
        'X4O10': 119.93636555644552,
        'XO': 18.318376789871962,
        'XO2': 26.577462970973343,
        'XO3': 50.24019882325915
    },
    'W': {
        'BCC': 16.145475241629192,
        'Diamond': 49.550458646460285,
        'FCC': 16.45788463611506,
        'SC': 18.36945980233768,
        'X2O': 41.78242344799182,
        'X4O6': 114.26866563392595,
        'X4O10': 125.5626796561379,
        'XO': 22.23505673850225,
        'XO2': 30.145159806010696,
        'XO3': 56.046966762546646
    },
    'Xe': {
        'BCC': 89.0348898019572,
        'Diamond': 331.8911738829618,
        'FCC': 87.00696515780551,
        'SC': 109.80518463805863,
        'X2O': 137.20449538641535,
        'X4O6': 197.87269875915695,
        'X4O10': 183.5134022043552,
        'XO': 45.25074808388441,
        'XO2': 47.367146218637515,
        'XO3': 66.91984208412325
    },
    'Y': {
        'BCC': 33.030456505716884,
        'Diamond': 87.51871019715836,
        'FCC': 32.47154445149464,
        'SC': 34.81696972636951,
        'X2O': 56.02747345371688,
        'X4O6': 152.5278120094212,
        'X4O10': 164.9592081970141,
        'XO': 28.054494905895083,
        'XO2': 40.19080449916217,
        'XO3': 88.91885642621565
    },
    'Yb': {
        'BCC': 34.63958822124242,
        'Diamond': 164.11669062811166,
        'FCC': 35.70435292729837,
        'SC': 38.64163965012807,
        'X2O': 57.46306293020211,
        'X4O6': 147.8453493453804,
        'X4O10': 158.485694644327,
        'XO': 27.99295019377839,
        'XO2': 37.303196002027036,
        'XO3': 75.698631528076
    },
    'Zn': {
        'BCC': 15.375108807414573,
        'Diamond': 49.34739674485901,
        'FCC': 15.162008472091504,
        'SC': 18.179032389787878,
        'X2O': 40.66830252174508,
        'X4O6': 113.25459434772472,
        'X4O10': 137.04058517287152,
        'XO': 20.277796523833768,
        'XO2': 30.392947522085286,
        'XO3': 56.550704684279864
    },
    'Zr': {
        'BCC': 22.844776018395795,
        'Diamond': 61.910070007697755,
        'FCC': 23.213378511600922,
        'SC': 24.743132170798372,
        'X2O': 48.08700889172749,
        'X4O6': 130.27051664761115,
        'X4O10': 143.9679739505779,
        'XO': 24.28500118769881,
        'XO2': 33.473297299752296,
        'XO3': 72.3223459052111
    }
}


if __name__ == '__main__':
    print(work('FCC', 'H'))
