# channel_configuration_row.py
#
# Copyright 2024 Christopher Talbot
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

from gi.repository import Adw
from gi.repository import Gtk
import gettext
import os
import base64
from enum import Enum

import gtk_meshtastic_client.utils as utils
import gtk_meshtastic_client.message_storage as message_storage
import gtk_meshtastic_client.channel_configuration_row as channel_configuration_row

@Gtk.Template(resource_path='/org/kop316/meshtastic/ui/channel_configuration_row.ui')
class ChannelConfigurationRow(Adw.ExpanderRow):
    __gtype_name__ = 'ChannelConfigurationRow'

    channel_name_entry = Gtk.Template.Child()
    downlink_enabled = Gtk.Template.Child()
    enable_precise_location = Gtk.Template.Child()
    enable_position = Gtk.Template.Child()
    location_precision_spin_row = Gtk.Template.Child()
    location_precision_spin_row_value = Gtk.Template.Child()
    psk_entry = Gtk.Template.Child()
    uplink_enabled = Gtk.Template.Child()


    def get_position(self):
        if not self.enable_position.get_active():
            return 0

        if self.enable_precise_location.get_active():
            return 32

        new_value = int(self.location_precision_spin_row.get_value())

        return new_value

    def get_channel_downlink(self):
        return self.uplink_enabled.get_active()

    def get_channel_uplink(self):
        return self.uplink_enabled.get_active()

    def set_position(self, position):
        if position == 0:
            self.enable_position.set_active(False)
            self.enable_precise_location.set_active(False)
            self.enable_precise_location.set_sensitive(False)
            self.location_precision_spin_row.set_visible(False)
        elif position == 32:
            self.enable_position.set_active(True)
            self.enable_precise_location.set_active(True)
            self.enable_precise_location.set_sensitive(True)
            self.location_precision_spin_row.set_visible(False)
        else:
            self.enable_position.set_active(True)
            self.enable_precise_location.set_active(False)
            self.enable_precise_location.set_sensitive(True)
            self.location_precision_spin_row.set_visible(True)
            self.location_precision_spin_row.set_value(position)

    def set_channel_downlink(self, downlink):
        self.downlink_enabled.set_active(downlink)

    def set_channel_uplink(self, uplink):
        self.uplink_enabled.set_active(uplink)

    def set_channel_psk(self, psk):
        psk_base64 = base64.b64encode(psk).decode('utf-8')
        self.psk_entry.set_text(psk_base64)

    def set_channel_name(self, name):
        if not name:
            name = "LongFast"
        elif not name:
            return

        self.channel_name_entry.set_text(name)

    def set_channel_index(self, index):
        self.index = index
        self.set_title("Channel " + str(index))

    def reset_psk(self):
        #32 is bytes, 32*8 = 256
        self.set_channel_psk(os.urandom(32))

    @Gtk.Template.Callback()
    def _delete_channel_button_clicked_cb(self, button):
        app = Gtk.Application.get_default()
        win = Gtk.Application.get_active_window(app)

        win.connection_page_bin.channel_configuration_page.channel_config_list_box.remove(self)
        win.connection_page_bin.channel_configuration_page.channel_config_list_box_children -= 1
        win.connection_page_bin.channel_configuration_page.reorder_channels()

    @Gtk.Template.Callback()
    def _reset_psk_button_clicked_cb(self, button):
        self.reset_psk()

    @Gtk.Template.Callback()
    def enable_position_notify_active_cb(self, button, user):
        if self.enable_position.get_active():
            self.enable_precise_location.set_sensitive(True)
            self.location_precision_spin_row.set_visible(not self.enable_precise_location.get_active())
        else:
            self.enable_precise_location.set_active(False)
            self.location_precision_spin_row.set_visible(False)
            self.enable_precise_location.set_sensitive(False)

    @Gtk.Template.Callback()
    def enable_position_precision_active_cb(self, button, user):
        if self.enable_position.get_active():
            self.enable_precise_location.set_sensitive(True)
            self.location_precision_spin_row.set_visible(not self.enable_precise_location.get_active())
        else:
            self.enable_precise_location.set_active(False)
            self.enable_precise_location.set_sensitive(False)
            self.location_precision_spin_row.set_visible(False)

    @Gtk.Template.Callback()
    def position_precision_value_changed_cb(self, user):
        new_value = int(self.location_precision_spin_row.get_value())

        match new_value:
            case 19:
                self.location_precision_spin_row.set_title("Position Precision: 45 m")
            case 18:
                self.location_precision_spin_row.set_title("Position Precision: 95 m")
            case 17:
                self.location_precision_spin_row.set_title("Position Precision: 190 m")
            case 16:
                self.location_precision_spin_row.set_title("Position Precision: 375 m")
            case 15:
                self.location_precision_spin_row.set_title("Position Precision: 750 m")
            case 14:
                self.location_precision_spin_row.set_title("Position Precision: 1.5 km")
            case 13:
                self.location_precision_spin_row.set_title("Position Precision: 3 km")
            case 12:
                self.location_precision_spin_row.set_title("Position Precision: 6 km")
            case 11:
                self.location_precision_spin_row.set_title("Position Precision: 12 km")
            case 10:
                self.location_precision_spin_row.set_title("Position Precision: 24 km")
            case _:
                self.location_precision_spin_row.set_title("Position Precision: Unknown")

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        app = Gtk.Application.get_default()
        self.logger = app.logger
