/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once


/////////////////////// Local includes
#include "MsXpS/export-import-config.h"

#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/CrossLinker.hpp"
#include "MsXpS/libXpertMassCore/IndexRangeCollection.hpp"
#include "MsXpS/libXpertMassCore/Monomer.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{


class Polymer;
typedef std::shared_ptr<Polymer> PolymerSPtr;
typedef std::shared_ptr<const Polymer> PolymerCstSPtr;
typedef QSharedPointer<Polymer> PolymerQSPtr;
typedef QSharedPointer<const Polymer> PolymerCstQSPtr;

class CrossLink;

typedef CrossLink *CrossLinkRPtr;
typedef const CrossLink *CrossLinkCstRPtr;

typedef std::unique_ptr<CrossLink> CrossLinkUPtr;
typedef std::unique_ptr<const CrossLink> CrossLinkCstUPtr;

typedef std::shared_ptr<CrossLink> CrossLinkSPtr;
typedef std::shared_ptr<const CrossLink> CrossLinkCstSPtr;

typedef std::weak_ptr<CrossLink> CrossLinkWPtr;
typedef std::weak_ptr<const CrossLink> CrossLinkCstWPtr;

using UuidCrossLinkCstWPtrPair = std::pair<QString, CrossLinkCstWPtr>;
using UuidCrossLinkWPtrPair    = std::pair<QString, CrossLinkWPtr>;

class DECLSPEC CrossLink
{
  public:
  CrossLink();

  CrossLink(PolChemDefCstSPtr pol_chem_def_csp,
            PolymerCstQSPtr polymer_cqsp = nullptr,
            const QString &name        = QString(),
            const QString &formula     = QString(),
            const QString &comment     = QString());

  CrossLink(CrossLinkerCstSPtr cross_linker_csp,
            PolymerCstQSPtr polymer_cqsp = nullptr,
            const QString &comment     = QString());

  CrossLink(const CrossLink &other);

  ~CrossLink();


  //////////////// THE POLYMER /////////////////////
  PolymerCstQSPtr getPolymerCstSPtr() const;

  //////////////// THE CROSS-LINKER /////////////////////
  const CrossLinkerCstSPtr getCrossLinkerCstSPtr() const;

  //////////////// THE COMMENT /////////////////////
  void setComment(const QString &);
  const QString &getComment() const;

  //////////////// THE NAME /////////////////////
  QString getCrossLinkerName() const;

  //////////////// THE MONOMERS /////////////////////
  const std::vector<MonomerCstSPtr> &getMonomersCstRef() const;
  std::vector<MonomerCstSPtr> &getMonomersRef();

  QString appendMonomer(MonomerCstSPtr monomer_csp);
  MonomerCstSPtr getMonomerAt(std::size_t index);
  bool removeMonomerAt(std::size_t index);

  std::size_t fillInMonomers(QString monomer_indices_text, bool &ok);

  std::vector<std::size_t> continuumOfLocationsOfInclusiveSequenceMonomers(
    Enums::LocationType location_type) const;
  QString continuumOfLocationsOfInclusiveSequenceMonomersAsText(
    Enums::LocationType location_type) const;

  std::vector<std::size_t>
  locationssOfOnlyExtremeSequenceMonomers(Enums::LocationType location_type) const;
  QString locationsOfOnlyExtremeSequenceMonomersAsText(
    Enums::LocationType location_type) const;

  std::size_t monomerIndex(MonomerCstSPtr monomer_csp, bool &ok) const;
  std::size_t monomerIndex(MonomerCstRPtr monomer_crp, bool &ok) const;

  MonomerCstSPtr getFirstMonomer() const;

  //////////////// OPERATORS /////////////////////
  CrossLink &operator=(const CrossLink &other);
  bool operator==(const CrossLink &) const;
  bool operator!=(const CrossLink &) const;

  //////////////// CROSS-LINK LOGIC /////////////////////
  Enums::CrossLinkEncompassed isEncompassedByIndexRange(std::size_t start,
                                                 std::size_t end,
                                                 std::size_t &in_count,
                                                 std::size_t &out_count) const;
  Enums::CrossLinkEncompassed
  isEncompassedByIndexRangeCollection(const IndexRangeCollection &index_ranges,
                                      std::size_t &in_count,
                                      std::size_t &out_count) const;

  //////////////// VALIDATIONS /////////////////////
  bool validate(ErrorList *error_list_p) const;
  bool isValid() const;

  //////////////// MASS OPERATIONS /////////////////////
  bool calculateMasses(const IsotopicDataCstSPtr &isotopic_data_csp,
                       double &mono,
                       double &avg) const;
  const CrossLink &
  accountMasses(double &mono, double &avg, int times = 1) const;
  double getMass(Enums::MassType mass_type) const;

  ////////////////UTILS /////////////////////
  QString formatContainerOfMonomerLocationsAsText(
    const std::vector<std::size_t> &locations) const;
  QString prepareResultsTxtString();

  QString storeMonomer(const MonomerCstSPtr &monomer_csp);
  bool hasMonomer(const MonomerCstSPtr &monomer_csp) const;
  bool hasUuid(const MonomerCstSPtr &monomer_csp) const;

  MonomerCstSPtr getMonomerForUuid(const QString &uuid) const;
  QString getUuidForMonomer(const MonomerCstSPtr &monomer_csp) const;

  std::vector<QString> getAllMonomerUuids() const;

  QString toString() const;

  protected:
  CrossLinkerCstSPtr mcsp_crossLinker = nullptr;
  PolymerCstQSPtr mcsp_polymer         = nullptr;
  QString m_comment;

  std::vector<MonomerCstSPtr> m_monomers;

  //  We need in code that uses this class,  to constantly keep at hand
  //  the Monomer instances that are involved in CrossLink_s. For example,
  //  we need to store the Monomer pointers as strings in QListWidget items
  //  (Qt:UserRole).
  //  We thus make use of the QUuid class to craft a Uuid string that
  //  we associate to the Monomer weak pointer that in turn relates
  //  to the Monomer shared pointer.
  //  Note that the Monomer stored here are those in the Polymer.Sequence's
  //  container of Monomer_s. The Polymer.Sequence's container of Monomer_s are
  //  MonomerSPtr,  but we want to have them stored here as CstSPtr,  so that we
  //  are sure not to modify them by error.
  std::vector<UuidMonomerCstWPtrPair> m_uuidMonomerPairs;
  mutable bool m_isValid = false;

  private:
  bool removeMonomer(MonomerCstSPtr monomer_sp);
  void cleanupMonomers();
};

typedef std::shared_ptr<CrossLink> CrossLinkSPtr;
typedef std::shared_ptr<const CrossLink> CrossLinkCstSPtr;


} // namespace libXpertMassCore
} // namespace MsXpS


Q_DECLARE_METATYPE(MsXpS::libXpertMassCore::CrossLink);
extern int crossLinkMetaTypeId;

Q_DECLARE_METATYPE(MsXpS::libXpertMassCore::CrossLink *);
extern int crossLinkPtrMetaTypeId;
