use core:io;
use util:serialize;

/**
 * Version of the Url class that also stores query parameters.
 */
class QueryUrl : extends Url, serializable {
	// Create an empty Url.
	init() {
		init() {}
	}

	// Create from a regular Url.
	cast(Url original) {
		var params = if (original as QueryUrl) {
			original.parameters;
		} else {
			Str->Str();
		};

		init(original) {
			parameters = params;
		}
	}

	// Copy a QueryUrl.
	init(QueryUrl original) {
		init(original) {
			parameters = original.parameters;
		}
	}

	// Create from an Url but add parameters.
	init(Url original, Str->Str params) {
		init(original) {
			parameters = params;
		}
	}

	// Create a relative url from parts and parameters.
	init(Str[] parts, Str->Str params) {
		init(parts) {
			parameters = params;
		}
	}

	// Create from a protocol and path segments.
	init(Protocol protocol, Str[] parts) {
		init(protocol, parts) {}
	}

	// Create from a protocol, path segments, and query parameters.
	init(Protocol protocol, Str[] parts, Str->Str params) {
		init(protocol, parts) {
			parameters = params;
		}
	}

	// Query parameters stored here.
	Str->Str parameters;

	// Add a query parameter. Note that this will mutate the object.
	QueryUrl push(QueryParam p) {
		parameters.put(p.key, p.value);
		this;
	}

	// Custom version of push for merging paths.
	QueryUrl push(Url url) : override {
		var merged = super:push(url);
		if (url as QueryUrl) {
			return QueryUrl(merged, Str->Str(url.parameters));
		} else if (merged as QueryUrl) {
			return merged;
		} else {
			throw InternalError("Should not happen.");
		}
	}

	// Override copying.
	protected Url copyShallow() : override {
		return QueryUrl(this);
	}

	// Compare.
	Bool ==(Url other) : override {
		if (!super:==(other))
			return false;

		unless (other as QueryUrl) {
			// We consider ourselves equal to the other one if we have no parameters.
			return parameters.empty;
		}

		if (parameters.count != other.parameters.count)
			return false;

		for (k, v in parameters) {
			if (o = other.parameters.at(k)) {
				if (o != v)
					return false;
			} else {
				return false;
			}
		}

		return true;
	}

	// Hash.
	Nat hash() : override {
		Nat out = super:hash();

		// Note: The order of keys is not necessarily deterministic, so we need to be careful to
		// ensure that the order of parameter keys does not impact the hash value!
		for (k, v in parameters) {
			Nat here = k.hash() ^ (v.hash() << 4);
			out ^= here;
		}

		out;
	}

	// Parameters.
	void toS(StrBuf to) : override {
		super:toS(to);

		if (parameters.any) {
			to << "?";
			Bool first = true;
			for (k, v in parameters) {
				if (!first)
					to << "&";
				first = false;
				to << k << "=" << v;
			}
		}
	}
}

// Operator to allow operating on an Url or a QueryUrl.
QueryUrl &(Url url, QueryParam p) {
	QueryUrl u = if (url as QueryUrl) {
		url;
	} else {
		QueryUrl(url);
	};
	u.push(p);
	return u;
}

// Query parameter.
value QueryParam {
	Str key;
	Str value;

	init(Str key, Str value) {
		init {
			key = key;
			value = value;
		}
	}
}
