use core:asm;
use core:lang;
use lang:bs:macro;

// Hints for the C++ module.
progvis:program:Hints hints() {
	progvis:program:Hints(CppCodeHints(), CppViewHints());
}

/**
 * Code hints.
 *
 * We can take all local variables and just put them inside a Variant directly, since they are
 * already allocated that way.
 */
class CppCodeHints extends progvis:program:CodeHints {
	init() {
		init() {
			// Anything not starting with a letter is an operator.
			operators("[^a-zA-Z_].*");
		}
	}

	// Regex for matching operators.
	lang:bnf:Regex operators;

	Fn<void>? findMain(Package pkg) : override {
		if (fn = pkg.find(SimplePart("main"), Scope(pkg)) as Function) {
			var ptr = fn.pointer;
			if (ptr as Fn<void>)
				return ptr;

			// TODO: Wrap output in a better way.
			if (ptr as Fn<Int>)
				return &ptr.wrapMain();
		}

		// TODO: Perhaps support "argc, argv" parameters as well!
		null;
	}

	Bool handlesFunction(Function fn) : override {
		if (fn as CppFunction)
			true;
		else
			false;
	}

	Str cleanName(Function fn, Package pkg) : override {
		Str name = fn.name;
		if (fn.isMember()) {
			if (t = fn.params[0].type) {
				if (fn.name == "__destroy") {
					return "~" + t.name + "()";
				} else if (fn.name == "__init") {
					name = t.name;
				} else if (operators.match(fn.name)) {
					name = "operator " + fn.name;
				}
			}
		}

		// TODO!
		name + "(...)";
	}

	Bool patchExternal(Function fn) : override {
		// Patch functions in our runtime. Otherwise memory accesses are not tracked very
		// well. Additionally, we need to patch things inside the Ptr<> classes.
		if (fn.hasParent(named{runtime}))
			return true;

		if (ptr = fn.parent as PtrType)
			return true;

		false;
	}

	progvis:program:Barrier functionBarrier(Function fn) : override {
		// No extra barriers for functions in our runtime.
		if (fn.hasParent(named{runtime}))
			return progvis:program:Barrier:none;

		// Neither for things in core.
		if (fn.hasParent(named{core}))
			return progvis:program:Barrier:none;

		// For other functions in Progvis, no barrier there either. These are implementation details
		// that we don't want to be visible.
		if (fn.hasParent(named{progvis}))
			return progvis:program:Barrier:none;

		// All other functions, we insert a barrier for. We don't know what they do.
		return progvis:program:Barrier:full;
	}

	Save saveVariable(Listing to, Var var, Listing:VarInfo info, Operand dest) : override {
		if (!info.ref)
			return Save:none;

		// If the variable is a reference to a value and a parameter, we assume that it originates
		// from a pointer allocation somewhere. It could, however, be allocated on the stack, which
		// in itself is fine as long as we don't try to read the type information from the RawPtr
		// created in the end.
		if (to.isParam(var)) {
			// Variables like this are followed by an integer variable specifying their original offset.
			Var next = nextParam(var, to);
			if (next == Var())
				return Save:none;

			if (next.size != sNat)
				return Save:none;

			unless (type = (named{}).find(SimplePart("Ptr", [Value(info.type)]), Scope()) as Type)
				throw InternalError("Could not find a pointer type for ${info.type}.");

			to << fnParam(ptrDesc, type.typeRef);
			to << fnParam(ptrDesc, ptrConst(1));
			to << fnCall(ref(BuiltIn:allocArray), false, ptrDesc, ptrA);

			to << mov(ptrB, var);
			to << ucast(ptrC, next);
			to << sub(ptrB, ptrC);

			to << mov(ptrRel(ptrA, Offset(sPtr)), ptrConst(1)); // filled
			to << mov(ptrRel(ptrA, Offset(sPtr*2)), ptrB); // base
			to << mov(intRel(ptrA, Offset(sPtr*3)), next); // offset

			to << mov(dest, ptrA);
		} else {
			to << mov(dest, var);
		}
		Save:ref;
	}

	// Find the next parameter.
	private Var nextParam(Var param, Listing in) : static {
		Var[] params = in.allParams();
		for (i, v in params) {
			if (v == param) {
				if (i + 1 < params.count)
					return params[i + 1];
			}
		}

		return Var();
	}
}

void wrapMain(Fn<Int> fn) {
	var program = progvis:program:findProgram();
	var result = fn.call.toS();
	if (program) {
		program.addOutput("\nMain returned ${result}\n");
		program.notifyChange(true);
	}
}
