#ifndef SCRIPTING_FIGURE_OF_TEXT_H_
#define SCRIPTING_FIGURE_OF_TEXT_H_

#include "figure.h"

namespace voronota
{

namespace scripting
{

class FigureOfText
{
public:
	static bool init_figure_of_text(const std::string& text, const bool outline, const std::vector<float>& origin, const float scale, const bool centered, Figure& figure)
	{
		if(text.empty())
		{
			return false;
		}

		TextGraphicsGenerator::TextGraphicsResultBundle text_graphics;
		if(!TextGraphicsGenerator::generate_text_graphics(text, outline, 1.0f, 1.0f, origin, scale*(1.5f/static_cast<float>(TextGraphicsGenerator::letter_size())), 0.7, centered, text_graphics))
		{
			return false;
		}

		figure.vertices.swap(text_graphics.vertices);
		figure.indices.swap(text_graphics.indices);
		figure.normals.clear();
		figure.normals.resize(figure.vertices.size(), 0.0f);
		for(std::size_t i=0;i<(figure.vertices.size()/3);i++)
		{
			figure.normals[i*3+0]=(origin.size()>0 ? origin[0] : 0.0f);
			figure.normals[i*3+1]=(origin.size()>1 ? origin[1] : 0.0f);
			figure.normals[i*3+2]=(origin.size()>2 ? origin[2] : 0.0f);
		}
		figure.normals_store_origin=true;

		figure.special_description_for_label.label_text=text;
		figure.special_description_for_label.label_origin=origin;
		figure.special_description_for_label.label_outline=outline;
		figure.special_description_for_label.label_scale=scale;
		figure.special_description_for_label.label_centered=centered;
		figure.special_description_for_label.is_label=true;

		return true;
	}

private:
	class TextGraphicsGenerator
	{
	public:
		struct TextGraphicsResultBundle
		{
			std::vector<float> vertices;
			std::vector<unsigned int> indices;

			bool empty() const
			{
				return indices.empty();
			}
		};

		static int letter_size()
		{
			return msize();
		}

		static bool generate_text_graphics(
				const std::string& text,
				const bool outline,
				const float horizontal_spacing,
				const float vertical_spacing,
				const std::vector<float>& raw_origin,
				const float scaling_factor,
				const float width_to_height_ratio,
				const bool centered,
				TextGraphicsResultBundle& result)
		{
			result=TextGraphicsResultBundle();

			if(text.empty())
			{
				return false;
			}

			result.vertices.reserve(text.size()*150);
			result.indices.reserve(text.size()*300);

			const float horizotal_addition=((static_cast<float>(msize())+horizontal_spacing)*scaling_factor*width_to_height_ratio);
			const float vertical_addition=0.0f-((static_cast<float>(msize())+vertical_spacing)*scaling_factor);

			std::vector<float> origin=raw_origin;
			if(origin.size()!=3)
			{
				origin.resize(3, 0.0f);
			}

			std::vector<float> offset=origin;

			std::vector<float> min_offset=offset;
			std::vector<float> max_offset=offset;

			for(std::size_t i=0;i<text.size();i++)
			{
				const char character=text[i];
				if(character=='\n')
				{
					offset[0]=origin[0];
					offset[1]+=vertical_addition;
				}
				else if(character>=' ')
				{
					const TextGraphicsResultBundle& charbundle=render_character_to_graphics_bundle(character, outline);
					if(!charbundle.empty())
					{
						unsigned int index_shift=result.vertices.size()/3;
						for(std::size_t j=0;j<charbundle.vertices.size();j++)
						{
							result.vertices.push_back(charbundle.vertices[j]*(j%3==0 ? scaling_factor*width_to_height_ratio : (j%3==1 ? scaling_factor : 1.0f))+offset[j%3]);
						}
						for(std::size_t j=0;j<charbundle.indices.size();j++)
						{
							result.indices.push_back(charbundle.indices[j]+index_shift);
						}
					}
					offset[0]+=horizotal_addition;
				}
				min_offset[0]=std::min(min_offset[0], offset[0]);
				min_offset[1]=std::min(min_offset[1], offset[1]);
				max_offset[0]=std::max(max_offset[0], offset[0]);
				max_offset[1]=std::max(max_offset[1], offset[1]);
			}

			if(centered)
			{
				for(std::size_t i=0;i<result.vertices.size();i+=3)
				{
					for(int j=0;j<2;j++)
					{
						result.vertices[i+j]-=(max_offset[j]-min_offset[j])/2.0;
					}
				}
			}

			return (!result.empty());
		}

	private:
		typedef std::vector< std::vector<unsigned int> > Matrix;

		static inline int msize()
		{
			static int msize_value=19;
			return msize_value;
		}

		static const TextGraphicsResultBundle& render_character_to_graphics_bundle(const char character, const bool outline)
		{
			static const TextGraphicsResultBundle blank_bundle;
			static std::vector<TextGraphicsResultBundle> bundles(128);
			static std::vector<TextGraphicsResultBundle> outline_bundles(128);

			const int character_id=static_cast<int>(character);
			if(character_id<33 || character_id>=128)
			{
				return blank_bundle;
			}

			TextGraphicsResultBundle& bundle=outline ? outline_bundles[character_id] : bundles[character_id];

			if(bundle.empty())
			{
				const Matrix& matrix=render_character_to_matrix(character, outline);
				{
					std::map< unsigned int, std::pair<int, int> > map_of_present_points;
					for(int row=0;row<msize();row++)
					{
						for (int col=0;col<msize();col++)
						{
							if(matrix[row][col]>0)
							{
								map_of_present_points[matrix[row][col]-1]=std::pair<int, int>(col, row);
							}
						}
					}
					for(std::map< unsigned int, std::pair<int, int> >::const_iterator it=map_of_present_points.begin();it!=map_of_present_points.end();++it)
					{
						const std::pair<int, int>& gp=it->second;
						bundle.vertices.push_back(static_cast<float>(gp.first));
						bundle.vertices.push_back(static_cast<float>(gp.second));
						bundle.vertices.push_back(0.0f);
					}
				}

				for(int row=0;row<(msize()-1);row++)
				{
					for (int col=0;col<(msize()-1);col++)
					{
						const int hits=(matrix[row][col]>0 ? 1 : 0)+(matrix[row][col+1]>0 ? 1 : 0)+(matrix[row+1][col]>0 ? 1 : 0)+(matrix[row+1][col+1]>0 ? 1 : 0);
						if(hits==4)
						{
							bundle.indices.push_back(matrix[row][col]-1);
							bundle.indices.push_back(matrix[row][col+1]-1);
							bundle.indices.push_back(matrix[row+1][col]-1);

							bundle.indices.push_back(matrix[row+1][col]-1);
							bundle.indices.push_back(matrix[row+1][col+1]-1);
							bundle.indices.push_back(matrix[row][col+1]-1);
						}
						else if(hits==3)
						{
							if(matrix[row][col]>0)
							{
								bundle.indices.push_back(matrix[row][col]-1);
							}
							if(matrix[row][col+1]>0)
							{
								bundle.indices.push_back(matrix[row][col+1]-1);
							}
							if(matrix[row+1][col]>0)
							{
								bundle.indices.push_back(matrix[row+1][col]-1);
							}
							if(matrix[row+1][col+1]>0)
							{
								bundle.indices.push_back(matrix[row+1][col+1]-1);
							}
						}
					}
				}
			}

			return bundle;
		}

		static const Matrix& render_character_to_matrix(const char character, const bool outline)
		{
			static const Matrix blank_matrix(msize(), std::vector<unsigned int>(msize(), 0));
			static std::vector<Matrix> matrices(128);
			static std::vector<Matrix> outline_matrices(128);

			const int character_id=static_cast<int>(character);
			if(character_id<33 || character_id>=128)
			{
				return blank_matrix;
			}

			const unsigned char* bitmap=get_character_bitmap(character);
			if(bitmap==0)
			{
				return blank_matrix;
			}

			Matrix& matrix=outline ? outline_matrices[character_id] : matrices[character_id];

			if(matrix.empty())
			{
				matrix=blank_matrix;

				for(int row=0;row<8;row++)
				{
					for (int col=0;col<8;col++)
					{
						if(bitmap[7-row] & 1 << col)
						{
							matrix[2+row*2][2+col*2]++;
							matrix[2+row*2-1][2+col*2]++;
							matrix[2+row*2][2+col*2-1]++;
							matrix[2+row*2+1][2+col*2]++;
							matrix[2+row*2][2+col*2+1]++;
						}
					}
				}

				for(int row=0;row<8;row++)
				{
					for (int col=0;col<8;col++)
					{
						if(matrix[2+row*2][2+col*2])
						{
							if(row<7 && matrix[2+row*2+2][2+col*2]>0)
							{
								matrix[2+row*2+1][2+col*2-1]++;
								matrix[2+row*2+1][2+col*2+1]++;
							}
							if(col<7 && matrix[2+row*2][2+col*2+2]>0)
							{
								matrix[2+row*2-1][2+col*2+1]++;
								matrix[2+row*2+1][2+col*2+1]++;
							}
							if(row<7 && col<7 && matrix[2+row*2+2][2+col*2+2]>0)
							{
								matrix[2+row*2+1][2+col*2+1]++;
							}
							if(row<7 && col>0 && matrix[2+row*2+2][2+col*2-2]>0)
							{
								matrix[2+row*2+1][2+col*2-1]++;
							}
							if(row>0 && col<7 && matrix[2+row*2-2][2+col*2+2]>0)
							{
								matrix[2+row*2-1][2+col*2+1]++;
							}
						}
					}
				}

				if(outline)
				{
					Matrix omatrix=blank_matrix;
					for(int row=1;row<(msize()-1);row++)
					{
						for (int col=1;col<(msize()-1);col++)
						{
							if(matrix[row][col]>0 && (matrix[row-1][col]==0 || matrix[row+1][col]==0 || matrix[row][col-1]==0 || matrix[row][col+1]==0))
							{
								omatrix[row][col]=1;
								if(matrix[row-1][col]==0)
								{
									omatrix[row-1][col]=1;
								}
								if(matrix[row+1][col]==0)
								{
									omatrix[row+1][col]=1;
								}
								if(matrix[row][col-1]==0)
								{
									omatrix[row][col-1]=1;
								}
								if(matrix[row][col+1]==0)
								{
									omatrix[row][col+1]=1;
								}
							}
						}
					}
					matrix=omatrix;
				}

				unsigned int i=1;
				for(int row=0;row<msize();row++)
				{
					for (int col=0;col<msize();col++)
					{
						if(matrix[row][col]>0)
						{
							matrix[row][col]=i++;
						}
					}
				}
			}

		    return matrix;
		}

		static const unsigned char* get_character_bitmap(const char character)
		{
			static unsigned char font[128][8] = {
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0000
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0001
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0002
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0003
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0004
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0005
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0006
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0007
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0008
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0009
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+000A
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+000B
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+000C
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+000D
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+000E
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+000F
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0010
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0011
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0012
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0013
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0014
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0015
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0016
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0017
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0018
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0019
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+001A
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+001B
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+001C
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+001D
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+001E
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+001F
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0020 (space)
			    { 0x18, 0x3C, 0x3C, 0x18, 0x18, 0x00, 0x18, 0x00},   // U+0021 (!)
			    { 0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0022 (")
			    { 0x36, 0x36, 0x7F, 0x36, 0x7F, 0x36, 0x36, 0x00},   // U+0023 (#)
			    { 0x0C, 0x3E, 0x03, 0x1E, 0x30, 0x1F, 0x0C, 0x00},   // U+0024 ($)
			    { 0x00, 0x63, 0x33, 0x18, 0x0C, 0x66, 0x63, 0x00},   // U+0025 (%)
			    { 0x1C, 0x36, 0x1C, 0x6E, 0x3B, 0x33, 0x6E, 0x00},   // U+0026 (&)
			    { 0x06, 0x06, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0027 (')
			    { 0x18, 0x0C, 0x06, 0x06, 0x06, 0x0C, 0x18, 0x00},   // U+0028 (()
			    { 0x06, 0x0C, 0x18, 0x18, 0x18, 0x0C, 0x06, 0x00},   // U+0029 ())
			    { 0x00, 0x66, 0x3C, 0xFF, 0x3C, 0x66, 0x00, 0x00},   // U+002A (*)
			    { 0x00, 0x0C, 0x0C, 0x3F, 0x0C, 0x0C, 0x00, 0x00},   // U+002B (+)
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x06},   // U+002C (,)
			    { 0x00, 0x00, 0x00, 0x3F, 0x00, 0x00, 0x00, 0x00},   // U+002D (-)
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x00},   // U+002E (.)
			    { 0x60, 0x30, 0x18, 0x0C, 0x06, 0x03, 0x01, 0x00},   // U+002F (/)
			    { 0x3E, 0x63, 0x73, 0x7B, 0x6F, 0x67, 0x3E, 0x00},   // U+0030 (0)
			    { 0x0C, 0x0E, 0x0C, 0x0C, 0x0C, 0x0C, 0x3F, 0x00},   // U+0031 (1)
			    { 0x1E, 0x33, 0x30, 0x1C, 0x06, 0x33, 0x3F, 0x00},   // U+0032 (2)
			    { 0x1E, 0x33, 0x30, 0x1C, 0x30, 0x33, 0x1E, 0x00},   // U+0033 (3)
			    { 0x38, 0x3C, 0x36, 0x33, 0x7F, 0x30, 0x78, 0x00},   // U+0034 (4)
			    { 0x3F, 0x03, 0x1F, 0x30, 0x30, 0x33, 0x1E, 0x00},   // U+0035 (5)
			    { 0x1C, 0x06, 0x03, 0x1F, 0x33, 0x33, 0x1E, 0x00},   // U+0036 (6)
			    { 0x3F, 0x33, 0x30, 0x18, 0x0C, 0x0C, 0x0C, 0x00},   // U+0037 (7)
			    { 0x1E, 0x33, 0x33, 0x1E, 0x33, 0x33, 0x1E, 0x00},   // U+0038 (8)
			    { 0x1E, 0x33, 0x33, 0x3E, 0x30, 0x18, 0x0E, 0x00},   // U+0039 (9)
			    { 0x00, 0x0C, 0x0C, 0x00, 0x00, 0x0C, 0x0C, 0x00},   // U+003A (:)
			    { 0x00, 0x0C, 0x0C, 0x00, 0x00, 0x0C, 0x0C, 0x06},   // U+003B (;)
			    { 0x18, 0x0C, 0x06, 0x03, 0x06, 0x0C, 0x18, 0x00},   // U+003C (<)
			    { 0x00, 0x00, 0x3F, 0x00, 0x00, 0x3F, 0x00, 0x00},   // U+003D (=)
			    { 0x06, 0x0C, 0x18, 0x30, 0x18, 0x0C, 0x06, 0x00},   // U+003E (>)
			    { 0x1E, 0x33, 0x30, 0x18, 0x0C, 0x00, 0x0C, 0x00},   // U+003F (?)
			    { 0x3E, 0x63, 0x7B, 0x7B, 0x7B, 0x03, 0x1E, 0x00},   // U+0040 (@)
			    { 0x0C, 0x1E, 0x33, 0x33, 0x3F, 0x33, 0x33, 0x00},   // U+0041 (A)
			    { 0x3F, 0x66, 0x66, 0x3E, 0x66, 0x66, 0x3F, 0x00},   // U+0042 (B)
			    { 0x3C, 0x66, 0x03, 0x03, 0x03, 0x66, 0x3C, 0x00},   // U+0043 (C)
			    { 0x1F, 0x36, 0x66, 0x66, 0x66, 0x36, 0x1F, 0x00},   // U+0044 (D)
			    { 0x7F, 0x46, 0x16, 0x1E, 0x16, 0x46, 0x7F, 0x00},   // U+0045 (E)
			    { 0x7F, 0x46, 0x16, 0x1E, 0x16, 0x06, 0x0F, 0x00},   // U+0046 (F)
			    { 0x3C, 0x66, 0x03, 0x03, 0x73, 0x66, 0x7C, 0x00},   // U+0047 (G)
			    { 0x33, 0x33, 0x33, 0x3F, 0x33, 0x33, 0x33, 0x00},   // U+0048 (H)
			    { 0x1E, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x1E, 0x00},   // U+0049 (I)
			    { 0x78, 0x30, 0x30, 0x30, 0x33, 0x33, 0x1E, 0x00},   // U+004A (J)
			    { 0x67, 0x66, 0x36, 0x1E, 0x36, 0x66, 0x67, 0x00},   // U+004B (K)
			    { 0x0F, 0x06, 0x06, 0x06, 0x46, 0x66, 0x7F, 0x00},   // U+004C (L)
			    { 0x63, 0x77, 0x7F, 0x7F, 0x6B, 0x63, 0x63, 0x00},   // U+004D (M)
			    { 0x63, 0x67, 0x6F, 0x7B, 0x73, 0x63, 0x63, 0x00},   // U+004E (N)
			    { 0x1C, 0x36, 0x63, 0x63, 0x63, 0x36, 0x1C, 0x00},   // U+004F (O)
			    { 0x3F, 0x66, 0x66, 0x3E, 0x06, 0x06, 0x0F, 0x00},   // U+0050 (P)
			    { 0x1E, 0x33, 0x33, 0x33, 0x3B, 0x1E, 0x38, 0x00},   // U+0051 (Q)
			    { 0x3F, 0x66, 0x66, 0x3E, 0x36, 0x66, 0x67, 0x00},   // U+0052 (R)
			    { 0x1E, 0x33, 0x07, 0x0E, 0x38, 0x33, 0x1E, 0x00},   // U+0053 (S)
			    { 0x3F, 0x2D, 0x0C, 0x0C, 0x0C, 0x0C, 0x1E, 0x00},   // U+0054 (T)
			    { 0x33, 0x33, 0x33, 0x33, 0x33, 0x33, 0x3F, 0x00},   // U+0055 (U)
			    { 0x33, 0x33, 0x33, 0x33, 0x33, 0x1E, 0x0C, 0x00},   // U+0056 (V)
			    { 0x63, 0x63, 0x63, 0x6B, 0x7F, 0x77, 0x63, 0x00},   // U+0057 (W)
			    { 0x63, 0x63, 0x36, 0x1C, 0x1C, 0x36, 0x63, 0x00},   // U+0058 (X)
			    { 0x33, 0x33, 0x33, 0x1E, 0x0C, 0x0C, 0x1E, 0x00},   // U+0059 (Y)
			    { 0x7F, 0x63, 0x31, 0x18, 0x4C, 0x66, 0x7F, 0x00},   // U+005A (Z)
			    { 0x1E, 0x06, 0x06, 0x06, 0x06, 0x06, 0x1E, 0x00},   // U+005B ([)
			    { 0x03, 0x06, 0x0C, 0x18, 0x30, 0x60, 0x40, 0x00},   // U+005C (\)
			    { 0x1E, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1E, 0x00},   // U+005D (])
			    { 0x08, 0x1C, 0x36, 0x63, 0x00, 0x00, 0x00, 0x00},   // U+005E (^)
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF},   // U+005F (_)
			    { 0x0C, 0x0C, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+0060 (`)
			    { 0x00, 0x00, 0x1E, 0x30, 0x3E, 0x33, 0x6E, 0x00},   // U+0061 (a)
			    { 0x07, 0x06, 0x06, 0x3E, 0x66, 0x66, 0x3B, 0x00},   // U+0062 (b)
			    { 0x00, 0x00, 0x1E, 0x33, 0x03, 0x33, 0x1E, 0x00},   // U+0063 (c)
			    { 0x38, 0x30, 0x30, 0x3e, 0x33, 0x33, 0x6E, 0x00},   // U+0064 (d)
			    { 0x00, 0x00, 0x1E, 0x33, 0x3f, 0x03, 0x1E, 0x00},   // U+0065 (e)
			    { 0x1C, 0x36, 0x06, 0x0f, 0x06, 0x06, 0x0F, 0x00},   // U+0066 (f)
			    { 0x00, 0x00, 0x6E, 0x33, 0x33, 0x3E, 0x30, 0x1F},   // U+0067 (g)
			    { 0x07, 0x06, 0x36, 0x6E, 0x66, 0x66, 0x67, 0x00},   // U+0068 (h)
			    { 0x0C, 0x00, 0x0E, 0x0C, 0x0C, 0x0C, 0x1E, 0x00},   // U+0069 (i)
			    { 0x30, 0x00, 0x30, 0x30, 0x30, 0x33, 0x33, 0x1E},   // U+006A (j)
			    { 0x07, 0x06, 0x66, 0x36, 0x1E, 0x36, 0x67, 0x00},   // U+006B (k)
			    { 0x0E, 0x0C, 0x0C, 0x0C, 0x0C, 0x0C, 0x1E, 0x00},   // U+006C (l)
			    { 0x00, 0x00, 0x33, 0x7F, 0x7F, 0x6B, 0x63, 0x00},   // U+006D (m)
			    { 0x00, 0x00, 0x1F, 0x33, 0x33, 0x33, 0x33, 0x00},   // U+006E (n)
			    { 0x00, 0x00, 0x1E, 0x33, 0x33, 0x33, 0x1E, 0x00},   // U+006F (o)
			    { 0x00, 0x00, 0x3B, 0x66, 0x66, 0x3E, 0x06, 0x0F},   // U+0070 (p)
			    { 0x00, 0x00, 0x6E, 0x33, 0x33, 0x3E, 0x30, 0x78},   // U+0071 (q)
			    { 0x00, 0x00, 0x3B, 0x6E, 0x66, 0x06, 0x0F, 0x00},   // U+0072 (r)
			    { 0x00, 0x00, 0x3E, 0x03, 0x1E, 0x30, 0x1F, 0x00},   // U+0073 (s)
			    { 0x08, 0x0C, 0x3E, 0x0C, 0x0C, 0x2C, 0x18, 0x00},   // U+0074 (t)
			    { 0x00, 0x00, 0x33, 0x33, 0x33, 0x33, 0x6E, 0x00},   // U+0075 (u)
			    { 0x00, 0x00, 0x33, 0x33, 0x33, 0x1E, 0x0C, 0x00},   // U+0076 (v)
			    { 0x00, 0x00, 0x63, 0x6B, 0x7F, 0x7F, 0x36, 0x00},   // U+0077 (w)
			    { 0x00, 0x00, 0x63, 0x36, 0x1C, 0x36, 0x63, 0x00},   // U+0078 (x)
			    { 0x00, 0x00, 0x33, 0x33, 0x33, 0x3E, 0x30, 0x1F},   // U+0079 (y)
			    { 0x00, 0x00, 0x3F, 0x19, 0x0C, 0x26, 0x3F, 0x00},   // U+007A (z)
			    { 0x38, 0x0C, 0x0C, 0x07, 0x0C, 0x0C, 0x38, 0x00},   // U+007B ({)
			    { 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 0x18, 0x00},   // U+007C (|)
			    { 0x07, 0x0C, 0x0C, 0x38, 0x0C, 0x0C, 0x07, 0x00},   // U+007D (})
			    { 0x6E, 0x3B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},   // U+007E (~)
			    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}    // U+007F
			};

			const int i=static_cast<int>(character);
			if(i>=0 && i<128)
			{
				return (&(font[i][0]));
			}

			return 0;
		}
	};
};

}

}

#endif /* SCRIPTING_FIGURE_OF_TEXT_H_ */
