-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/procedure_handler.adb,v $
--  Description     : handle the procedures of a project
--  Author          : Michael Erdmann                                        --
--  Created         : 8.8.2001                                               --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2002/05/30 07:19:16 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Text_IO;              use Ada.Text_IO;
with Ada.Strings.Unbounded;    use Ada.Strings.Unbounded;

with Project_File;             use Project_File;
with Datasource_Handler;       use Datasource_Handler;
with Contents_Manager;         use Contents_Manager;
with Text_Block;               use Text_Block;
with Preferences;              use Preferences;

with Contents_Manager.GUI;     use Contents_Manager.GUI;
with Gui_Preferences;          use GUI_Preferences;
with GUI_Logging;              use GUI_Logging;
with GUI_Tools;                use GUI_Tools;
with GUI_Worksheet;            use GUI_Worksheet;

package body Procedure_Handler  is

   Version : constant String :=
      "$Id: procedure_handler.adb,v 1.9 2002/05/30 07:19:16 merdmann Exp $";

   -- ********************************************************************* --
   -- **                 L O C A L   D A T A                             ** --
   -- ********************************************************************* --
   Procedure_Section : Element_ID := Null_Element_ID;

   -- ********************************************************************* --
   -- **              L O C A L    P R O C E D U R E S                   ** --
   -- ********************************************************************* --

   -----------------
   -- Instanciate --
   -----------------
   function Instanciate(
      Key : in String ) return Contents_Handler.Handle is
      Result : Handle := new Object;
   begin
      Put_Line("Instanciate procedure entry :" & Key );
      return Contents_Handler.Handle(Result);
   end Instanciate;

   -- ********************************************************************* --
   -- **               P U B L I C   P R O C E D U R E S                 ** --
   -- ********************************************************************* --

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      This : in out Object ) is
      -- initialize the query (instance)
   begin
      Initialize( Contents_Handler.Object(This) );
      This.Id := GUI.Add_Element( Key(This), Procedure_Section, Self(This) );
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      This : in out Object ) is
      Name : constant Unbounded_String := Key( This );
   begin
      if GUI_Worksheet.Is_Open( Name ) then
         GUI_Worksheet.Remove_Sheet( Name );
      end if;
      Project_File.Remove( Procedure_Prefix, Key(This) );
      GUI.Delete_Element( This.Id );
      Finalize( Contents_Handler.Object(This) );
   end Finalize;

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize is
      Item : Handle := new Object;
   begin
      Item.Constructor  := Instanciate'Access;
      Procedure_Section := Add_Element(
                               Title => "Procedures",
                               Content => Contents_Handler.Handle(Item) );

      Contents_Manager.Register( Procedure_Prefix, Instanciate'Access );
      Contents_Manager.Initialize( Procedure_Prefix, Procedure_Section);

      Put_Line("procedure handler initialized");
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize is
   begin
      Contents_Manager.Finalize( Procedure_Prefix );
   end Finalize;

   -----------
   -- Value --
   -----------
   procedure Value(
      This : in Object;
      Text : out Text_Buffer;
      Last : out Positive ) is
   begin
      GUI_Editor.Get_Result( This.Editor, Text, Last);
   end Value;

   ----------
   -- Save --
   ----------
   procedure Save(
      Item    : in out Object;
      Value   : in Text_Buffer;
      Name    : in String := "" ) is
      -- save the procedure text in the project file. If the procedure
      -- name does not exist yet create a new entry.
      -- Afterwards update the project file.
      Current : constant String := Key( Item );
      PName   : constant String := To_String( Value(1) );
      NewObj  : Handle;
   begin
      if not Section_Exists( Procedure_Prefix, PName ) then
         NewObj := Handle(Contents_Manager.Create( Procedure_Prefix, PName ) );
         Gui_Worksheet.Remove_Sheet( Item.Editor );
      end if;

      Project_File.Update( Procedure_Prefix, PName, Value);
   end Save;

   ----------
   -- Load --
   ----------
   procedure Load(
      Item  : in out Object;
      Value : out Text_Buffer;
      Last  : out Natural ) is
      -- load the procedure from the project file. If nothing is
      -- there generate a template procedure.
      Description : constant Text_Buffer := (
                  To_Unbounded_String( "-- Fill in your description" ),
                  To_Unbounded_String( "-- Author: " & Preference(Prf_Author))
                  );

      SQL_Body    : constant Text_Buffer := (
                  To_Unbounded_String( "( IN aName type )  "),
                  To_Unbounded_String( "MODIFIES SQL DATA  "),
                  To_Unbounded_String( "BEGIN              "),
                  To_Unbounded_String( "END                ")
                  );
   begin
      Project_File.Get(
         Procedure_Prefix,
         Key(Item),
         Value,
         Last
      );
   exception
      when Section_Not_Found =>
         Last := Value'First;
         Value(Last) := To_Unbounded_String( "PROCEDURE/FUNCTION " & Key(Item) );
         Last := Last + 1;

         Put_Text_Block( Value, Description, Last );
         Put_Text_Block( Value, SQL_Body, Last );
   end Load;

   --------------
   -- Selected --
   --------------
   procedure Selected(
      This        : in out Object ) is
      --
      -- R.1 - If only the class is selected the toolbar is
      --       activated for the query class.
      --
      -- R.2 - If an instance is selected, the instance is
      --       loaded into the editor.
      --
   begin
      if This.Constructor /= null then                              -- R.1
         GUI_Tools.Activate( Procedure_Prefix  );
      else                                                          -- R.2
         Open_Editor( This );
      end if;
   end Selected;

   -----------------
   -- Open_Editor --
   -----------------
   procedure Open_Editor(
      This : in out Object ) is
      -- load the referenced instance into the query editor
      Name        : Unbounded_String := Key( This );
   begin
      if not GUI_Worksheet.Is_Open( Name ) then
         Initialize( This.Editor );
         GUI_Worksheet.Add_Sheet( This.Editor, Self(This), Name );
      else
         GUI_Worksheet.Bring_In_Front( Name );
      end if;
   end Open_Editor;

   ------------------
   -- Close_Editor --
   ------------------
   procedure Close_Editor(
      This : in out Object ) is
   begin
      GUI_Worksheet.Remove_Sheet( This.Editor );
   end Close_Editor;

   -------------
   -- Execute --
   -------------
   procedure Execute(
      This : in out Object ) is
      -- execute the current editor contents
      Name : Unbounded_String := Key( This );
      Text : Text_Buffer( 1..Max_Section_Length );
      Last : Positive;
   begin
      if Datasource_Handler.Current_Database = Not_Attached then
         Error("No data base connected");
         return;
      end if;
      -- Datasource_Handler.Execute( GUI_Editor.Current_Editor_Contents );
      Log("Removing " & To_String(Name) );
      Datasource_Handler.Execute( "DROP " & To_String(Name) );

      Log("Installing " & To_String(Name) );

      GUI_Editor.Query(This.Editor, Text, Last);
      Datasource_Handler.Execute(
         "CREATE " & To_String(Name) &  To_String( Text(1..Last) )
      );
      Log("Done");
   end Execute;

end Procedure_Handler;
