# -*-ruby; coding: utf-8 -*-
# vim:set ft=ruby:
#
# Copyright (c) 2009, 2011, 2012 Kazuhiro NISHIYAMA
#
# This program is free software with ABSOLUTELY NO WARRANTY.
# You can re-distribute and/or modify this program under
# the same terms of the Ruby's license.
#

=begin

== Abstract

Reply title of URL.

== Configuration

  BotConfig << {
    :name => :TitleBot,
    :ch => //,
    :timeout => 10,
    :headers => {
      #"User-Agent" => "Ruby/#{RUBY_VERSION}",
    }
  }

=end

begin
  require 'iconv'
rescue LoadError
end
require 'nkf'
require 'open-uri'
require 'tempfile'
require 'timeout'
require 'tmpdir'

begin
  require 'cgi/util'
rescue LoadError
  require 'cgi'
end

begin
  require 'rubygems'
  require 'nokogiri'
rescue LoadError
end

unless OpenURI.redirectable?(URI("http://example.com"), URI("https://example.com"))
  def OpenURI.redirectable?(uri1, uri2) # :nodoc:
    uri1.scheme.downcase == uri2.scheme.downcase ||
    (/\A(?:http|ftp)\z/i =~ uri1.scheme && /\A(?:https?|ftp)\z/i =~ uri2.scheme)
  end
end

module URL2Title
  module_function

  BLACKLIST_HOST = [
    /\blocalhost\b/,
    /\A127\./,
    /\A192\.168\./,
    /\A10\./,
    /\A169\.254\./,
  ]

  if defined?(::Nadoka::VERSION)
    DEFAULT_USER_AGENT = "Ruby/#{RUBY_VERSION} NadokaTitleBot/#{Nadoka::VERSION}"
  else
    DEFAULT_USER_AGENT = "Ruby/#{RUBY_VERSION}"
  end

  def get_title(url, headers)
    uri = URI(url)
    info = { :uri => uri }
    info[:errors] = []
    case uri.host
    when *BLACKLIST_HOST
      info[:title] = "(ignored)"
      return info
    when /\A(?:twitter\.com)\z/
      # Twitter response depends on User-Agent
      headers = headers.merge!('User-Agent' => DEFAULT_USER_AGENT)
    end
    headers['User-Agent'] ||= DEFAULT_USER_AGENT
    headers = {
      :content_length_proc => proc{|x| raise Errno::EFBIG if x && x > 1048576},
    }.merge!(headers)
    uri.open(headers) do |f|
      info[:uri] = f.base_uri
      body = f.read

      if /\.blog\d+\.fc2\.com\z/ =~ uri.host
        # set last content-type only
        f.meta_add_field("content-type", f.meta["content-type"].split(/, /)[-1])
      end

      case f.content_type
      when /\Atext\//
        charset = f.charset{} # without block, returns "iso-8859-1"

        # Content-Encoding
        case
        when f.content_encoding.empty?
          # ignore
        when f.content_encoding.any?{|c_e| /deflate/ =~ c_e }
          require "zlib"
          body = Zlib::Inflate.inflate(body)
        when f.content_encoding.any?{|c_e| /gzip/ =~ c_e }
          require "zlib"
          body = Zlib::GzipReader.new(StringIO.new(body)).read || ''
        end


        # encoding
        if charset
          # ok
        elsif /charset=[\'\"]?([^;\'\">]+)/ni =~ body
          charset = $1
        end
        case charset
        when /shift[_-]jis/i
          charset = "cp932"
        when /euc-jp/i # euc-jp, x-euc-jp
          charset = "eucjp-ms"
        end
        if /\A(?:utf-8|eucjp-ms)\z/i =~ charset
          # avoid #<ArgumentError: invalid byte sequence in UTF-8>
          # or Iconv::IllegalSequence
          body = NKF.nkf("-wm0x --ic=#{charset}", body)
        elsif charset
          charset.sub!(/\Ax-?/i, '')
          if defined?(Iconv)
            begin
              body = Iconv.conv("utf-8", charset, body)
            rescue Iconv::IllegalSequence => e
              info[:errors] << e
              body = NKF.nkf("-wm0x", body)
            rescue Iconv::InvalidEncoding => e
              info[:errors] << e
              body = NKF.nkf("-wm0x", body)
            end
          else
            body = body.encode("utf-8", charset, :invalid => :replace, :undef => :replace)
          end
        else
          body = NKF.nkf("-wm0x", body)
        end

        title = nil
        if %r"<title\b(?>[^<>]*)>(.*?)</title(?>[^<>]*)>"miu =~ body
          title = $1
        end

        case f.base_uri.host
        when /\A(?:twitter\.com)\z/
          if defined?(::Nokogiri)
            doc = Nokogiri::HTML(body, uri.to_s, 'utf-8')
            title = doc.xpath("//meta[@property='og:description'][1]/@content").text
          end
        when /\A(?:mobile\.twitter\.com)\z/
          if defined?(::Nokogiri)
            doc = Nokogiri::HTML(body, uri.to_s, 'utf-8')
            tweet, = doc.css('.main-tweet .tweet-text')
            if tweet
              tweet = tweet.inner_html
              tweet.gsub!(/<.*?>/, '')
              tweet.strip!
              title = tweet unless tweet.empty?
            end
          end
        when /\A(?:github\.com)\z/
          # ignore og:title
        when /\A(?:www\.youtube\.com)\z/
          if %r'"title":"(.+?)"' =~ body
            title = $1
          end
        else
          if defined?(::Nokogiri)
            doc ||= Nokogiri::HTML(body, uri.to_s, 'utf-8')
            og_title = doc.xpath("//meta[@property='og:title'][1]/@content").text
            unless og_title.empty?
              # title is escaped string when get by regexp
              title = CGI.escapeHTML(og_title)
            end
          elsif /<meta property="og:title" content="(.+?)">/ =~ body
            title = $1
          end
          if uri.fragment && defined?(::Nokogiri)
            begin
              doc ||= Nokogiri::HTML(body, uri.to_s, 'utf-8')
              xpath = "//*[@id='#{uri.fragment}' or @name='#{uri.fragment}']"
              fragment_element = doc.xpath(xpath)
              # tDiary style
              unless fragment_element.xpath("span[@class='sanchor']").empty?
                fragment_element = fragment_element.xpath("..")
              end
              info[:fragment_text] = truncate(fragment_element.text)
            rescue Exception => e
              info[:errors] << e
            end
          end
        end
        if defined?(::Nokogiri)
          doc ||= Nokogiri::HTML(body, uri.to_s, 'utf-8')
          canonical_uri = doc.xpath("//link[@rel='canonical'][1]/@href")
          unless canonical_uri.empty?
            info[:uri] = shorten_url(canonical_uri.text)
          end
        elsif /<link rel="canonical" href="(.+?)"/i =~ body
          info[:uri] = shorten_url(CGI.unescapeHTML($1))
        end
        if title
          title = CGI.unescapeHTML(title)
          title.gsub!(/\xE3\x80\x9C/u, "\xEF\xBD\x9E") # WAVE DASH -> FULLWIDTH TILDE
        end
        info[:title] = title || body
        return info
      when /\Aimage\//
        if f.respond_to?(:path) && f.path
          info[:title] = `identify '#{f.path}'`.sub(/\A#{Regexp.quote(f.path)}/, '').strip
          return info
        else
          info[:title] = "(unknown image format)"
          temp = Tempfile.new("titlebot", :encoding => 'ascii-8bit')
          begin
            temp.write(body)
            temp.close
            info[:title] = `identify '#{temp.path}'`.sub(/\A#{Regexp.quote(temp.path)}/, '').strip
          ensure
            temp.unlink
          end
          return info
        end
      else
        info[:title] = "#{f.content_type} #{f.size} bytes"
        return info
      end
    end
  rescue Errno::EFBIG
    info[:title] = "(too big)"
    return info
  end

  def truncate s
    if /\A(?>(.{197})....)/mu =~ s
      return $1+'...'
    else
      return s
    end
  end

  def prepare_url(url)
    url.sub(/\/\#!\//, '/')
  end

  def shorten_url(url)
    case url
    when %r!\Ahttps?://www\.amazon\.co\.jp/.*(/dp/[^/]+)!
      URI("https://amazon.jp#{$1}")
    else
      # default: do nothing
      URI(url)
    end
  rescue URI::InvalidURIError
    url
  end

  def cleanup(title)
    title.gsub(/(\s|&nbsp;|&#xA0;|&#160;|\u{A0})+/i, ' ')
  end

  def url2title(url, headers)
    url = prepare_url(url)
    info = get_title(url, headers)
    info[:title] = truncate(cleanup(info[:title]))
    info
  end
end

if __FILE__ == $0
  require File.expand_path('../../ndk/version', __FILE__)
  def u2t(url)
    firefox = "Mozilla/5.0 (X11; Ubuntu; Linux x86_64; rv:57.0) Gecko/20100101 Firefox/57.0"
    ua = "NadokaTitleBot/#{Nadoka::VERSION}"
    headers = {
      "User-Agent" => "#{firefox} Ruby/#{RUBY_VERSION} #{ua}",
    }
    URL2Title.url2title(url, headers)
  rescue
    puts $!.backtrace
    info = { :uri => url }
    info[:title] = $!.inspect
    info
  end
  URL2Title::BLACKLIST_HOST << /\.test\z/
  if ARGV.empty?
    # TODO: test
  else
    ARGV.each do |url|
      info = u2t(url)
      p info
      puts info[:uri]
      puts info[:title]
    end
  end
  exit
end

class TitleBot < Nadoka::NDK_Bot
  include URL2Title

  def bot_initialize
    if @bot_config.key?(:channels)
      channels = '\A(?:' + @bot_config[:channels].collect{|ch|
        Regexp.quote(ch)
      }.join('|') + ')\z'
      @available_channel = Regexp.compile(channels)
    else
      @available_channel = @bot_config.fetch(:ch, //)
    end

    @same_bot = @bot_config.fetch(:same_bot, /(?!)/)
    @nkf_options = @bot_config.fetch(:nkf, "--oc=CP50221 --ic=UTF-8 --fb-xml")
    @timeout = @bot_config.fetch(:timeout, 60)
    @skip_nkf_msg = @bot_config.fetch(:skip_nkf_msg, false)
    @fragment_size_range = @bot_config.fetch(:fragment_size_range, 5..100)
    @headers = @bot_config.fetch(:headers, {})
  end

  def send_notice(ch, msg)
    msg = msg.tr("\r\n", "  ")
    if @nkf_options
      msg = NKF.nkf(@nkf_options, msg)
    end
    super(ch, msg)
  end

  def on_privmsg prefix, ch, msg
    return unless @available_channel === ch

    msg = NKF.nkf('-w', msg) unless @skip_nkf_msg
    case msg
    when /^[^<>]+>/
      # ignore messages to other bots
    when /https?:/
      return if @state.channel_users(ccn(ch)).find{|x| @same_bot =~ x }

      msg.scan(%r<(?<l>\()?(?<url>https?://(?(<l>)(?<ascii_without_paren>[!-~&&[^()]]+|\(\g<ascii_without_paren>\))+|[!-~]+))>) do
        url = $~[:url]
        info = Timeout.timeout(@timeout) do
          url2title(url, @headers)
        end
        return unless info[:title]
        if url != info[:uri].to_s
          send_notice(ch, "title bot: #{info[:title]} - #{info[:uri]}")
        else
          send_notice(ch, "title bot: #{info[:title]}")
        end
        if info[:fragment_text]
          # ignore when fragment_text is too long or too short
          if @fragment_size_range === info[:fragment_text].size
            send_notice(ch, "title bot:: #{info[:fragment_text]}")
          end
        end
        info[:errors].each do |e|
          @manager.ndk_error e
          send_notice(ch, "title bot error: #{e}")
        end
      end
    end
  rescue Exception => e
    send_notice(ch, "title bot error! #{e}")
    @manager.ndk_error e
  end
end
