/* pdu-tcp.c
 
   PDU builder for TCP segments

   Copyright (C) 2007, 2008, 2009, 2010, 2011, 2012 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"

#define TCPOLEN_MD5             18
#define TCPOLEN_MAXSEG          4
#define TCPOLEN_WINDOW          3
#define TCPOLEN_TIMESTAMP       10

struct ipv4_pseudo_hdr {
    in_addr_t saddr;
    in_addr_t daddr;
    uint8_t zero;
    uint8_t protocol;
    uint16_t len;
};

struct ipv6_pseudo_hdr {
    ip6_addr_t saddr;
    ip6_addr_t daddr;
    uint32_t len;
    uint16_t zero1;
    uint8_t zero2;
    uint8_t nxt;
};

static void
build(const GNode *pdu, void *dest)
{
    struct tcp_hdr *tcp;
    uint16_t u16;
    uint32_t u32;
    void *field;
    size_t opts_len;

    tcp = dest;

    /*
     * We take care of the header size (tcp->doff) later in the post-builder
     * since we don't know at this point the size of the entire header.
     */

    u16 = htons( (field = _pb_pdata(pdu, "src") ) ? num_next(field) : 1073);
    SSVAL(tcp, offsetof(struct tcp_hdr, th_sport), u16);

    u16 = htons( (field = _pb_pdata(pdu, "dst") ) ? num_next(field) : 80);
    SSVAL(tcp, offsetof(struct tcp_hdr, th_dport), u16);

    u32 = htonl(num_next(_pb_pdata(pdu, "seq") ) );
    SIVAL(tcp, offsetof(struct tcp_hdr, th_seq), u32);

    u32 = htonl(num_next(_pb_pdata(pdu, "ack-seq") ) );
    SIVAL(tcp, offsetof(struct tcp_hdr, th_ack), u32);

    u16 = htons( (field = _pb_pdata(pdu, "window") ) ? num_next(field) : 8192);
    SSVAL(tcp, offsetof(struct tcp_hdr, th_win), u16);

    u16 = htons(num_next(_pb_pdata(pdu, "urg-ptr") ) );
    SSVAL(tcp, offsetof(struct tcp_hdr, th_urp), u16);

    tcp->th_x2 = 0;

    /* TCP flags */
    if ( (field = _pb_pdata(pdu, "flags") ) )
	tcp->th_flags = num_next(field);
    if (_pb_pdata(pdu, "urg") )
	tcp->th_flags |= TH_URG;
    if (_pb_pdata(pdu, "ack") )
	tcp->th_flags |= TH_ACK;
    if (_pb_pdata(pdu, "psh") )
	tcp->th_flags |= TH_PUSH;
    if (_pb_pdata(pdu, "rst") )
	tcp->th_flags |= TH_RST;
    if (_pb_pdata(pdu, "syn") )
	tcp->th_flags |= TH_SYN;
    if (_pb_pdata(pdu, "fin") )
	tcp->th_flags |= TH_FIN;

    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
    tcp->th_off = (sizeof(struct tcp_hdr) + opts_len)/4; /* # of words in the
							    TCP hdr */
}

static void
postbuild(const GNode *pdu, void *dest, void *prev_pdu_hdr)
{
    struct tcp_hdr *tcp;
    struct ip_hdr *ip;
    uint16_t cksum;
    void *field;
    size_t opts_len, payload_len;

    tcp = dest;

    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

    /*
     * If we have a numspec for the checksum we force a (most likely bogus)
     * checksum, otherwise we calculate the real checksum.
     */
    if ( (field = _pb_pdata(pdu, "cksum") ) ) {
	cksum = htons(num_next(field) );
    } else {
	/*
	 * There is not explicit (most likely bogus) checksum. Calculate
	 * one.
	 */

	/* Must be 0 before calculating cksum! */
	SSVAL(tcp, offsetof(struct tcp_hdr, th_sum), 0);

	ip = prev_pdu_hdr;

	if (ip) {
	    /*
	     * PDU definition does include an explicit IP PDU. Get everything
	     * we need for the pseudo-header from there.
	     */

	    if (ip->ip_v == 4) {
		/* IPv4 packet */
		struct ipv4_pseudo_hdr *ph;
		int segment_size;

		segment_size = sizeof(struct tcp_hdr) + opts_len + payload_len;

		ph = xmalloc(sizeof(struct ipv4_pseudo_hdr) + segment_size);
		ph->saddr = ip->ip_src;
		ph->daddr = ip->ip_dst;
		ph->zero = 0;
		ph->protocol = ip->ip_p;
		ph->len = htons(segment_size);
		memcpy( (uint8_t *) ph + sizeof(struct ipv4_pseudo_hdr),
		       dest, segment_size);

		cksum = _pb_cksum(ph, sizeof(struct ipv4_pseudo_hdr)
				      + segment_size);

		free(ph);
	    } else {
		/* IPv6 packet */
		struct ipv6_pseudo_hdr *ph;
		int segment_size;
		struct ip6_hdr *ip6;

		ip6 = prev_pdu_hdr;

		segment_size = sizeof(struct tcp_hdr) + opts_len + payload_len;
		ph = xmalloc(sizeof(struct ipv6_pseudo_hdr) + segment_size);

		ph->saddr = ip6->ip6_src;
		ph->daddr = ip6->ip6_dst;
		ph->len = htonl(segment_size);
		ph->zero1 = 0;
		ph->zero2 = 0;
		ph->nxt = IPPROTO_TCP;

		memcpy( (uint8_t *) ph + sizeof(struct ipv6_pseudo_hdr),
		       dest, segment_size);

		cksum = _pb_cksum(ph, sizeof(struct ipv6_pseudo_hdr)
				  + segment_size);

		free(ph);
	    }
	} else {
	    /*
	     * PDU definition does not include an explicit IP PDU. Get
	     * everything we need for the pseudo-header from user-specified
	     * data and some assumptions (IP proto is 6, and IP version is 4.)
	     */

	    struct ipv4_pseudo_hdr *ph;
	    int segment_size;

	    segment_size = sizeof(struct udp_hdr) + opts_len + payload_len;

	    ph = xmalloc(sizeof(struct ipv4_pseudo_hdr) + segment_size);
	    ph->saddr = ip_next(_pb_pdata(pdu, "srcip") );
	    ph->daddr = ip_next(_pb_pdata(pdu, "dstip") );
	    ph->zero = 0;
	    ph->protocol = IP_PROTO_TCP;
	    ph->len = htons(segment_size);
	    memcpy( (uint8_t *) ph + sizeof(struct ipv4_pseudo_hdr),
		   dest, segment_size);

	    cksum = _pb_cksum(ph, sizeof(struct ipv4_pseudo_hdr)
		    + segment_size);

	    free(ph);
	}
    }

    SSVAL(tcp, offsetof(struct tcp_hdr, th_sum), cksum);
}

#if 0
static void
pdu_tcphdr_dumper(pdu_t *p, const char *prefix)
{
    struct tcphdr_options *hdr_data;

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Source port: %s\n", prefix, num_info(hdr_data->src_port) );
    printf("%s    Destination port: %s\n", prefix, num_info(hdr_data->dst_port) );
    printf("%s    Sequence number: %s\n", prefix, num_info(hdr_data->sequence_number) );
    printf("%s    Ack/Seq: %s\n", prefix, num_info(hdr_data->ack_seq) );
    printf("%s    Window: %s\n", prefix, num_info(hdr_data->window) );
    printf("%s    Urgent pointer: %s\n", prefix, num_info(hdr_data->urg_ptr) );
    printf("%s    TCP flags: ", prefix);

    if (hdr_data->flags) {
	if (hdr_data->flags & TH_URG) printf("URG ");
	if (hdr_data->flags & TH_ACK) printf("ACK ");
	if (hdr_data->flags & TH_PUSH) printf("PSH ");
	if (hdr_data->flags & TH_RST) printf("RST ");
	if (hdr_data->flags & TH_SYN) printf("SYN ");
	if (hdr_data->flags & TH_FIN) printf("FIN ");
    } else
	printf("none set");
    printf("\n");

    printf("%s    TCP checksum: %s\n", prefix,
	   hdr_data->cksum ? num_info(hdr_data->cksum) : "automatic");
}
#endif

/********************************************************************
 *			       Options                              *
 ********************************************************************/

static void
tcpopt_eol(const GNode *option _U_, void *dest, void *curr_pdu_hdr _U_,
	   void *prev_pdu _U_)
{

    *(uint8_t *) dest = TCP_OPT_EOL;
}

static void
tcpopt_nop(const GNode *option _U_, void *dest, void *curr_pdu_hdr _U_,
	   void *prev_pdu _U_)
{
    *(uint8_t *) dest = TCP_OPT_NOP;
}

static void
tcpopt_maxseg(const GNode *option, void *dest, void *curr_pdu_hdr _U_,
	      void *prev_pdu _U_)
{
    uint16_t mss;

    *(uint8_t *) dest++ = TCP_OPT_MSS;
    *(uint8_t *) dest++ = TCPOLEN_MAXSEG;
    mss = htons(num_next( ( (struct node_data *) option->data)->_data_parm.data) );
    SSVAL(dest, 0, mss);
}

#if 0
static void
tcpopt_maxseg_dumper(pdu_option_t *p, const char *prefix)
{
    struct tcp_mss_option *option_data;

    option_data = p->data;

    printf("%s      Parameters:\n", prefix);
    printf("%s        Maximum Segment Size: %s\n", prefix,
	    num_info(option_data->mss) );
}
#endif

static void
tcpopt_window(const GNode *option, void *dest, void *curr_pdu_hdr _U_,
	      void *prev_pdu _U_)
{
    *(uint8_t *) dest++ = TCP_OPT_WSCALE;
    *(uint8_t *) dest++ = TCPOLEN_WINDOW;
    *(uint8_t *) dest = num_next( ( (struct node_data *) option->data)->_data_parm.data);
}

#if 0
static void
tcpopt_window_dumper(pdu_option_t *p, const char *prefix)
{
    struct tcp_wscale_option *option_data;

    option_data = p->data;

    printf("%s      Parameters:\n", prefix);
    printf("%s        Window scale factor: %s\n", prefix,
	    num_info(option_data->wscale) );
}
#endif

/*
 * RFC2385 describes how the TCP MD5 signature is calculated.
 */
static void
tcpopt_signature(const GNode *option,
		 void *dest,
		 void *curr_pdu_hdr,	/* Points to TCP header */
		 void *prev_pdu_hdr)	/* Points to IP header */
{
    MD5_CTX c;
    struct ip_hdr *ip;
    const GNode *pdu;
    size_t hdr_len, opts_len, payload_len;
    struct payload *secret;

    pdu = _pb_parent_pdu(option);
    hdr_len = ( (struct node_data *) pdu->data)->_data_pdu.hdr_len;
    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

    *(uint8_t *) dest++ = TCP_OPT_MD5;
    *(uint8_t *) dest++ = TCPOLEN_MD5;

    MD5_Init(&c);

    ip = (struct ip_hdr *) prev_pdu_hdr;

    /*
     * Calculate MD5 sum of pseudo header (step 1 in section 2.0 of
     * RFC2385)
     */
    if (ip->ip_v == 4) {
	struct ipv4_pseudo_hdr ph;

	ph.saddr = ip->ip_src;
	ph.daddr = ip->ip_dst;
	ph.zero = 0;
	ph.protocol = ip->ip_p;
	/*
	 * Remember: the length field in the pseudo-header is the length
	 * of the TCP segment, that is, size of the TCP header (which may
	 * or may not include TCP options) + the size of the TCP payload.
	 * This easy to calculate: total IP packet size - size of IP header.
	 */
	ph.len = htons(hdr_len + opts_len + payload_len);

	MD5_Update(&c, &ph, sizeof(ph) );
    } else {
	/* IPv6 packet */
	struct ipv6_pseudo_hdr ph;
	struct ip6_hdr *ip6;

	ip6 = (struct ip6_hdr *) prev_pdu_hdr;

	memcpy(&ph.saddr, &ip6->ip6_src, INET_ADDRSTRLEN);
	memcpy(&ph.daddr, &ip6->ip6_src, INET_ADDRSTRLEN);
	ph.zero1 = 0;
	ph.zero2 = 0;
	ph.nxt = IPPROTO_TCP;
	ph.len = htons(hdr_len + opts_len + payload_len);

	MD5_Update(&c, &ph, sizeof(ph) );
    }

    /*
     * Calculate MD5 sum of TCP header (_excluding_ options!) (step 2 in
     * section 2.0 of RFC2385).
     */
    MD5_Update(&c, curr_pdu_hdr, hdr_len);

    /* Calculate MD5 sum of TCP data (step 3 in section 2.0 of RFC2385) */
    MD5_Update(&c, curr_pdu_hdr + hdr_len + opts_len, payload_len);

    secret = ( (struct node_data *) option->data)->_data_parm.data;

    /*
     * Now calculate MD5 sum of the secret (step 4 in section 2.0
     * of RFC2385.
     */
    MD5_Update(&c, secret->data, secret->len);

    MD5_Final(dest, &c);
}

#if 0
static void
tcpopt_signature_dumper(pdu_option_t *p, const char *prefix)
{
    struct tcp_signature_option *option_data;

    option_data = p->data;

    printf("%s      Parameters:\n", prefix);
    printf("%s        Secret: \"%s\"\n", prefix, option_data->secret);
}
#endif

/*
 * RFC 1323 describes the TCP timestamp option.
 */
static void
tcpopt_timestamp(const GNode *option,
		 void *dest,
		 void *curr_pdu_hdr _U_,    /* Points to TCP header */
		 void *prev_pdu_hdr _U_)    /* Points to IP header */
{
    uint32_t tmp;
    void *field;

    *(uint8_t *) dest++ = TCP_OPT_TIMESTAMP;
    *(uint8_t *) dest++ = (field = _pb_pdata(option, "length") )
			  ? num_next(field) : TCPOLEN_TIMESTAMP;

    tmp = htonl(num_next(_pb_pdata(option, "tsval") ) );
    SIVAL(dest, 0, tmp);
    dest += sizeof(uint32_t);

    /* The SIVAL macro expands the val parameter so num_next() would
     * be called more than once. We need a temp. variable then. */
    tmp = htonl(num_next(_pb_pdata(option, "tsecr") ) );
    SIVAL(dest, 0, tmp);
}

#if 0
static void
tcpopt_timestamp_dumper(pdu_option_t *p, const char *prefix)
{
    struct tcp_timestamp_option *option_data;

    option_data = p->data;

    printf("%s      Parameters:\n", prefix);
    printf("%s        Option length: %s\n", prefix,
	   option_data->len ? num_info(option_data->len) : "automatic");
    printf("%s        TSval: %s\n", prefix, num_info(option_data->TSval) );
    printf("%s        TSecr: %s\n", prefix, num_info(option_data->TSecr) );
}
#endif

static void
tcpopt_raw(const GNode *option,
	   void *dest,
	   void *curr_pdu_hdr _U_,  /* Points to TCP header */
	   void *prev_pdu_hdr _U_)  /* Points to IP header */
{
    struct payload *raw_option;

    raw_option = ( (struct node_data *) option->data)->_data_parm.data;

    memcpy(dest, raw_option->data, raw_option->len);
}

static size_t
tcpopt_raw_len(const GNode *option)
{
    struct payload *payload;

    payload = ( (struct node_data *) option->data)->_data_parm.data;

    return payload->len;
}

#if 0
static void
tcpopt_raw_dumper(pdu_option_t *p, const char *prefix)
{
    struct payload *option_data;

    option_data = p->data;

    printf("%s      Parameters:\n", prefix);
    printf("%s        Length: %d\n", prefix, option_data->len);
}
#endif

static void
tcpopt_sackok(const GNode *option _U_, void *dest, void *curr_pdu_hdr _U_,
	      void *prev_pdu _U_)
{
    *(uint8_t *) dest++ = TCP_OPT_SACKOK; /* Option type */
    *(uint8_t *) dest = 2; /* Length */
}



/* PDU definition */
static pdu_t pdu_tcp = {
    .name = "tcp",
    .description = "TCP segment",
    .documented_in = "RFC 793",
    .len = sizeof(struct tcp_hdr),
    .fields = (field_t []) {
	{.name = "src", .type = PDU_FTYPE_NUMTYPE},
	{.name = "dst", .type = PDU_FTYPE_NUMTYPE},
	{.name = "seq", .type = PDU_FTYPE_NUMTYPE},
	{.name = "ack-seq", .type = PDU_FTYPE_NUMTYPE},
	{.name = "window", .type = PDU_FTYPE_NUMTYPE},
	{.name = "urg-ptr", .type = PDU_FTYPE_NUMTYPE},
	{.name = "cksum", .type = PDU_FTYPE_NUMTYPE},
	{.name = "flags", .type = PDU_FTYPE_NUMTYPE},
	{.name = "urg", .type = PDU_FTYPE_BIT},
	{.name = "ack", .type = PDU_FTYPE_BIT},
	{.name = "psh", .type = PDU_FTYPE_BIT},
	{.name = "rst", .type = PDU_FTYPE_BIT},
	{.name = "syn", .type = PDU_FTYPE_BIT},
	{.name = "fin", .type = PDU_FTYPE_BIT},
	{.name = "srcip", .type = PDU_FTYPE_IP},
	{.name = "dstip", .type = PDU_FTYPE_IP},
	{.name = "options", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
	    {.name = "nop", .type = PDU_FTYPE_BIT}, 
	    {.name = "eol", .type = PDU_FTYPE_BIT},
	    {.name = "raw", .type = PDU_FTYPE_DATA},
	    {.name = "mss", .type = PDU_FTYPE_NUMTYPE},
	    {.name = "wscale", .type = PDU_FTYPE_NUMTYPE},
	    {.name = "md5", .type = PDU_FTYPE_DATA},
	    {.name = "ts", .type = PDU_FTYPE_BRACED_ARGS, .data = (field_t []) {
		{.name = "tsval", .type = PDU_FTYPE_NUMTYPE}, 
		{.name = "tsecr", .type = PDU_FTYPE_NUMTYPE}, 
		{.name = "length", .type = PDU_FTYPE_NUMTYPE}, 
		{.name = NULL} }, .defval = NULL},
	    {.name = NULL} }, .defval = NULL},
	{.name = NULL}
    },
    .options_class = "tcp",
    .build = &build,
    .postbuild = &postbuild
};

static const pdu_option_t pdu_option_tcpeol = {
    .name = "eol",
    .description = "End of options list",
    .len = sizeof(uint8_t),
    .build = &tcpopt_eol
};

static const pdu_option_t pdu_option_tcpnop = {
    .name = "nop",
    .description = "No operation",
    .len = sizeof(uint8_t),
    .build = &tcpopt_nop
};

static const pdu_option_t pdu_option_tcpmaxseg = {
    .name = "mss",
    .description = "Maximum segment size (MSS)",
    .len = TCPOLEN_MAXSEG,
    .build = &tcpopt_maxseg
};

static const pdu_option_t pdu_option_tcpwindow = {
    .name = "wscale",
    .description = "Window scale",
    .len = TCPOLEN_WINDOW,
    .build = &tcpopt_window
};

static const pdu_option_t pdu_option_tcpsignature = {
    .name = "md5",
    .description = "TCP MD5 signature option",
    .len = TCPOLEN_MD5,
    .build = &tcpopt_signature,
};

static const pdu_option_t pdu_option_tcptimestamp = {
    .name = "ts",
    .description = "TCP timestamp option",
    .len = TCPOLEN_TIMESTAMP,
    .build = &tcpopt_timestamp
};

static const pdu_option_t pdu_option_tcpraw = {
    .name = "raw",
    .description = "Raw TCP option",
    .build = &tcpopt_raw,
    .get_len = &tcpopt_raw_len
};

static const pdu_option_t pdu_option_tcpsackok = {
    .name = "sack-ok",
    .description = "Selective ACK permitted",
    .len = sizeof(uint8_t),
    .build = &tcpopt_sackok
};


void
_pb_register_tcp(void)
{
    _pb_register_protocol(&pdu_tcp);

    _pb_register_protocol_option("tcp", &pdu_option_tcpeol);
    _pb_register_protocol_option("tcp", &pdu_option_tcpnop);
    _pb_register_protocol_option("tcp", &pdu_option_tcpmaxseg);
    _pb_register_protocol_option("tcp", &pdu_option_tcpwindow);
    _pb_register_protocol_option("tcp", &pdu_option_tcpsignature);
    _pb_register_protocol_option("tcp", &pdu_option_tcptimestamp);
    _pb_register_protocol_option("tcp", &pdu_option_tcpraw);
    _pb_register_protocol_option("tcp", &pdu_option_tcpsackok);
}
