import ldap
from arcnagios import ldapschema, ldaputils, ldapnagutils, glue2, nagutils
from arcnagios.utils import Statdict
from arcnagios.nagutils import OK, WARNING, CRITICAL, UNKNOWN, status_by_name

class RelationTrackerEntry(object):
    def __init__(self):
	self.seen_in = set([])	# List of DNs.
	self.referred_by = {}	# From GLUE2ForeignKey to list of DNs.

class RelationTracker(object):
    def __init__(self):
	self._d = {}

    def _ensure(self, a, k):
	if not (a, k) in self._d:
	    self._d[(a, k)] = RelationTrackerEntry()
	return self._d[(a, k)]

    def seen_pk(self, a, k):
	return (a, k) in self._d and self._d[(a, k)].seen_in != []

    def add_pk(self, a, k, dn):
	self._ensure(a, k).seen_in.add(dn)

    def add_ref(self, a, k, fk, dn):
	ent = self._ensure(a, k)
	if not fk in ent.referred_by:
	    ent.referred_by[fk] = []
	ent.referred_by[fk].append(dn)

    def iteritems(self):
	return self._d.iteritems()

class Check_arcglue2(ldapnagutils.LDAPNagiosPlugin):

    main_config_section = ['arcglue2', 'arcinfosys']

    def __init__(self):
	ldapnagutils.LDAPNagiosPlugin.__init__(self)

	ap = self.argparser.add_argument_group('GLUE2 Probe Options')
	ap.add_argument('--glue2-schema',
		default = '/etc/ldap/schema/GLUE20.schema',
		help = 'Path of GLUE 2.0 LDAP schema. '
		       'Default is /etc/ldap/schema/GLUE20.schema.')
	ap.add_argument('--if-dependent-schema',
		type = status_by_name, default = WARNING,
		metavar = 'NAGIOS-STATUS',
		help = 'Nagios status to report if LDAP schema had to be '
		       'fetched from the tested server.')
	ap.add_argument('--warn-if-missing',
		default = 'GLUE2AdminDomain,GLUE2ComputingService,'
			  'GLUE2ComputingEndpoint',
		metavar = 'OBJECTCLASSES',
		help = 'Report warning if there are no entries of the given '
		       'comma-separated list of LDAP objectclasses. '
		       'Default: GLUE2AdminDomain,GLUE2ComputingService,'
		       'GLUE2ComputingEndpoint')
	ap.add_argument('--critical-if-missing', default = '',
		metavar = 'OBJECTCLASSES',
		help = 'Report critical if there are no entries of the given '
		       'comma-separated list of LDAP objectclasses. '
		       'Empty by default.')
	ap.add_argument('--hierarchical-foreign-keys',
		metavar = 'FOREIGN-KEY-NAMES', default = '',
		help = 'A comma-separating list of foreign key attribute types '
		       'which should be reflected in the DIT.')
	ap.add_argument('--hierarchical-aggregates',
		default = False, action = 'store_true',
		help = 'Require that all foreign keys which represent '
		       'aggregation or composition are reflected in the '
		       'DIT.')

    def _dn_error(self, dn, msg):
	if getattr(self, '_last_dn', None) != dn:
	    self.log.error('Issues for dn "%s"'%dn)
	    self._last_dn = dn
	self.log.error('  - %s'%msg)

    def check(self):
	self.prepare_check()
	try:
	    self.subschema = \
		    ldapschema.parse(self.opts.glue2_schema, log = self.log,
				     builddir = self.tmpdir())
	except ldapschema.ParseError:
	    self.nagios_report.update_status(UNKNOWN,
		    'Could not parse GLUE2 schema.')
	    return
	except IOError:
	    self.nagios_report.update_status(self.opts.if_dependent_schema,
		    'Using schema from tested server.')
	    self.log.warning(
		    'The schema definition had to be fetched from the CE '
		    'itself, as %s is missing or inaccessible. '
		    'Use --glue2-schema to specify an alternative URL.'
		    % self.opts.glue2_schema)
	    self.subschema = self.fetch_subschema()

	# Stage 1.  Run though each entry and check what can be checked
	# without information about the full dataset.  Accumulate what we need
	# for the next phase.
	sr = self.search_s(self.opts.ldap_basedn or 'o=glue',
			   ldap.SCOPE_SUBTREE, '(objectClass=GLUE2Entity)',
			   attrlist = ['*', 'structuralObjectClass'])
	entry_count = 0
	entry_counts = {}
	if_missing = {}
	for class_name in self.opts.warn_if_missing.split(','):
	    if class_name:
		entry_counts[class_name] = 0
		if_missing[class_name] = WARNING
	for class_name in self.opts.critical_if_missing.split(','):
	    if class_name:
		entry_counts[class_name] = 0
		if_missing[class_name] = CRITICAL
	errors = Statdict()
	rt = RelationTracker()
	for dn, ent in sr:
	    entry_count += 1
	    try:
		o = glue2.construct_from_ldap_entry(self.subschema, dn, ent)
		if o is None:
		    self.log.warning(
			    '%s: Unknown structural object class, using '
			    'generic LDAP validation.'%dn)
		    ldaputils.LDAPObject(self.subschema, dn, ent)
		    continue
	    except (ValueError, ldaputils.LDAPValidationError), xc:
		errors['invalid entries'] += 1
		self.log.error('%s: %s'%(dn, xc))
		continue

	    # This is to check consistency of a link to a parent admin domain.
	    parent_domain_id = None
	    for dncomp in ldaputils.str2dn(dn)[1:]:
		if dncomp[0] == 'GLUE2DomainID':
		    parent_domain_id = dncomp[1]
		    break

	    # Check uniqueness of the primary key and record its dn.
	    if o.glue2_primary_key:
		pk = o.glue2_primary_key
		pkv = getattr(o, pk)
		if isinstance(pkv, list):
		    if len(pkv) > 1:
			self._dn_error(dn,
				'Multivalued primary key %s.'%pk)
			errors['multivalued PKs'] += 1
		if rt.seen_pk(pk, pkv):
		    self._dn_error(dn,
			    'Duplicate primary key %s=%s.'%(pk, pkv))
		    errors['duplicate PKs'] += 1
		else:
		    rt.add_pk(pk, pkv, dn)

	    # Checks and accumulations related to foreign keys.
	    for fk in o.glue2_all_foreign_keys():
		pk = fk.other_class.glue2_primary_key
		links = o.glue2_get_fk_links(fk)

		# Check target multiplicity and record the link.
		if not glue2.matching_multiplicity(fk.other_mult,
						   len(links)):
		    self._dn_error(dn,
			    '%s contains %d links, but the multiplicity '
			    'of the target of this relation is %s.'
			    %(fk.name, len(links),
			      glue2.multiplicity_indicator(fk.other_mult)))
		    errors['multiplicity violations'] += 1
		for link in links:
		    rt.add_ref(pk, link, fk, dn)

		# If the entry appears under an admin domain and links to an
		# admin domain, they should be the same.  We don't distinguish
		# user and admin domains here, but user domains should not be
		# the parent of services, anyway.
		if parent_domain_id and \
			fk.other_class == glue2.GLUE2AdminDomain:
		    for link in links: # 0 or 1
			if link != parent_domain_id:
			    self._dn_error(dn,
				    'This entry links to an admin domain other '
				    'than the domain under which it appears.')

	    # Count entries of each objectclass.
	    class_name = o.glue2_class_name()
	    if class_name in entry_counts:
		entry_counts[class_name] += 1

	# Stage 2.  Run though each foreign key link and perform related
	# checks.
	hierarchical_foreign_keys = \
		set(self.opts.hierarchical_foreign_keys.split(','))
	for ((pk, v), rte) in rt.iteritems():
	    for fk, refs in rte.referred_by.iteritems():

		# Multiplicity Checks.
		if not rte.seen_in:
		    self.log.error(
			    'No match for primary key %s=%s referred by %s'
			    %(pk, v, fk.name))
		    errors['missing primary keys'] += 1
		if not glue2.matching_multiplicity(fk.local_mult, len(refs)):
		    self.log.error(
			    '%d entries refer to %s=%s through %s but the '
			    'source multiplicity of this relation is %s.'
			    %(len(refs), pk, v, fk.name,
			      glue2.multiplicity_indicator(fk.local_mult)))
		    errors['multiplicity violations'] += 1

		# DIT Checks.
		if len(rte.seen_in) != 1:
		    continue
		dn = list(rte.seen_in)[0]
		if fk.name == 'GLUE2ExtensionEntityForeignKey':
		    for ref in refs:
			if not ldaputils.is_immediate_subdn(ref, dn):
			    self.log.error(
				    'The extension entry %s should be '
				    'immediately below %s.'%(ref, dn))
			    errors['DIT issues'] += 1
		elif fk.other_class == glue2.GLUE2Service and \
			fk.relation >= glue2.AGGREGATION:
		    for ref in refs:
			if not ldaputils.is_proper_subdn(ref, dn):
			    self.log.error(
				    'The component %s belongs below its '
				    'service %s.'%(ref, dn))
			    errors['DIT issues'] += 1
		elif fk.name in hierarchical_foreign_keys or \
			self.opts.hierarchical_aggregates \
			    and fk.relation >= glue2.AGGREGATION:
		    for ref in refs:
			if not ldaputils.is_proper_subdn(ref, dn):
			    self.log.error('"%s" belongs below "%s"'%(ref, dn))
			    errors['DIT issues'] += 1

	# Report.
	for class_name, count in entry_counts.iteritems():
	    if count == 0:
		errors['absent expected object classes'] += 1
		self.log.error('There are no entries with objectclass %s.'
			       % class_name)
	if errors:
	    self.nagios_report.update_status(CRITICAL, 'Found %s.'
		    % ', '.join('%d %s'%(c, s) for s, c in errors.iteritems()))
	self.nagios_report.update_status(OK,
		'Validated %d entries.'%entry_count)
