"""Expanding substitutions in the configuration file."""

import os
import random
import re
import time
import ldap3
from arcnagios.nagutils import ServiceUnknown

def get_interp_opt(config, section, var, reputation_tracker, target_env):
    if config.has_option(section, var):
        import_interpolated_variables(
                config, section, var, reputation_tracker, target_env)
        return config.get(section, var, vars = target_env)
    else:
        return None

def get_interp(config, section, var, reputation_tracker, target_env,
               default = None):
    y = get_interp_opt(config, section, var, reputation_tracker, target_env) \
            or default
    if y is None:
        raise ServiceUnknown('Configuration error: Missing %s in [%s].'
                             % (var, section))
    return y

def _subst_option(config, section, var, reputation_tracker, target_env):
    default = get_interp_opt(
            config, section, 'default', reputation_tracker, target_env)
    if default is None:
        raise ServiceUnknown('Missing required option (-O %s=...).'%var)
    target_env[var] = default

def _subst_getenv(config, section, var, reputation_tracker, target_env):
    default = get_interp_opt(
            config, section, 'default', reputation_tracker, target_env)
    if config.has_option(section, 'envvar'):
        envvar = get_interp_opt(
                config, section, 'envvar', reputation_tracker, target_env)
    else:
        prefix = get_interp(
                config, section, 'prefix', reputation_tracker, target_env, '')
        envvar = prefix + var
    v = os.getenv(envvar, default)
    if v is None:
        raise ServiceUnknown('Missing required option (-O) or '
                             'enviroment variable %s.' % envvar)
    target_env[var] = v

def _subst_ldap(config, section, var, reputation_tracker, target_env):
    basedn = get_interp(
            config, section, 'basedn', reputation_tracker, target_env)
    filterstr = get_interp(
            config, section, 'filter', reputation_tracker, target_env)
    attribute = get_interp(
            config, section, 'attribute', reputation_tracker, target_env)
    attrlist = list(map(str.strip, attribute.split(',')))
    errors = []
    entries = []
    for uri in get_interp(config, section, 'uri',
                          reputation_tracker, target_env).split():
        try:
            ldap_server = ldap3.Server(uri)
            ldap_conn = ldap3.Connection(ldap_server, auto_bind = True)
            if ldap_conn.search(basedn, filterstr, attributes = attrlist):
                entries = ldap_conn.entries
            break
        except ldap3.core.exceptions.LDAPCommunicationError as exn:
            errors.append('%s gives %s' % (uri, exn))
        except ldap3.core.exceptions.LDAPExceptionError as exn:
            raise ServiceUnknown('LDAP query to %s from %s failed with: %s'
                    % (uri, section, exn))
    else:
        raise ServiceUnknown(
                'Communication failed for all LDAP servers from %s: %s'
                % (section, ', '.join(errors)))
    for ent in entries:
        for attr in attrlist:
            try:
                target_env[var] = ent[attr].value
                return
            except ldap3.core.exceptions.LDAPExceptionError:
                 # ldap3.core.exceptions.LDAPKeyError for older versions
                 # ldap3.core.exceptions.LDAPCursorError for newer versions
                pass
    if config.has_option(section, 'default'):
        target_env[var] = get_interp(
                config, section, 'default', reputation_tracker, target_env)
    else:
        raise ServiceUnknown('LDAP query %s did not provide a value for %s.'
                             % (filterstr, section))

def _subst_pipe(config, section, var, reputation_tracker, target_env):
    cmd = get_interp(config, section, 'command', reputation_tracker, target_env)
    fh = os.popen(cmd)
    target_env[var] = fh.read().strip()
    fh.close()

def _subst_random_line(config, section, var, reputation_tracker, target_env):
    path = get_interp(
            config, section, 'input_file', reputation_tracker, target_env)

    rng = random.Random(time.time())
    include = None
    exclude = set()
    if config.has_option(section, 'exclude'):
        exclude = set(get_interp(
            config, section, 'exclude', reputation_tracker, target_env).split())
    if config.has_option(section, 'include'):
        include = set(get_interp(
            config, section, 'include', reputation_tracker, target_env).split())
        include.difference_update(exclude)
    if config.has_option(section, 'reputation_dist'):
        dist_name = config.get(section, 'reputation_dist', vars=target_env)
        with open(path) as fh:
            lines = set(line for line in map(str.strip, fh)
                             if line != '' and line[0] != '#')
        if not include is None:
            lines.intersection_update(include)
        else:
            lines.difference_update(exclude)
        if not lines:
            raise ServiceUnknown('%s must contain at least one non-excluded '
                                 'line' % path)
        target_env[var] = reputation_tracker.choose(dist_name, lines)
    else:
        try:
            fh = open(path)
        except IOError as exn:
            raise ServiceUnknown(str(exn))
        n = 0
        chosen_line = None
        while True:
            try:
                line = fh.next().strip()
                if not line or line.startswith('#') \
                        or not include is None and not line in include \
                        or include is None and line in exclude:
                    continue
                if rng.randint(0, n) == 0:
                    chosen_line = line
            except StopIteration:
                break
            n += 1
        fh.close()
        if chosen_line is None:
            raise ServiceUnknown('%s must contain at least one non-excluded '
                                 'line' % path)
        target_env[var] = chosen_line

def _subst_strftime(config, section, var, reputation_tracker, target_env):
    if config.has_option(section, 'raw_format'):
        fmt = config.get(section, 'raw_format', vars = target_env, raw=True)
    else:
        fmt = get_interp(
                config, section, 'format', reputation_tracker, target_env)
    target_env[var] = time.strftime(fmt)

def _subst_switch(config, section, var, reputation_tracker, target_env):
    case = 'case[%s]' % get_interp(
            config, section, 'index', reputation_tracker, target_env)
    if config.has_option(section, case):
        import_interpolated_variables(
                config, section, case, reputation_tracker, target_env)
        target_env[var] = get_interp(
                config, section, case, reputation_tracker, target_env)
    else:
        if not config.has_option(section, 'default'):
            raise ServiceUnknown(
                    'No %s and no default in section variable.%s.'%(case, var))
        import_interpolated_variables(
                config, section, 'default', reputation_tracker, target_env)
        target_env[var] = get_interp(
                config, section, 'default', reputation_tracker, target_env)

_method_by_name = {
    'getenv': _subst_getenv,
    'ldap': _subst_ldap,
    'option': _subst_option,
    'pipe': _subst_pipe,
    'random_line': _subst_random_line,
    'strftime': _subst_strftime,
    'switch': _subst_switch,
}

def register_substitution_method(name, f):
    _method_by_name[name] = f

_interp_re = re.compile(r'%\(([a-zA-Z0-9_]+)\)')

def import_interpolated_variables(
        config, section, var, reputation_tracker, target_env):
    """Import variables needed for expanding ``var`` in ``section``."""

    raw_value = config.get(section, var, raw = True)
    for mo in re.finditer(_interp_re, raw_value):
        v = mo.group(1)
        if not v in target_env:
            import_variable(config, v, reputation_tracker, target_env)

def import_variable(config, var, reputation_tracker, target_env):
    """Import ``var`` by executing its defining section, populating
    ``target_env`` with its value and the values of any dependent
    variables."""

    section = 'variable.' + var
    method = config.get(section, 'method')
    try:
        return _method_by_name[method]\
                (config, section, var, reputation_tracker, target_env)
    except KeyError:
        raise ServiceUnknown('Unknown substitution method %s.'%method)
