/*
 ** Copyright 2004-2007 - INL
 ** Written by Eric Leblond <eric.leblond@inl.fr>
 **            Vincent Deffontaines <vincent@inl.fr>
 ** INL http://www.inl.fr/
 **
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation, version 2 of the License.
 **
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "thread_session.h"
#include "thread_session.moc"
#include <QtGui>

extern nuclient_error_t *err; /* defined in main.cpp */

ThreadCheckConn::ThreadCheckConn(QObject* _parent) : QThread(_parent)
{
	session = NULL;
	username = "";
	password = "";

	connect(this, SIGNAL(tray_set_connected()), parent(), SLOT(tray_set_connected()));
	connect(this, SIGNAL(tray_set_trying()), parent(), SLOT(tray_set_trying()));
	connect(this, SIGNAL(tray_set_stopped()), parent(), SLOT(tray_set_stopped()));
	connect(this, SIGNAL(tray_report_error()), parent(), SLOT(tray_report_error()));
	connect(this, SIGNAL(tray_report_connected()), parent(), SLOT(tray_report_connected()));
	connect(this, SIGNAL(tray_report_disconnected()), parent(), SLOT(tray_report_disconnected()));
}

void ThreadCheckConn::askStop()
{
	loop_mutex.lock();
	loop_continue = false;
	loop_mutex.unlock();
}

void ThreadCheckConn::setUserInfos(QString _username, QString _password)
{
	username = _username;
	password = _password;

	// We don't know if those username/passwords are valid
	// so if the authentication fails, there is no need to
	// trying reauthenticating to Nuauth in a loop
	loop_continue = false;
	report_errors = true;;
}

void ThreadCheckConn::run()
{
	startAuth();
	while(1)
	{
		loop_mutex.lock();
		if(!loop_continue)
		{
			loop_mutex.unlock();
			break;
		}
		loop_mutex.unlock();

		if(session)
		{
			// A session is opened, so check we are still connected to Nuauth
			if(nu_client_check(session, err) == -1)
			{
				nu_client_delete(session);
				session = NULL;
				if(report_errors)
				{
					emit tray_set_trying();
					emit tray_report_disconnected();
					report_errors = false;
				}
			}
		}
		else
		{
			// We have been disconnected
			// Restart the authentication using the last username and password used
			startAuth();
		}
	}
	stopAuth();
}

void ThreadCheckConn::startAuth()
{
	// Stop automatic reauthentication after the connection has been lost
	session = nu_client_new(username.toStdString().c_str(),
				password.toStdString().c_str(),
				1,
				err);

	nu_client_set_client_info(session, "nuapplet", "3.0");

	QSettings settings;
	settings.sync();

	// Configure certificate related parameters
	int cert_verify_remote = settings.value("cert_verify_remote").toBool();
	int cert_use_client_cert = settings.value("cert_use_client_cert").toBool();
	QString str_ca = settings.value("ca").toString();
	QString str_key = settings.value("key").toString();
	QString str_cert = settings.value("cert").toString();
	QString str_crl = settings.value("crl").toString();
	int cert_check_fqdn = settings.value("cert_check_fqdn").toBool();

	if (cert_verify_remote) {
		if (!nu_client_set_ca(session, (const char*)str_ca.toAscii(), err)) {
			goto label_exit_connection;
		}

		if (cert_use_client_cert) {
			if (str_key.length() > 0 &&
					!nu_client_set_key(session, (const char*)str_key.toAscii(), (const char*)str_cert.toAscii(), err)) {
				goto label_exit_connection;
			}
		}

		if (str_crl.length()>0 && !nu_client_set_crlfile(session, (const char*)str_crl.toAscii(), err)) {
			goto label_exit_connection;
		}

		if (!cert_check_fqdn)
			nu_client_set_fqdn_suppress_verif(session, 1);
	} else { // we don't check server certificate
		nu_client_set_cert_suppress_verif(session, 1);
	}

	// Connect to NuAuth
	char* chr_hostname;
	char* chr_port;

#if defined(HAVE_DEFAULT_NUAUTH_IP) && !defined(HAVE_DEFAULT_HOSTNAME_FUNC)
	if(strcmp(settings.value("hostname").toString().toStdString().c_str(), "")==0 && session->default_hostname)
	{
		printf("setting hostname\n");
		settings.setValue("hostname", session->default_hostname);
	}
	if(strcmp(settings.value("port").toString().toStdString().c_str(), "")==0 && session->default_port)
		settings.setValue("port", session->default_port);
	settings.sync();
#endif // #if defined(HAVE_DEFAULT_NUAUTH_IP) && !defined(HAVE_DEFAULT_HOSTNAME_FUNC)

	chr_hostname = strdup(settings.value("hostname").toString().toStdString().c_str());
	chr_port = strdup(settings.value("port").toString().toStdString().c_str());

	if(!nu_client_connect(session, chr_hostname, chr_port, err))
	{
fprintf(stderr,"%s\n", nu_client_strerror(session, err));

		if(report_errors)
			emit tray_report_error();
		free(chr_hostname);
		free(chr_port);

		if(session)
			nu_client_delete(session);
		session = NULL;
		return;
	}
	
	free(chr_hostname);
	free(chr_port);

	emit tray_set_connected();
	emit tray_report_connected();
	loop_continue = true;
	report_errors = true;

	return;

label_exit_connection:
	printf("Initialization error: %s\n", nu_client_strerror(session, err));
	// If the timer is active, this connection is done automatically
	// so we don't display the error message
	if(report_errors)
		emit tray_report_error();
	if(session)
		nu_client_delete(session);
	session = NULL;
	return;
}

void ThreadCheckConn::stopAuth()
{
	if(session)
		nu_client_delete(session);
	session = NULL;
	emit tray_set_stopped();
}

const char* ThreadCheckConn::GetNuclientError()
{
#if (defined(NUCLIENT_VERSION_MAJOR) && NUCLIENT_VERSION_MAJOR>=4)
	return nu_client_strerror(session, err);
#else
	return nu_client_strerror(err);
#endif
}
