/* Pipeline_Configuration

PIRL CVS ID: Pipeline_Configuration.java,v 1.6 2012/04/16 07:01:53 castalia Exp

Copyright (C) 2011-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Conductor.Pipeline_Configuration;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Vector;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;

import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;
import PIRL.Database.Database;
import PIRL.Database.Database_Exception;


/** A <i>Pipeline_Configuration provides an API and command-line tool for
	reading and writing pipeline configurations stored a database.
<p>
	A Pipeline_Configuration database table has the following fields:
<dl>
<dt>ID
	<dd>A unique auto-incremented integer.
<dt>PIPELINE
	<dd>A pipeline name.
<dt>PARAMETER
	<dd>A pipeline configuration parameter. This corresponds to the
		command-line "pipeline" specification.
<dt>VALUE
	<dd>The value of the configuration parameter.
<dt>VALIDATION_STRING
	<dd>A regular expression used to validate the contents of the value.
<dt>DESCRIPTION
	<dd>A short description of the parameter
<dt>LAST_UPDATE
	<dd>The time when the record was last modified.
</dl>
<p>
<h3>API
</h3>
<p>
	Pipeline_Configuration provides the ability to read and write all the
	columns except the ID. Typical usage of the API involves constructing
	a Pipeline_Configuration object with a database connection, catalog,
	pipeline, and whether strict-mode should be enforced. For example:
<p>
<pre>
	Configuration
		configuration = new Configuration ();
	Database
		database = new Database (configuration);
	database.Connect ();
	String
		catalog = "HiRISE",
		pipeline = "HiColorNorm";
	boolean
		strict = true;
	Pipeline_Configuration
		pipeline_configuration = new Pipeline_Configuration
			(database.Connection (), catalog, pipeline, strict);
</pre>
<p>
Where:
<dl>
<dt>configuration
	<dd>A PIRL.Configuration object that provides Database access
		information. A Database configuration file may be used for this
		purpose.
<dt>database
	<dd>A PIRL.Database object.
<dt>catalog
	<dd>The name of the database catalog that contains the table. By
		default this is obtained from the Configuration "Catalog"
		parameter.
<dt>pipeline
	<dd>The name of the pipeline to use when reading or writing a
		database table record.
<dt>strict
	<dd>When true all read operations will have their VALUE validated
		against the VALIDATION_STRING
</dl>
<p>
	The pipeline_configuration can then be used to read and write
	table records. For example:
<p>
<pre>
	Conf
		conf = pipeline_configuration.read ("Make_Unfiltered_Cube");
	System.out.println ("Make_Unfiltered_Cube = " + conf.value);
	System.out.println ("Description: " + conf.description);
</pre>
<p>
	The {@link Conf} class is used to contain the field values of a table
	record.
<p>
	A Pipeline_Configuration.write updates a table record if a named
	pipeline parameter is present in a table record. A new record is
	inserted in the table otherwise. Table record updates will only set
	field values for which data has supplied. For example:
<p>
<pre>
	pipeline_configuration.write
		("Make_Unfiltered_Cube", "TRUE");
	pipeline_configuration.write
		("Make_Unfiltered_Cube", "TRUE", "TRUE|FALSE");
	pipeline_configuration.write
		("Make_Unfiltered_Cube", "TRUE", Validator.BOOLEAN);
	pipeline_configuration.write
		("Make_Unfiltered_Cube", "TRUE", "TRUE|FALSE", "A description.");
</pre>
<p>
	Each write will modify the same record (except, perhaps, the first
	which might insert a new record) that has the "HiColorNorm" value in
	the PIPELINE field and the "Make_Unfiltered_Cube" value in the
	PARAMETER field of a record of the default  "Pipeline_Configuration"
	table in the "HiRISE" catalog.
<p>
<h3>Application
</h3>
<p>
	The command-line usage can be obtained by using the -help option:
<p>
	Here is an example of using Pipeline_Configuration for querying a
	pipeline's parameter for a value, getting all parameter
	information, and writing parameter information for a pipeline.
<p>
<h4>Querying a pipeline parameter value:
</h4>
<p>
<pre>
Pipeline_Configuration \ 
	-configuration ~/HiRISE/Configuration/EDRgen/EDRgen.conf \ 
	-pipeline HiColorNorm \ 
	-parameter Make_Unfiltered_Cube
</pre> 
<p>
	Returns:
<p>
<pre>
TRUE
</pre>
<p>
<h4>Getting all information for a parameter:
</h4>
<p>
<pre>
Pipeline_Configuration \ 
	-configuration ~/HiRISE/Configuration/EDRgen/EDRgen.conf \ 
	-pipeline HiColorNorm \ 
	-parameter Make_Unfiltered_Cube \ 
	-info
</pre>
<p>
	Returns:
<p>
<pre>
PIPELINE = HiColorNorm
PARAMETER = Make_Unfiltered_Cube
VALUE = TRUE
VALIDATION_STRING = TRUE|FALSE
DESCRIPTION = A boolean value to indicate if the HiColorNorm pipeline
			  should create the unfiltered color cube that has not passed
			  through the noise or furrow filtering corrections
LAST_UPDATE = 2011-05-17 13:51:59.0
</pre>
<p>
<h4>Writing parameter information for pipeline:
<h4>
<p>
The result of this execution will be an exit code of 0
and no output
<pre>
Pipeline_Configuration \ 
	-configuration ~/HiRISE/Configuration/EDRgen/EDRgen.conf \ 
      -pipeline HiColorNorm \ 
	  -parameter Make_Unfiltered_Cube \ 
	  -write \ 
      -value FALSE \ 
	  -validation "TRUE|FALSE|true|false"
</pre>
<p>
Note that for write operations the -write option must be specified.
<p>
The result will be no output; the application will produce an exit status
value of zero if there were no errors.
 
 @author	Chris Van Horne - UA/HiROC
 @author	Rodney Heyd - UA/HiROC
 @version	1.6
*/
public class Pipeline_Configuration
{
/*	Programmer notes:

	TODO(cwvh): The Pipeline_Configuration class currently has
	System.exit calls within its methods. This can be quite a headache
	for client code of this API since execution may abruptly terminate in
	the library. This should be fixed some day.

	TODO(rsh): Pipeline_Configuration needs to be tested against
	PostgreSQL. No testing of this utility against PostgreSQL has been
	done yet.
*/
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Pipeline_Configuration.Pipeline_Configuration (1.6 2012/04/16 07:01:53)";

/**	The default table name.
*/
public static final String
	DEFAULT_DATABASE_TABLE_NAME	= "Pipeline_Configuration";

/**	The default configuration filename.
*/
private static final String
	DEFAULT_CONFIGURATION		= "Database.conf";

/**	A <i>Validator</i>contains a list of commonly used validators for
	typical values.

	These validators may be used for convenience when doing a
	<code>write</code>. Java Strings used as regexps must still follow
	proper escaping using double-backslashes.

	Example:

	hiColorNormPipeConf.write
		("Make_Unfiltered_Cube", "TRUE", Validator.BOOLEAN);
*/
public enum Validator
{
FLOATING_POINT ("[-+]?[0-9]*\\.?[0-9]+([eE][-+]?[0-9]+)?"),
BOOLEAN ("TRUE|FALSE");

private final String regexp;

private Validator
	(
	String	regexp
	)
{this.regexp = regexp;}

public String regexp ()
{return regexp;}
}

/**	Application exit status values.
*/
public enum ExitCode
{
SUCCESS (0),
COMMAND_LINE_PARSE_ERROR (1),
CONFIGURATION_ERROR (2),
DATABASE_ERROR (3),
STRICT_VALIDATION_FAILURE (4);

private final int
	code;

private ExitCode
	(
	int	code
	)
{this.code = code;}

public int code ()
{return code;}
}

/**	The fields of a record in the Pipeline_Configuration database table.

	When queries are made via read and write methods, a <code>Conf</code>
	is used as a data transfer object.
*/
public static final class Conf
{
public final String
	pipeline,
    parameter,
	value,
	validation,
	description,
	lastUpdate;

public Conf ()
{this (null, null, null, null, null, null);}

public Conf
	(
	String	pipeline,
	String	parameter,
	String	value,
	String	validation,
	String	description,
	String	lastUpdate
	)
{
this.pipeline		= pipeline;
this.parameter		= parameter;
this.value			= value;
this.validation		= validation;
this.description	= description;
this.lastUpdate		= lastUpdate;
}
}

private final Database
	database;
private final Connection
	conn;
private final String
	catalog;
private final String
	table;
private final String
	pipeline;
private final boolean
	strict;

private static String
	NL					= Configuration.NL;


/**	Constructs a Pipeline_Configuration with non-strict semantics.

	@param	database The Database object providing access to the
		database contents.
	@param	catalog	The name of catalog in the database where the database
		tables are located.
	@param	table	The name of the table to use for pipeline
		configuration queries.
	@param	pipeline	The name of the pipeline to query for parameters.
 */
public Pipeline_Configuration
	(
	Database	database,
	String		catalog,
	String		table,
	String		pipeline
	)
{this (database, catalog, table, pipeline, false);}

/**	Constructs a Pipeline_Configuration.

	@param	database The Database object providing access to the
		database contents.
	@param	catalog	The name of catalog in the database where the database
		tables are located.
	@param	table	The name of the table to use for pipeline
		configuration queries.
	@param	pipeline	The name of the pipeline to query for parameters.
	@param strict	A flag that indicates whether to treat validation failures
		as global errors.
*/
public Pipeline_Configuration
	(
	Database	database,
	String		catalog,
	String		table,
	String		pipeline,
	boolean		strict
	)
{
this.database	= database;
this.conn		= database.Connection ();
this.catalog	= catalog;
this.table		= table;
this.pipeline	= pipeline;
this.strict		= strict;
}

/**	Read the most up-to-date row for a parameter.

	@param	parameter	The name of a parameter in the table for which
		to read the record.
	@return A {@link Conf} containing all the field values of the record.
*/
public Conf read
	(
	String	parameter
	)
{
String
	table = catalog + "." + this.table,
	fields = "value, validation_string, description, last_update",
	sql = "SELECT %s FROM %s WHERE pipeline = "
		+ "? AND parameter = ? ORDER BY last_update DESC",
	query = String.format (sql, fields, table);
try
	{
    PreparedStatement
		stmt = conn.prepareStatement (query);
    stmt.setString (1, pipeline);
    stmt.setString (2, parameter);
    ResultSet
		results = stmt.executeQuery();

    if (results.next ())
		{
        String
			value = results.getString ("value"),
        	validation = results.getString ("validation_string"),
        	/*
				The (?i:X) format allows case-insensitive searching
				in String.match.
			*/
        	regexp = String.format ("(?i:%s)", validation);
        /*
			If strict mode is enabled, values must pass validation
       		otherwise the program is exited with a failed exit code.
		*/
        if (strict && ! value.matches (regexp))
			{
            String
				msg =
					"error: parameter '%s' did not pass value validation!\n";
            System.err.printf (msg, parameter);
            System.exit (ExitCode.STRICT_VALIDATION_FAILURE.code ());
        	}

        String
			descr = results.getString ("description"),
        	timestamp = results.getString ("last_update");
        return new Conf
			(pipeline, parameter, value, validation, descr, timestamp);
    	}
	}
catch (SQLException exception)
	{
    String
		msg = "error: prepared statement for parameter '%s' failed with '%s'!\n";
    System.err.printf (msg, parameter, exception);
    System.exit (ExitCode.DATABASE_ERROR.code());
	}
return new Conf ();
}

/**	Write a simple parameter-value pair into a table record.

	@param	parameter	The parameter name.
	@param	value	The value String for the parameter.
*/
public void write
	(
	String	parameter,
	String	value
	)
{write (parameter, value, null);}

/**	Write a parameter-value pair with validation string into a table record.

	@param	parameter	The parameter name.
	@param	value	The value String for the parameter.
	@param	validation	The validation String for the value.
*/
public void write
	(
	String	parameter,
	String	value,
	String	validation
	)
{write (parameter, value, validation, null);}

/**	Write a parameter-value pair with validation string and description
	into a table record.

	@param	parameter	The parameter name.
	@param	value	The value String for the parameter.
	@param	validation	The validation String for the value.
	@param	description	The description String for the parameter.
*/
public void write
	(
	String	parameter,
	String	value,
	String	validation,
	String	description
	)
{
Conf
	conf = new Conf
		(pipeline, parameter, value, validation, description, null);
write (conf);
}

/**	Write a configuration (Conf) for a parameter into a table record.

	The string-only variants of the <code>write</code> methods are
	preferable; they use this method.

	@param	conf	A {@link Conf} object which contains the field
		values of the record to be inserted into the table
*/
public void write
	(
	Conf	conf
	)
{
String
	table = catalog + "." + this.table,
	fields = "pipeline, parameter, value, validation_string, description",
	query = String.format
		("SELECT %s FROM %s WHERE pipeline = "
		+ "? AND parameter = ? ORDER BY last_update ASC",
		fields, table),
	insert = String.format
		("INSERT INTO %s (%s) VALUES (?, ?, ?, ?, ?)",
		table, fields),
	update = String.format
		("UPDATE %s SET value = ?, validation_string = "
		+ "?, description = ? WHERE pipeline = ? AND parameter = ?",
		table);
try
	{
    PreparedStatement
		stmt;
    ResultSet
		results;

    stmt = conn.prepareStatement (query);
    stmt.setString (1, conf.pipeline);
    stmt.setString (2, conf.parameter);
    results = stmt.executeQuery ();

    if (results.next ())
		{
		/*	The SELECT returned rows; do an UPDATE.
			Get the values in the row.
		*/
        String
			value = conf.value;
        if (value == null)
            value = results.getString ("value");

        String
			validation = conf.validation;
        if (validation == null)
            validation = results.getString ("validation_string");

        String
			description = conf.description;
        if (description == null)
            description = results.getString ("description");

        stmt = conn.prepareStatement (update);
        stmt.setString (1, value);
        stmt.setString (2, validation);
        stmt.setString (3, description);
        stmt.setString (4, conf.pipeline);
        stmt.setString (5, conf.parameter);
    	}
	else
		{
        /*	The query didn't return a row; do an INSERT.
        	Convert null conf values to empty Strings.
		*/
        String
			value = conf.value;
        if (value == null)
            value = "";

        String
			validation = conf.validation;
        if (validation == null)
            validation = "";

        String
			description = conf.description;
        if (description == null)
            description = "";

        stmt = conn.prepareStatement (insert);
        stmt.setString (1, conf.pipeline);
        stmt.setString (2, conf.parameter);
        stmt.setString (3, value);
        stmt.setString (4, validation);
        stmt.setString (5, description);
		}
	stmt.execute ();
	}
catch (SQLException exception)
	{
    String
		msg = "error: prepared statement for write failed with '%s'\n";
    System.err.printf (msg, exception);
    System.exit (ExitCode.DATABASE_ERROR.code());
	}
}

/**	Close the database connection.

	Closing the database connection is usually not necessary.
	Use this method to ensure that the database is closed when needed.

	<b>WARNING</b> Do not attempt to read or write a database table
	record after the connection to the database has been closed.
*/
public void close ()
{
try {conn.close ();}
catch (SQLException exception)
	{exception.printStackTrace ();}
}

/** Build a configuration table given a database catalog and table name.

	A database table for holding configuration parameters is created
	in the specified catalog.

	@param catalog	The name of the database catalog to contain the table.
	@param	table	The name of the database table for containing
		configuration parameters.
	@throws	Database_Exception if the database table could not be created.
 */
public void buildConfigurationTable
	(
	String	catalog,
	String	table
	)
	throws Database_Exception
{
Vector<String>
	field_names = new Vector<String> ();
field_names.add ("ID");
field_names.add ("PIPELINE");
field_names.add ("PARAMETER");
field_names.add ("VALUE");
field_names.add ("VALIDATION_STRING");
field_names.add ("DESCRIPTION");
field_names.add ("LAST_UPDATE");

Vector<String>
	field_types = new Vector<String> ();
field_types.add ("INT(4)");
field_types.add ("VARCHAR(75)");
field_types.add ("VARCHAR(75)");
field_types.add ("TEXT");
field_types.add ("TEXT");
field_types.add ("TEXT");
field_types.add ("TIMESTAMP");

try {database.Create (catalog + "." + table, field_names, field_types);}
catch (Database_Exception exception)
	{
	throw new Database_Exception (ID + NL +
		"Could not create the \"" + table
			+ "\" in the \"" + catalog + "\" catalog." + NL
		+ exception.getMessage ());
	}
}

/**	Application main.

	@param args	The program command-line arguments.
*/
public static void main
	(
	String[]	args
	)
{
try
	{
    CommandLineParser
		parser = new GnuParser ();
    Options
		options = buildCommandLine ();
    CommandLine
		line = parser.parse (options, args);
    String
		config = null;

    if (args.length == 0 ||
		line.hasOption ("help"))
		{
        usage (options);
        System.exit (ExitCode.SUCCESS.code ());
		}

    /*	Get the configuration while respecting $HiRISE_ROOT.
		TODO(cwvh): Might want to allow overriding $HiRISE_ROOT
		from the command-line; e.g.
		String root = System.getenv ("HiRISE_ROOT");
	*/
    if (line.hasOption ("configuration"))
        config = line.getOptionValue ("configuration");
    else
        config += DEFAULT_CONFIGURATION;

    //	Get the catalog from the configuration.
    Configuration
		configuration = new Configuration (config);
    String
		catalog = configuration.Get_One ("CATALOG");
    if (line.hasOption ("catalog"))
		{
        catalog = line.getOptionValue ("catalog");
        configuration.Set ("CATALOG", catalog);
    	}

    String
		table = DEFAULT_DATABASE_TABLE_NAME;
    if (line.hasOption ("table"))
        table = line.getOptionValue ("table");

    if (line.hasOption ("buildtable"))
    	{
        Database
			db = new Database (configuration);
        db.Connect ();
        Pipeline_Configuration
			pipeline_configuration = new Pipeline_Configuration
				(db, catalog, table, "all", true);
        pipeline_configuration.buildConfigurationTable (catalog, table);
        System.exit (ExitCode.SUCCESS.code ());
		}

    //	Get the pipeline for the query.
    if (! line.hasOption ("pipeline"))
		{
        System.err.println ("error: need a 'pipeline' for this query");
        System.exit (ExitCode.COMMAND_LINE_PARSE_ERROR.code ());
    	}
    String
		pipeline = line.getOptionValue ("pipeline");

    //	Get the parameter for this pipeline.
    if (! line.hasOption ("parameter"))
		{
        usage (options);
        System.err.println ("error: need a 'parameter' argument to query");
        System.exit (ExitCode.COMMAND_LINE_PARSE_ERROR.code ());
    	}
    String
		parameter = line.getOptionValue ("parameter");

    //	Enable strict-mode?
    boolean
		strict = line.hasOption ("strict");

    //	Construct the database and pipeline configuration.
    Database
		database = new Database (configuration);
    database.Connect ();
    Pipeline_Configuration
		pipeline_configuration = new Pipeline_Configuration
			(database, catalog, table, pipeline, strict);

    if (line.hasOption ("write"))
		{
		/*	Write to the table.
        	Get the value to write.
		*/
        if (! line.hasOption ("value"))
			{
            System.err.println
				("error: need a 'value' to write in write-mode");
            System.exit (ExitCode.COMMAND_LINE_PARSE_ERROR.code());
        	}
        String
			value = line.getOptionValue ("value");

        //	Optional validation string.
        String
			validation = null;
        if (line.hasOption ("validation"))
            validation = line.getOptionValue("validation");

        //	Optional description.
        String
			description = null;
        if (line.hasOption ("description"))
            description = line.getOptionValue ("description");

        pipeline_configuration.write (parameter, value, validation, description);
		}
	else
		{
        //	Read from the table.
        Conf
			conf = pipeline_configuration.read (parameter);
        String
			out;
        if (line.hasOption ("info"))
			{
            StringBuilder
				fmt = new StringBuilder();
            fmt
				.append ("PIPELINE = %s").append ("\n")
                .append ("PARAMETER = %s").append ("\n")
                .append ("VALUE = %s").append ("\n")
                .append ("VALIDATION_STRING = %s").append ("\n")
                .append ("DESCRIPTION = %s").append ("\n")
                .append ("LAST_UPDATE = %s");
            out = String.format (fmt.toString (),
				conf.pipeline,
				conf.parameter,
				conf.value,
                conf.validation,
				conf.description,
				conf.lastUpdate);
			}
		else
            out = String.format ("%s", conf.value);
        System.out.println (out);
		}
	}
catch (ParseException exception)
	{
    System.err.println (exception.getMessage ());
    System.exit (ExitCode.COMMAND_LINE_PARSE_ERROR.code ());
	}
catch (Configuration_Exception exception)
	{
    System.err.println (exception.getMessage ());
    System.exit (ExitCode.CONFIGURATION_ERROR.code ());
	}
catch (Database_Exception exception)
	{
    System.err.println (exception.getMessage ());
    System.exit (ExitCode.DATABASE_ERROR.code ());
	}
}

/** Print application usage.
*/
private static void usage
	(
	Options	options
	)
{
HelpFormatter
	formatter = new HelpFormatter ();
formatter.printHelp ("Pipeline_Configuration", options);
}

//	Build the command-line parameters for further parsing.
@SuppressWarnings ("static-access")
private static Options buildCommandLine ()
{
Option
	help = new Option
		("help", "print this message"),
	strict = new Option
		("strict", "treat value validation failures as global errors"),
	info = new Option
		("info", "show all available information for a parameter"),
	write = new Option
		("write", "enable write-mode for updating or adding a pipeline configuration");
Option
	buildtable = OptionBuilder
        .withArgName ("buildtable")
        .hasArg (false)
        .withDescription ("Create a new configuration table.")
        .create ("buildtable");
Option
	catalog = OptionBuilder
        .withArgName ("catalog")
        .hasArg ()
        .withDescription
			("override the configuration catalog (useful for testing)")
        .create ("catalog");
Option
	configuration = OptionBuilder
        .withArgName ("configuration")
        .hasArg ()
        .withDescription ("configuration file")
        .create ("configuration");
Option
	parameter = OptionBuilder
        .withArgName ("parameter")
        .hasArg ()
        .withDescription ("the parameter to query")
        .create ("parameter");
Option
	pipeline = OptionBuilder
        .withArgName ("pipeline")
        .hasArg ()
        .withDescription
			("the pipeline to use when querying for a parameter")
        .create ("pipeline");
Option
	table = OptionBuilder
        .withArgName ("table")
        .hasArg ()
        .withDescription
			("the table containing the configuration "
			+ "parameters, default: " + DEFAULT_DATABASE_TABLE_NAME)
        .create("table");
Option
	value = OptionBuilder
        .withArgName ("value")
        .hasArg ()
        .withDescription ("write-mode: the value to write")
        .create ("value");
Option
	validation = OptionBuilder
        .withArgName ("validation")
        .hasArg ()
        .withDescription ("write-mode: the validation string to write")
        .create ("validation");
Option
	description = OptionBuilder
        .withArgName ("description")
        .hasArg ()
        .withDescription ("write-mode: the description to write")
        .create ("description");

Option
	quiet = OptionBuilder
		.withArgName("quiet")
		.hasArg(false)
		.withDescription("suppress the output of command and version information")
		.create("quiet");
Options
	options = new Options ();
options.addOption (help);
options.addOption(quiet);
options.addOption (catalog);
options.addOption (configuration);
options.addOption (strict);
options.addOption (write);
options.addOption (parameter);
options.addOption (pipeline);
options.addOption (table);
options.addOption (info);
options.addOption (value);
options.addOption (validation);
options.addOption (description);
options.addOption (buildtable);

return options;
}

}
