/*	Draggable_Rows
 
idaeim CVS ID: Draggable_Rows.java,v 1.7 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	javax.swing.JLayeredPane;
import	javax.swing.SizeRequirements;
import	javax.swing.SwingUtilities;
import	javax.swing.event.MouseInputListener;
import	javax.swing.event.MouseInputAdapter;
import	javax.swing.event.ChangeEvent;
import	javax.swing.event.ChangeListener;
import	javax.swing.event.ListSelectionListener;
import	javax.swing.event.ListSelectionEvent;
import	java.awt.LayoutManager2;
import	java.awt.Component;
import	java.awt.Container;
import	java.awt.Dimension;
import	java.awt.Point;
import	java.awt.Insets;
import	java.awt.Color;
import	java.awt.event.MouseEvent;
import	java.util.Vector;
import	java.util.Iterator;


/**	<i>Draggable_Rows</i> manages the display of Components as a stack of
	draggable rows.
<p>
	A <i>Draggable_Rows_Layout</i> is used to maintain the display of
	the components as a single column of rows. All components, regardless
	of the layer they have been assigned to, are displayed in top-to-bottom
	order. Any components may be used as row components. These components
	may have their own functionality, including subcomponents.
<p>
	A MouseInputListener is attached to the base pane and to each row
	component and any subcomponents it might have. Whenever a mouse
	button is pressed in the pane the row component that it falls within
	is designated the {@link #Selected_Row() selected row} and any {@link
	#Add_Selection_Listener(ListSelectionListener) registered list
	selection listeners} are notified. If the shift key modifier is also
	used then the selected row may be dragged vertically to a new row
	position. Any {@link #Add_Row_Moved_Listener(ChangeListener)
	registered row move listeners} are notified whenever the position
	of a row changes.
<p>
	@author		Bradford Castalia, idaeim studio
	@version	1.7
	@see		JLayeredPane
*/
public class Draggable_Rows
	extends JLayeredPane
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Viewers.Draggable_Rows (1.7 2012/04/16 06:22:58)";	

//!	The width of the panel if it is empty.
private static final int
	EMPTY_WIDTH				= 150;

//!	The LayoutManager for the panel.
public Draggable_Rows_Layout
	Layout_Manager			= new Draggable_Rows_Layout ();

//!	Background color for selected panel (actiated row)
private Color
	Background_Color		= null;

//!	The list of listeners to be notified when a row location changes.
private Vector
	Row_Moved_Listeners		= new Vector ();

//!	The list of listeners to be notified when a row selection changes.
private Vector
	Selection_Listeners		= new Vector ();

//!	The currently selected row component.
public Component
	Selected_Row			= null;


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTOR	= 1 << 0,
	DEBUG_ROWS			= 1 << 1,
	DEBUG_LAYOUT		= 1 << 2,
	DEBUG_SIZES			= 1 << 3,
	DEBUG_DRAG			= 1 << 4,
	DEBUG_DRAGGING		= 1 << 5,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;
//	DEBUG				= DEBUG_ALL & ~DEBUG_DRAGGING;

/*==============================================================================
	Constructors
*/
/**	Constructs a Draggable_Rows pane containing a list of row components.
<p>
	The row components will be be displayed top-to-bottom in the
	order they occur in the list.
<p>
	@param	rows A Vector of Components to be added to the pane.
*/
public Draggable_Rows
	(
	Vector	rows
	)
{
if ((DEBUG & (DEBUG_CONSTRUCTOR | DEBUG_ROWS)) != 0)
	System.out.println (">>> Draggable_Rows: "
		+ ((rows == null) ? 0 : rows.size ()) + " rows");
setLayout (Layout_Manager);

if (rows != null)
	{
	//	Add the row components to the pane.
	for (int
			index = 0;
			index < rows.size ();
			index++)
		{
		if ((DEBUG & (DEBUG_CONSTRUCTOR | DEBUG_ROWS)) != 0)
			{
			((Component)rows.get (index)).setName ("Row " + index);
			System.out.println
				("    Add row " + index + ": Row " + index);
			}
		add ((Component)rows.get (index), JLayeredPane.DEFAULT_LAYER);
		}
	}

//	Register the mouse events handler with the pane.
addMouseListener (Mouse_Events_Handler);
addMouseMotionListener (Mouse_Events_Handler);
if ((DEBUG & (DEBUG_CONSTRUCTOR | DEBUG_ROWS)) != 0)
	System.out.println ("<<< Draggable_Rows");
}

/**	Constructs an empty Draggable_Rows pane.
*/
public Draggable_Rows ()
{this (null);}

/*==============================================================================
	Manipulators
*/
/**	Sets the currently selected row component.
<p>
	If the specified component is a row component it is selected. Each
	{@link #Add_Selection_Listener registered selection listener} is
	notified of the change. The ListSelectionEvent sent to each listener
	will contain this Draggable_Rows as the source and the index of the
	selected component (for both first and last index). However, because
	the index of a component might change, (@link #Selected_Row()} is the
	best way to obtain the currently selected row component.
<p>
	@param	component	The row component to be selected. If the component
		is not a row component, or is null, nothing is done.
	@return	This Draggable_Rows.
*/

public Draggable_Rows Selected_Row
	(
	Component	component
	)
{
final int
	index = getIndexOf (component);
if (index >= 0 &&
	component != Selected_Row)
	{
	if ((DEBUG & DEBUG_ROWS) != 0)
		System.out.println (">-< Draggable_Rows.Selected_Row: "
			+ component.getName () + " at index " + getIndexOf (component));
	Selected_Row = component;
	if (! Selection_Listeners.isEmpty ())
		javax.swing.SwingUtilities.invokeLater (new Runnable ()
			{
			public void run ()
			{
			//	Notify all selection listeners.
			if ((DEBUG & DEBUG_ROWS) != 0)
				System.out.println
					("--- Draggable_Rows.Selected_Row: "
						+"Notifying Selection_Listeners");
			Iterator
				listeners = Selection_Listeners.iterator ();
			while (listeners.hasNext ())
				((ListSelectionListener)listeners.next ()).valueChanged
					(new ListSelectionEvent (this, index, index, false));
			}});
	}
return this;
}

/**	Gets the currently selected row component.
<p>
	@return	The selected row Component. This will be null if not
		component is selected.
	@see	#Selected_Row(Component)
*/
public Component Selected_Row ()
{return Selected_Row;}

/**	Adds a list selection listener.
<p>
	Each registered listener is notified whenever the row selection changes.
<p>
	@param	listener	A ListSelectionListener that will have its
		valueChanged method called whenever the row selection changes. A
		duplicate or null listener is not added.
	@return	This Draggable_Rows.
	@see	ListSelectionListener
*/
public Draggable_Rows Add_Selection_Listener
	(
	ListSelectionListener	listener
	)
{
if (listener != null &&
	! Selection_Listeners.contains (listener))
	Selection_Listeners.add (listener);
return this;
}

/**	Removes a list selection listener.
<p>
	@param	listener	The ListSelectionListener to be removed.
	@return	true	If the listener was removed; false if it was not
		registered.
	@see	#Add_Selection_Listener(ListSelectionListener)
*/
public boolean Remove_Selection_Listener
	(
	ChangeListener	listener
	)
{return Selection_Listeners.remove (listener);}

/**	Adds a row moved listener.
<p>
	Each registered listener is notified whenever a row position changes.
<p>
	@param	listener	A ChangeListener that will have its
		stateChanged method called whenever a row position changes. A
		duplicate or null listener is not added.
	@return	This Draggable_Rows.
	@see	ChangeListener
*/
public Draggable_Rows Add_Row_Moved_Listener
	(
	ChangeListener	listener
	)
{
if (listener != null &&
	! Row_Moved_Listeners.contains (listener))
	Row_Moved_Listeners.add (listener);
return this;
}

/**	Removes a row moved listener.
<p>
	@param	listener	The ChangeListener to be removed.
	@return	true	If the listener was removed; false if it was not
		registered.
	@see	#Add_Row_Moved_Listener(ChangeListener)
*/
public boolean Remove_Row_Moved_Listener
	(
	ChangeListener	listener
	)
{return Row_Moved_Listeners.remove (listener);}

/**	Removes an indexed row component from the pane.
<p>
	Before the component is removed from the pane it, and any
	subcomponents, has this pane's mouse event listener removed. If the
	component being removed is the currently selected row, the {@link
	#Selected_Row() selected row} is reset to null.
<p>
	@param	index	The index of the component to remove. An invalid
		index is ignored.
*/
public void remove
	(
	int		index
	)
{
if (index >= 0 &&
	index < getComponentCount ())
	{
	Component
		component = getComponent (index);
	Remove_Mouse_Listeners (component);
	if (Selected_Row == component)
		Selected_Row = null;
	super.remove (index);
	}
}

/**	Adds a component to the pane.
<p>
	Before the component is added, this pane's mouse event listener is
	added to the component and any subcomponents it might have. The
	listener is not added if the component is already contained in this
	pane.
<p>
	@param	component	The Component to be added.
	@param	constraints	Layout contraints. This is expected to be
		a layer number Integer.
	@param	index		The position in the container's list at which to
		insert the component, where -1 means append to the end.
*/
protected void addImpl
	(
	Component	component,
	Object		constraints,
	int		index
	)
{
if ((DEBUG & DEBUG_LAYOUT) != 0)
	System.out.println (">>> Draggable_Rows.addtImpl");
if (getIndexOf (component) < 0)
	{
	//	New component.
	if ((DEBUG & DEBUG_LAYOUT) != 0)
		System.out.println ("    Add_Mouse_Listeners");
	Add_Mouse_Listeners (component);
	}
super.addImpl (component, constraints, index);
if ((DEBUG & DEBUG_LAYOUT) != 0)
	System.out.println ("<<< Draggable_Rows.addImpl");
}

//------------------------------------------------------------------------------
private void Add_Mouse_Listeners
	(
	Component	component
	)
{
if (component == null)
	return;
component.addMouseListener (Mouse_Events_Handler);
component.addMouseMotionListener (Mouse_Events_Handler);
if (component instanceof Container)
	{
	int
		index = ((Container)component).getComponentCount ();
	while (--index >= 0)
		Add_Mouse_Listeners
			(((Container)component).getComponent (index));
	}
}


private void Remove_Mouse_Listeners
	(
	Component	component
	)
{
if (component == null)
	return;
component.removeMouseListener (Mouse_Events_Handler);
component.removeMouseMotionListener (Mouse_Events_Handler);
if (component instanceof Container)
	{
	int
		index = ((Container)component).getComponentCount ();
	while (--index >= 0)
		Remove_Mouse_Listeners
			(((Container)component).getComponent (index));
	}
}

/*=*****************************************************************************
	Draggable_Rows_Layout
*/
/**	<i>Draggable_Rows_Layout</i> is a LayoutManager for a vertical stack of
	row Components.
<p>
	@see	LayoutManager2
*/
public class Draggable_Rows_Layout
	implements LayoutManager2
{
//!	Horizontal alignment coeficient: 0.0 left, 1.0 right, 0.5 center, etc.
private float
	Horizontal_Alignment	= 0.0f;

/**	Size constraint triplets - minimum, preferred, maximum -
	for each component dimension - width and height.
*/
private SizeRequirements[]
	Component_Widths		= new SizeRequirements[0],
	Component_Heights		= new SizeRequirements[0];

/**	Size constraint triplets - minimum, preferred, maximum -
	for the container dimension - width and height.
*/
private SizeRequirements
	Container_Widths		= null,
	Container_Heights		= null;

/*==============================================================================
	Constructors
*/
/**	Constructs a Draggable_Rows_Layout
*/
Draggable_Rows_Layout ()
{
if ((DEBUG & (DEBUG_CONSTRUCTOR | DEBUG_LAYOUT)) != 0)
	System.out.println
		(">-< Draggable_Rows_Layout");
}

/*==============================================================================
	Accessors
*/
/**	Sets the horizontal row alignment factor.
<p>
	An alignment of 0.0 aligns the rows against their left sides;
	1.0 aligns against the rights sides; 0.5 aligns rows along their
	centers; etc.
<p>
	The initial, default, alignment factor is 0.0.
<p>
	@param	alignment The horizontal alignment factor.
	@return	This Draggable_Rows_Layout.
*/
public Draggable_Rows_Layout setLayoutAlignmentX
	(
	float	alignment
	)
{
Horizontal_Alignment = Math.max (0.0f, Math.min (alignment, 1.0f));
return this;
}

/*==============================================================================
	Manipulators
*/
/**	Calculates a container layout size.
<p>
	If the container is empty or its size requirements have not been
	{@link #invalidateLayout(Container) invalidated} since they were last
	set, then nothing is done.
<p>
	A set of width and height {@link SizeRequirements} is maintained for
	each component and for the entire container. The components' minimum,
	preferred and maximum sizes are used; the {@link
	#setLayoutAlignmentX(float) horizontal alignment factor} is applied.
	Components that are not visible have no affect on the layout.
<p>
	The container's width size requirements are based on the {@link
	SizeRequirements#getAlignedSizeRequirements(SizeRequirements[])
	aligned size requirements} of the components; the height size
	requirements are based on the {@link
	SizeRequirements#getTiledSizeRequirements(SizeRequirements[]) tiled
	size requirements} of the components.
<p>
	@param container  The container associated with this layout manager.
*/
protected void Update_Sizes
	(
	Container	container
	)
{
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println (">>> Draggable_Rows_Layout.Update_Sizes:");
int
	total_rows = container.getComponentCount ();
if (total_rows == 0 ||
	Container_Widths != null)
	{
	if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
		System.out.println ("<<< Draggable_Rows_Layout.Update_Sizes:");
	return;
	}
if (Component_Widths.length != total_rows)
	{
	Component_Widths  = new SizeRequirements[total_rows];
	Component_Heights = new SizeRequirements[total_rows];
	}
for (int
		row = 0;
		row < total_rows;
		row++)
	{
	Component
		component = container.getComponent (row);
	Dimension
		minimum   = component.getMinimumSize (),
		preferred = component.getPreferredSize (),
		maximum   =	component.getMaximumSize ();
	if (! component.isVisible ())
		//	Invisible component.
		minimum.height   = minimum.width   =
		preferred.height = preferred.width =
		maximum.height   = maximum.width   = 0;
	Component_Widths[row]  = new SizeRequirements
		(minimum.width,  preferred.width,  maximum.width, Horizontal_Alignment);
	Component_Heights[row] = new SizeRequirements
		(minimum.height, preferred.height, maximum.height, 0.0f);
	if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
		System.out.println
			("    Component " + row + ": " + component.getName ()
				+ " is " + (component.isVisible () ? "" : " not") + " visible\n"
			+"    Component_Widths  = " + Component_Widths[row] + '\n'
			+"    Component_Heights = " + Component_Heights[row]);
	}
Container_Widths =
	//	Aligned, left side.
	SizeRequirements.getAlignedSizeRequirements (Component_Widths);
Container_Heights =
	//	Tiled, vertically.
	SizeRequirements.getTiledSizeRequirements (Component_Heights);
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println
		("<<< Draggable_Rows_Layout.Update_Sizes:\n"
		+"    Container_Widths  = " + Container_Widths + '\n'
		+"    Container_Heights = " + Container_Heights);
}

/**	Convenience constants for specifying which {@link
	#Layout_Size(Container, int) layout size} dimensions to calculate.
*/
protected static final int
	MINIMUM_DIMENSION		= -1,
	PREFERRED_DIMENSION		= 0,
	MAXIMUM_DIMENSION		= 1;

/**	Gets the dimensions for a container suitable for laying out the
	components it contains.
<p>
	The container, if it is not empty, has the preferred, minimum or maximum
	size, depending on which dimension is specified, of each component
	it contains examined. The
	container width is the maximum value of the component widths, or the
	{@link #EMPTY_WIDTH} if there are no components, plus the left and
	right insets amount. The container height is the sum of the heights
	of all visible components, or zero if there are no components, plus
	the top and bottom insets amount.
<p>
	@param container  The container associated with this layout manager.
	@param	which_dimension	If the {@link #MINIMUM_DIMENSION}, the
		minimum sizes will be calculated; if the {@link
		#PREFERRED_DIMENSION}, the preferred size is determined; if the
		{@link #MAXIMUM_DIMENSION} the maximum size.
	@return	The requested Dimension for the container.
	@see	#Update_Sizes(Container)
	@see	#minimumLayoutSize(Container)
	@see	#preferredLayoutSize(Container)
	@see	#maximumLayoutSize(Container)
*/
protected Dimension Layout_Size
	(
	Container	container,
	int			which_dimension
	)
{
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println
		(">>> Draggable_Rows_Layout.Layout_Size: " + which_dimension + " ("
		+ ((which_dimension == 0) ? "preferred" :
		  ((which_dimension  < 0) ? "minimum" : "maximum")) + ")");
Dimension
	size = new Dimension (EMPTY_WIDTH, 0);
if (container.getComponentCount () != 0)
	{
	synchronized (this)
		{
		Update_Sizes (container);
		size = new Dimension
			((which_dimension == 0) ? Container_Widths.preferred :
			((which_dimension  < 0) ? Container_Widths.minimum :
			/*which_dimension  > 0)*/ Container_Widths.maximum),
			((which_dimension == 0) ? Container_Heights.preferred :
			((which_dimension  < 0) ? Container_Heights.minimum :
			/*which_dimension  > 0)*/ Container_Heights.maximum)));
		}
	}
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println
		("    Size - " + size);

//	Add the insets boundary size.
Insets
	insets = container.getInsets ();
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println
		("    Container insets - "
			+ insets.left + "l, " + insets.right + "r, "
			+ insets.top + "t, " + insets.bottom + "b");
//	Clip to max, no integer overflow.
size.width = (int)Math.min
	((long)size.width + (long)insets.left + (long)insets.right,
	Integer.MAX_VALUE);
size.height = (int)Math.min
	((long)size.height + (long)insets.top + (long)insets.bottom,
	Integer.MAX_VALUE);
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println
		("<<< Draggable_Rows_Layout.Layout_Size: " + size);
return size;
}

//	LayoutManager interface ////////////////////////////////////////////////////

/**	Layout the container components as stacked rows.
<p>
	All components of the container are distributed in top-to-bottom
	order as they occur in the container's component list. The
	SizeRequirements for the components are first {@link
	#Update_Sizes(Container) updated}. The the bounds of each component
	are then determined using horizontally alignment into a single column
	and vertical top-to-bottom distribution into continguous rows.
<p>
	@param container  The container associated with this layout manager.
*/
public void layoutContainer
	(
	Container	container
	)
{
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println
		(">>> Draggable_Rows_Layout.layoutContainer");
int
	total_rows = container.getComponentCount ();
if (total_rows == 0)
	{
	if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
		System.out.println
			("<<< Draggable_Rows_Layout.layoutContainer: no rows");
	return;
	}

Dimension
	dimension = container.getSize ();
Insets
	insets    = container.getInsets ();
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println
		("    Container size - " + dimension + '\n'
		+"            insets - " + insets);
//	Shrink by boundary removal.
dimension.width  -= insets.left + insets.right;
dimension.height -= insets.top  + insets.bottom;

//	Determine the component positions.
int[]
	x_offsets = new int[total_rows],
	widths    = new int[total_rows],
	y_offsets = new int[total_rows],
	heights   = new int[total_rows];
synchronized (this)
	{
	Update_Sizes (container);
	//	Horizontally aligned.
	SizeRequirements.calculateAlignedPositions (dimension.width,
		Container_Widths,  Component_Widths,  x_offsets, widths);
	//	Vertically tiled (distributed).
	SizeRequirements.calculateTiledPositions   (dimension.height,
		Container_Heights, Component_Heights, y_offsets, heights);
	}

//	Position the components.
for (int
		row = 0;
		row < total_rows;
		row++)
	{
	Component
		component = container.getComponent (row);
	if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
		System.out.println
			("    Component " + row + ": " + component.getName ()
				+ " setBounds ("
				+ (int)Math.min ((long)insets.left + (long)x_offsets[row],
					Integer.MAX_VALUE) + "x, "
				+ (int)Math.min ((long)insets.top + (long)y_offsets[row],
				 	Integer.MAX_VALUE) + "y, "
				+ widths[row] + "w, "
				+ heights[row] + "h)");
	component.setBounds
		(
		//	Horizontal (clip to max, no integer overflow).
		(int)Math.min ((long)insets.left + (long)x_offsets[row],
			Integer.MAX_VALUE),
		//	Vertical (clip to max, no integer overflow).
		(int)Math.min ((long)insets.top + (long)y_offsets[row],
			Integer.MAX_VALUE),
		widths[row],
		heights[row]
		);
	}
if ((DEBUG & (DEBUG_LAYOUT | DEBUG_SIZES)) != 0)
	System.out.println ("<<< Draggable_Rows_Layout.layoutContainer");
}

/**	Gets the minumum layout dimensions for the container.
<p>
	@param container  The container associated with this layout manager.
	@return	The requested Dimension for the container.
	@see	#Layout_Size(Container, int)
*/
public Dimension minimumLayoutSize
	(
	Container	container
	)
{return Layout_Size (container, MINIMUM_DIMENSION);}

/**	Gets the preferred layout dimensions for the container.
<p>
	@param container  The container associated with this layout manager.
	@return	The requested Dimension for the container.
	@see	#Layout_Size(Container, int)
*/
public Dimension preferredLayoutSize
	(
	Container	container
	)
{return Layout_Size (container, PREFERRED_DIMENSION);}

//!	Ignored.
public void addLayoutComponent (String name, Component component) {}
//!	Ignored.
public void removeLayoutComponent (Component component) {}

// LayoutManager2 /////////////////////////////////////////////////////////

/**	Gets the maximum layout dimensions for the container.
<p>
	@param container  The container associated with this layout manager.
	@return	The requested Dimension for the container.
	@see	#Layout_Size(Container, int)
*/
public Dimension maximumLayoutSize
	(
	Container	container
	)
{return Layout_Size (container, MAXIMUM_DIMENSION);}

/**	Gets the horizontal alignment factor for the layout.
<p>
	@param container  The container associated with this layout manager.
	@return	The horizontal alignment factor.
	@see	#setLayoutAlignmentX(float)
*/
public float getLayoutAlignmentX (Container container)
{return Horizontal_Alignment;}

/**	Gets the vertical alignment factor for the layout.
<p>
	@param container  The container associated with this layout manager.
	@return	The horizontal alignment factor which is always 0.0.
*/
public float getLayoutAlignmentY (Container container)
{return 0.0f;}

//!	Ignored.
public void addLayoutComponent
	(
	Component	component,
	Object		constraints
	)
{addLayoutComponent (component.getName (), component);}


/**	The layout constraints for the container are invalidated.
<p>
	The next time a {@link #layoutContainer(Container) layout} is to
	occur the layout constraints will be {@link #Update_Sizes(Container)
	updated}.
<p>
	This method is called by AWT when the invalidate method is called
	on the Container. Since the invalidate method may be called 
	asynchronously to the event thread, this method may be called
	asynchronously.
<p>
	@param container  The affected container.
*/
public synchronized void invalidateLayout
	(
	Container	container
	)
{Container_Widths = null;}

};	//	End of Draggable_Rows_Layout class.

/*==============================================================================
	Mouse events handler for dragging.
*/
private MouseInputListener
	Mouse_Events_Handler	= new MouseInputAdapter ()
{
//!	Row component being dragged.
private Component
	Row						= null;

//!	The row's layer.
private int
	Row_Layer;

//!	Position of the row in the layer.
private int
	Row_Position;

//!	Location of the row in the container when dragging started.
private Point
	Row_Origin				= new Point ();

//!	Y offset of the mouse in the container when dragging started.
private int
	Mouse_Y_Offset;

//!	Maximum y row drap location in the container.
private int
	Max_Y;

//!	Flag to signal when the handlers are ready to accept events.
private boolean
	Ready					= true;

//	MouseInputListener /////////////////////////////////////////////////////////

/**	Handle a mouse button press event.
<p>
	The row component where the mouse press occured becomes the {@link
	#Selected_Row(Component) selected row}. If the event handlers are not
	ready (row move notifications are pending), the Shet key modifier is
	not set, there are less than two row components, or no component row
	was selected then the event is ignored. Otherwise the event is
	consumed (to prevent other row component actions) and the row
	dragging state variables are set. The row component is elevated to
	the DRAG_LAYER so it will be displayed over the other row components
	while it is being moved.
<p>
	@param	event	A MouseEvent.
*/
public void mousePressed
	(
	MouseEvent	event
	)
{
Point
	mouse_point = this_Point (event);
if ((DEBUG & DEBUG_DRAG) != 0)
	System.out.println
		(">>> Draggable_Rows.Mouse_Events_Handler.mousePressed: "
			+ mouse_point);
Selected_Row (Row = getComponentAt (mouse_point));
if (! Ready ||
	! event.isShiftDown () ||
	getComponentCount () < 2 ||
	Row == null)
	{
	if ((DEBUG & DEBUG_DRAG) != 0)
		System.out.println
			("<<< Draggable_Rows.Mouse_Events_Handler.mousePressed: ignored\n"
			+"    "
				+ (Ready ?
				  (event.isShiftDown () ?
				  ((getComponentCount () < 2) ?
				  ((Row == null) ?
				"No row component!" : "No reason!?!") :
				"Insufficient components.") :
				"No Shift modifier.") :
				"Not Ready; Row_Moved_Event notifications pending."));
	Row = null;
	return;
	}

Mouse_Y_Offset = mouse_point.y;
Row.getLocation (Row_Origin);
Row_Layer = getLayer (Row);
Row_Position = getPosition (Row);
Max_Y = getHeight () - Row.getHeight ();

//	The event is meant to initiate a row drag; consume it.
event.consume ();

if ((DEBUG & DEBUG_DRAG) != 0)
	{
	int
		row_index = getIndexOf (Row);
	System.out.println
		("    Source: " + Row.getName () + '\n'
		+"    Container screen location = " + getLocationOnScreen () + '\n'
		+"          Row screen location = " + Row.getLocationOnScreen () + '\n'
		+"       Row container location = " + Row_Origin + '\n'
		+"                    Row layer = " + Row_Layer + '\n'
		+"           Row layer position = " + Row_Position + '\n'
		+"                    Row index = " + getIndexOf (Row) + '\n'
		+"    Current positions -");
	for (int index = 0;
			 index < getComponentCount ();
			 index++)
		System.out.println
			("    Component " + index + ": " + getComponent (index).getName ());
	}
if ((DEBUG & DEBUG_DRAG) != 0)
	System.out.println ("    setLayer (Source, layer "
		+ JLayeredPane.DRAG_LAYER.intValue () + ")");

/*	Elevate the drag row component to the drag layer.

	N.B.: Moving the component to the drag layer moves it to position 0.
	All other components shift down in position.
*/

setLayer (Row, JLayeredPane.DRAG_LAYER.intValue ());
if ((DEBUG & DEBUG_DRAG) != 0)
	{
	System.out.println
		("    Drag positions -");
	for (int index = 0;
			 index < getComponentCount ();
			 index++)
		System.out.println
			("    Component " + index + ": " + getComponent (index).getName ());
	System.out.println
		("<<< Draggable_Rows.Mouse_Events_Handler.mousePressed");
	}
}

/**	Handle a mouse drag event.
<p>
	The event is ignored if there is no active row component being
	dragged or the event handlers are not ready (row move notifications
	are pending). Otherwise the row component being dragged is moved to
	the new vertical location in the container relative to the mouse
	event location. The location is limited to keep the row component
	within the bounds of the container.
<p>
	@param	event	A MouseEvent.
*/
public void mouseDragged
	(
	MouseEvent	event
	)
{
if (Row == null ||
	! Ready)
	return;
if ((DEBUG & DEBUG_DRAG) != 0 &&
	(DEBUG & DEBUG_DRAGGING) != 0)
	System.out.println
		(">>> Draggable_Rows.Mouse_Events_Handler.mouseDragged: "
			+ this_Point (event));
int
	y = Math.max (0,
		Math.min (Max_Y,
		Row_Origin.y + this_Point (event).y - Mouse_Y_Offset));
if ((DEBUG & DEBUG_DRAG) != 0 &&
	(DEBUG & DEBUG_DRAGGING) != 0)
	System.out.println
		("    Row.getLocationOnScreen = "
			+ Row.getLocationOnScreen () + '\n'
		+"    Row.getLocation         = "
			+ Row.getLocation () + '\n'
		+"    Row.setLocation (" + Row_Origin.x
			+ ", " + y + ")");
Row.setLocation (Row_Origin.x, y);
if ((DEBUG & DEBUG_DRAG) != 0 &&
	(DEBUG & DEBUG_DRAGGING) != 0)
	System.out.println
		("<<< Draggable_Rows.Mouse_Events_Handler.mouseDragged");
}

/**	Handle a mouse button released event.
<p>
	The event is ignored if there is no active row component being
	dragged or the event handlers are not ready (row move notifications
	are pending). Otherwise the {@link #Insert_Position(int) insert
	position} of the mouse location is determined and the row is put
	back in its original layer at this position. If the location of
	the row as moved at all (not just its position) the layout is
	invalidated and revalidation is scheduled on the event queue.
<p>
	If the row component has changed position and there are any {@link
	#Add_Row_Moved_Listener(ChangeListener) registered row move
	listeners} the event handlers are set to be not ready and
	notification of all the row move listeners is scheduled on the event
	queue. When all notifications are complete the event handlers are put
	back in their ready state and dragging row is deactivated.
<p>
	@param	event	A MouseEvent.
*/
public void mouseReleased
	(
	MouseEvent	event
	)
{
if (Row == null ||
	! Ready)
	return;
Point
	mouse_point = this_Point (event);
if ((DEBUG & DEBUG_DRAG) != 0)
	System.out.println
		(">>> Draggable_Rows.Mouse_Events_Handler.mouseReleased: "
			+ mouse_point);
int
	insert_position = Insert_Position
		(Row.getY () + (Row.getHeight () >> 1));
if ((DEBUG & DEBUG_DRAG) != 0)
	{
	System.out.println
		("    From " + Row_Position + " to " + insert_position + '\n'
		+"    Pre-move positions -");
	for (int index = 0;
			 index < getComponentCount ();
			 index++)
		System.out.println
			("    Component " + index + ": " + getComponent (index).getName ());
	System.out.println
		("    setLayer (Row, layer " + Row_Layer
			+ ", position " + insert_position + ")");
	}
setLayer (Row, Row_Layer, insert_position);
if ((DEBUG & DEBUG_DRAG) != 0)
	{
	System.out.println
		("    Post-move positions -");
	for (int index = 0;
			 index < getComponentCount ();
			 index++)
		System.out.println
			("    Component " + index + ": " + getComponent (index).getName ());
	}

if (mouse_point.y != Mouse_Y_Offset)
	{
	//	The row has been moved.
	invalidate ();
	javax.swing.SwingUtilities.invokeLater (new Runnable ()
		{
		public void run ()
		{validate ();}
		});
	}

if (insert_position < 0)
	insert_position = getComponentCount () - 1;
if (insert_position != Row_Position &&
	! Row_Moved_Listeners.isEmpty ())
	{
	Ready = false;
	final int
		position = insert_position;
	javax.swing.SwingUtilities.invokeLater (new Runnable ()
		{
		public void run ()
		{
		//	Notify all change listeners.
		if ((DEBUG & DEBUG_DRAG) != 0)
			System.out.println
				("--- Draggable_Rows.Mouse_Events_Handler.mouseReleased: "
					+"Notifying Row_Moved_Listeners");
		Iterator
			listeners = Row_Moved_Listeners.iterator ();
		while (listeners.hasNext ())
			((ChangeListener)listeners.next ()).stateChanged
				(new Row_Moved_Event (Row, Row_Position, position));
		//	Drag completed.
		Ready = true;
		Row = null;
		}});
	if ((DEBUG & DEBUG_DRAG) != 0)
		System.out.println
			("    Row_Moved_Listeners notification scheduled.");
	}
else
	Row = null;
if ((DEBUG & DEBUG_DRAG) != 0)
	System.out.println
		("<<< Draggable_Rows.Mouse_Events_Handler.mouseReleased");
}

/**	Gets the insert position for a row.
<p>
	Each component that is showing is checked to see if the reference y
	position is within the component. If so, the index of that component
	is the returned position. This assumes that all components are laid
	out in top-to-bottom order, which is the case for a
	Draggable_Rows_Layout.
<p>
	@param	reference_y	The vertical reference location for the insertion.
	@return	The index of the showing component that contains the reference
		y location.
*/
private int Insert_Position
	(
	int		reference_y
	)
{
if ((DEBUG & DEBUG_DRAG) != 0)
	System.out.println
		(">>> Draggable_Rows.Mouse_Events_Handler.Insert_Position: "
			+ reference_y);
int
	component_y = getInsets ().top,
	total_rows = getComponentCount (),
	index;
for (index = 0;
	 index < total_rows;
	 index++)
	{
	Component
		component = getComponent (index);
	if (component.isShowing ())
		{
		if (component != Row)
			{
			int
				top = component.getY ();
			if (top < 0)
				//	Component extends above the container top.
				component_y -= top;
			}
		component_y += component.getHeight ();
		if ((DEBUG & DEBUG_DRAG) != 0)
			System.out.println
				("    " + index + ": " + component_y);
		if (reference_y < component_y)
			//	The reference falls within the component bounds.
			break;
		}
	}
if (index == total_rows)
	index = -1;
if ((DEBUG & DEBUG_DRAG) != 0)
	System.out.println
		("<<< Draggable_Rows.Mouse_Events_Handler.Insert_Position: " + index);
return index;
}

};	//	End of Mouse_Events_Handler.

//------------------------------------------------------------------------------
/**	A Row_Moved_Event carries the information that a component
	row has moved from one position to another in the pane.
*/
public class Row_Moved_Event
	extends ChangeEvent
{
//!	The position from which the component moved.
public int
	From;
//!	The position to which the component moved.
public int
	To;

/**	Constructs a Row_Moved_Event.
<p>
	@param	row	The Component row that has moved.
	@param	from	The row position from which the component moved.
	@param	to	The row position to which the component moved.
*/
Row_Moved_Event
	(
	Component	row,
	int			from,
	int			to
	)
{
super (row);
From = from;
To = to;
}
};	//	End of Row_Moved_Event.

/*==============================================================================
	Helpers
*/
/**	Gets the location of a mouse event relative to the coordinates of
	this pane.
<p>
	The event's location point is translated from the coordinates of the
	event's component to the coordinates of this pane.
<p>
	@param	event	A MouseEvent.
	@return	The mouse event location in this pane's coordinate space.
	@see	SwingUtilities#convertPoint(Component, Point, Component)
*/
public Point this_Point
	(
	MouseEvent	event
	)
{
Component
	component = event.getComponent ();
if (component == this)
	return event.getPoint ();
return SwingUtilities.convertPoint (component, event.getPoint (), this);
}

/**	Gets the location of a mouse event relative to the coordinates of
	the display screen.
<p>
	The event's location point is translated from the coordinates of the
	event's component to the coordinates of the display screen.
<p>
	@param	event	A MouseEvent.
	@return	The mouse event location in the display screen's coordinate space.
	@see	SwingUtilities#convertPointToScreen(Point, Component)
*/
public static Point Screen_Point
	(
	MouseEvent	event
	)
{
Point
	point = event.getPoint ();
SwingUtilities.convertPointToScreen (point, event.getComponent ());
return point;
}

}	//	End of Draggable_Rows class.
