/*	Parser

PIRL CVS ID: Parser.java,v 1.35 2012/04/16 06:14:23 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.PVL;

import	java.io.File;
import	java.io.FileInputStream;
import	java.io.InputStream;
import	java.io.InputStreamReader;
import	java.io.BufferedReader;
import	java.io.Reader;
import	java.io.IOException;
import	java.io.UnsupportedEncodingException;
import	java.lang.Character;
import	java.lang.Long;
import	java.lang.Double;
import	java.lang.Math;
import	java.util.Vector;

import	PIRL.Strings.String_Buffer;
import	PIRL.Strings.String_Buffer_Reader;

/**	The <I>Parser</I> extends the <I>String_Buffer_Reader</I> to interpret
	the characters as a sequence of Parameter Value Language (PVL) syntax
	statements.
<P>
	This Parser implements the syntax of the PVL used by the
	Planetary Data System (PDS) as specified by the <A
	HREF="http://www.ccsds.org" TARGET="_top"> Consultative Committee for
	Space Data Systems</A> in the <A
	HREF="http://www.ccsds.org/blue_books.html" TARGET="_top"> Blue
	Book</A> "Parameter Value Language Specification (CCSDS0006,8)", June
	2000 [CCSDS 641.0-B-2] and <A
	HREF="http://www.ccsds.org/green_books.html" TARGET="_top"> Green
	Book</A> "Parameter Value Language - A Tutorial", May 1992 [CCSDS
	641.0-G-1] documents. PVL has been accepted by the <A
	HREF="http://www.iso.ch" TARGET="_top"> International Standards
	Organization (ISO)</A>, as a Draft Standard (ISO/CD 14961:1997). The
	PVL syntax defines a <I>Parameter</I> with this basic format:

<P>
	<BLOCKQUOTE>
	[<B><I>Comments</I></B>]<BR>
	<B><I>Name</I></B> [<B>= <I>Value</I></B>][<B>;</B>]
	</BLOCKQUOTE>
<P>
	The optional Comments are enclosed in C-style delimiters, or
	optionally preceeded by a crosshatch ('#') character on each line.
	The PVL syntax for a <I>Value</I> follows this format:
<P>
	<BLOCKQUOTE>
	[<B>(</B>|<B>{</B>]<B><I>Datum</I></B> [<B><<I>Units</I>></B>][<B>, <I>Datum</I></B> [...]][<B>)</B>|<B>}</B> [<B><<I>Units</I>></B>]]
	</BLOCKQUOTE>
<P>
	The purpose of a Parser object is to assemble Parameter and Value
	objects using the PVL statements obtained from the associated
	String_Buffer_Reader.
<P>
	The class methods that perform the parsing of the character source are
	organized into a hierarchy:
<P>
<UL TYPE="DISC">
<LI><CODE>Get</CODE>
<LI><CODE>Add_To</CODE>
	<UL TYPE="DISC">
	<LI><CODE>Get_Parameter</CODE>
		<UL TYPE="DISC">
		<LI><CODE>Get_Comments</CODE>
		<LI><CODE>Get_Value</CODE>
			<UL TYPE="DISC">
			<LI><CODE>Get_Datum</CODE>
				<UL TYPE="DISC">
				<LI><CODE>Get_Quoted_String</CODE>
				</UL>
			<LI><CODE>Get_Units</CODE>
			</UL>
		</UL>
	</UL>
</UL>
<P>
	Higher level methods utilize lower level methods to assemble their
	constituent parts. At the top level an aggregate of all Parameters
	that can be interpreted from the input will be collected by getting
	as many Parameters as possible; a Parameter is produced from the
	input stream by getting any comments, a name String, and a Value; a
	Value includes as many datum and optional units descriptions that can
	be sequentially found in the input stream; and a datum is composed
	from primitive syntactic elements, including integer or real number
	representations or character strings which may be quoted (a Parameter
	name may also be a quoted string). Typically applications will only
	use the top method(s). Applications needing finer grained control
	over input stream parsing may, of course, use the lower level methods
	directly, however it is much easier to just get all of the Parameters
	from an input source and then manipulate the Parameter-Value object
	hierarchy.
<P>
	Each method that parses the input stream interprets the contents of
	the logical String_Buffer_Reader which presents the entire virtual
	contents of the stream from the current location onwards. Except for
	the top level which does not interpret the character source directly,
	these methods first seek forward from the current location to the
	beginning of a potentially relevant syntactic character sequence. If
	the sequence is recognized as suitable for the item the method is
	responsible for interpreting then the appropriate end of the sequence
	is found and the characters it contains are translated into the
	corresponding internal form of object class variable. If the
	translation is successful then the logical <CODE>Next_Location</CODE>
	of the String_Buffer_Reader is moved forward to the end of the
	sequence before the iterative interpretation of the stream continues.
	If, however, the beginning of a recognizable syntactic sequence is
	not found, or the translation of a sequence fails, then the method
	returns empty handed, and without having advanced the logical
	String_Buffer_Reader, to the invoking method which may invoke a
	different method in an attempt to get a different item or itself
	discontinue its efforts to assemble an item. Thus each parsing method
	either gets its item and advances the current location in the
	character stream, or does not get its item nor advance the stream;
	i.e. the PVL statements encountered in the character source are
	sequentially translated at the same time that the input stream is
	incrementally moved forward.
<P>
	@see		String_Buffer_Reader
	@see		Parameter
	@see		Value

	@author		Bradford Castalia, UA/PIRL
	@version	1.35
*/
public class Parser
	extends String_Buffer_Reader
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.PVL.Parser (1.35 2012/04/16 06:14:23)";

/*------------------------------------------------------------------------------
	PVL syntax elements
*/
/**	The default for enforcing {@link #Strict(boolean) strict} PVL syntax rules.
*/
public static boolean
	Strict_Default						= false;
private boolean
	Strict								= Strict_Default;

/**	{@link #Verbatim_Strings(boolean) Verbatim strings} default.
*/
public static boolean
	Verbatim_Strings_Default			= false;
private boolean
	Verbatim_Strings					= Verbatim_Strings_Default;

/**	{@link #String_Continuation(boolean) String continuation} default.
*/
public static boolean
	String_Continuation_Default			= true;
private boolean
	String_Continuation					= String_Continuation_Default;

/**	The default for allowing {@link #Crosshatch_Comments(boolean)
	crosshatched-to-EOL comments}.
*/
public static boolean
	Crosshatch_Comments_Default			= (! Strict_Default);
private boolean
	Crosshatch_Comments					= Crosshatch_Comments_Default;

/**	The default for treating {@link #All_Values_Strings(boolean) all
	values as strings}.
*/
public static boolean
	All_Values_Strings_Default			= false;
private boolean
	All_Values_Strings					= All_Values_Strings_Default;

/**	Begins a "crosshatch comment" that extends to the end of the line: '#'.
*/
public static final char
	CROSSHATCH							= '#';
/*	The crosshatch comment start character must not be the same as 
	the first character of COMMENT_START_DELIMITERS.
*/

/**	The PVL character encoding: "US-ASCII".
*/
public static final String
	CHARACTER_ENCODING					= "US-ASCII";

/**	Characters reserved by the PVL syntax.
<P>
	<BLOCKQUOTE>
	{}()[]<>&\"',=;#%~|+! \t\r\n\f\013
	</BLOCKQUOTE>
<P>
	Some of these characters have special meanings in specific contexts
	as delimiters of PVL items.
*/
public static final String
	RESERVED_CHARACTERS					= "{}()[]<>&\"',=;#%~|+! \t\r\n\f\013";


//	Delimiter characters:

/**	Delimits a Parameter name from its Value: '='.
*/
public static final char
	PARAMETER_NAME_DELIMITER			= '=';

/**	Delimits elements of an ARRAY Value: ','.
*/
public static final char
	PARAMETER_VALUE_DELIMITER			= ',';

/**	Encloses a TEXT STRING Value: '"'.
*/
public static final char
	TEXT_DELIMITER						= '"';

/**	Encloses a SYMBOL STRING Value: '\''.
*/
public static final char
	SYMBOL_DELIMITER					= '\'';

/**	Marks the start of a SET ARRAY Value: '{'.
*/
public static final char
	SET_START_DELIMITER					= '{';

/**	Marks the end of a SET ARRAY Value: '}'.
*/
public static final char
	SET_END_DELIMITER					= '}';

/**	Marks the start of a SEQUENCE ARRAY Value: '('.
*/
public static final char
	SEQUENCE_START_DELIMITER			= '(';

/**	Marks the end of a SEQUENCE ARRAY Value: ')'
*/
public static final char
	SEQUENCE_END_DELIMITER				= ')';

/**	Marks the start of a Value units description: '<'.
*/
public static final char
	UNITS_START_DELIMITER				= '<';

/**	Marks the end of a Value units description: '>'.
*/
public static final char
	UNITS_END_DELIMITER					= '>';

/**	Encloses the datum of a Value in radix base notation: '#'.
*/
public static final char
	NUMBER_BASE_DELIMITER				= '#';

/**	Marks the end of a PVL statement: ';'.
*/
public static final char
	STATEMENT_END_DELIMITER				= ';';

/**	Indicates that the statement continues in the next record: '&'.
*/
public static final char
	STATEMENT_CONTINUATION_DELIMITER	= '&';

/**	Indicates that the quoted string continues unbroken in the next record: '-'.
*/
public static final char
	STRING_CONTINUATION_DELIMITER		= '-';


//	Delimiter strings:

/**	Character sequence that separates PVL statement lines: "\r\n" (CR-NL).
*/
public static final String
	LINE_BREAK							= "\r\n";

/**	Set of "whitespace" characters between PVL tokens: " \t\r\n\f\013"
	(SP, HT, CR, NL, FF, and VT).
*/
public static final String
	WHITESPACE							= " \t\r\n\f\013";

/**	Marks the start of a comment string: '/' and '*'.
*/
public static final String
	COMMENT_START_DELIMITERS			= "/*";

/**	Marks the end of a comment string: '*' and '/'.
*/
public static final String
	COMMENT_END_DELIMITERS				= "*/";

/**	Set of characters that suggests a DATE_TIME type of STRING Value: "-:".
*/
public static final String
	DATE_TIME_DELIMITERS				= "-:";

/**	Encloses a verbatim (uninterpreted) string: "\\v".
*/
public static final String
	VERBATIM_STRING_DELIMITERS			= "\\v";


/*	For convenience.

	Put WHITESPACE first to force the following characters to
	be cast as Strings for the concatenation.
*/
private static final String
	LINE_DELIMITERS						= 
		"\f\013" +
		LINE_BREAK,
	PARAMETER_NAME_DELIMITERS			=
		WHITESPACE +
		PARAMETER_NAME_DELIMITER +
		STATEMENT_END_DELIMITER,
	PARAMETER_VALUE_DELIMITERS			=
		WHITESPACE +
		PARAMETER_VALUE_DELIMITER +
		SET_START_DELIMITER +
		SET_END_DELIMITER +
		SEQUENCE_START_DELIMITER +
		SEQUENCE_END_DELIMITER +
		UNITS_START_DELIMITER +
		STATEMENT_END_DELIMITER;


/*	Each reserved parameter name is associated with a specific
	parameter classification in this list.
*/
private static final String[]
	SPECIAL_NAMES =
	{
	/*
		BEGIN_XXXX classifications are identical to XXXX
		classifications. Both forms must be in the list so they can be
		identified during parsing. However when translating a
		classification code to its special name (see Special_Name) the
		first entry will be returned. Since the BEGIN form is more
		"proper" they are put first in the list.
	*/
	"BEGIN_OBJECT",
	"OBJECT",
	"BEGINOBJECT",
	"END_OBJECT",
	"ENDOBJECT",
	"BEGIN_GROUP",
	"GROUP",
	"BEGINGROUP",
	"END_GROUP",
	"ENDGROUP",
	"END"
	};

private static final int[]
	SPECIAL_CLASSIFICATIONS =
	{
	Parameter.BEGIN_OBJECT,
	Parameter.OBJECT,
	Parameter.BEGIN_OBJECT,
	Parameter.END_OBJECT,
	Parameter.END_OBJECT,
	Parameter.BEGIN_GROUP,
	Parameter.GROUP,
	Parameter.BEGIN_GROUP,
	Parameter.END_GROUP,
	Parameter.END_GROUP,
	Parameter.END_PVL
	};


/**	The default name of the aggregate Parameter to contain all
	Parameters when a Parser Get finds more than one Parameter:
	"The Container".
*/
public static final String
	CONTAINER_NAME			= "The Container";


//	When throwing an exception is inappropriate.
private PVL_Exception
	First_Warning			= null,
	Last_Warning			= null;	
private boolean
	Use_First_Warning		= true;


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_SETUP			= 1 << 0,
	DEBUG_AGGREGATE		= 1 << 1,
	DEBUG_PARAMETER		= 1 << 2,
	DEBUG_VALUE			= 1 << 3,
	DEBUG_DATUM			= 1 << 4,
	DEBUG_UNITS			= 1 << 5,
	DEBUG_UTILITIES		= 1 << 6,
	DEBUG_COMMENTS		= 1 << 7,
	DEBUG_QUOTED_STRING	= 1 << 8,
	DEBUG_WARNINGS		= 1 << 9,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;


/*==============================================================================
	Constructors
*/
/*------------------------------------------------------------------------------
	File input
*/
/**	Creates a Parser using a Reader as the source of PVL statements,
	and sets a limit on the amount to read.
<P>
	@param	reader	The Reader to use as the source of characters.
	@param	read_limit	The maximum amount to read.
	@throws	PVL_Exception	From <CODE>Set_Reader</CODE>.
	@see	#Set_Reader(Reader, long)
*/
public Parser
	(
	Reader		reader,
	long		read_limit
	)
	throws PVL_Exception
{Set_Reader (reader, read_limit);}

/**	Creates a Parser using a Reader as the source of PVL statements,
	with no limit on the amount to read.
<P>
	@param	reader	The Reader to use as the source of characters.
	@throws	PVL_Exception	From <CODE>Set_Reader</CODE>.
	@see	#Set_Reader(Reader, long)
*/
public Parser
	(
	Reader		reader
	)
	throws PVL_Exception
{Set_Reader (reader, NO_READ_LIMIT);}

/**	Creates a Parser using a File to create a new Reader as the
	source of PVL statements, and sets a limit on the amount to read.
<P>
	@param	file	The File to be the basis for a new Reader.
	@param	read_limit	The maximum amount to read.
	@throws	PVL_Exception	From <CODE>Set_Reader</CODE>.
	@see	#Set_Reader(File, long)
*/
public Parser
	(
	File		file,
	long		read_limit
	)
	throws PVL_Exception
{Set_Reader (file, read_limit);}

/**	Creates a Parser using a File to create a new Reader as the
	source of PVL statements, with no limit on the amount to read.
<P>
	@param	file	The File to be the basis for a new Reader.
	@throws	PVL_Exception	From <CODE>Set_Reader</CODE>.
	@see	#Set_Reader(File, long)
*/
public Parser
	(
	File		file
	)
	throws PVL_Exception
{Set_Reader (file, NO_READ_LIMIT);}

/**	Creates a Parser using an InputStream to create a new Reader as the
	source of PVL statements, and sets a limit on the amount to read.
<P>
	@param	input_stream	The InputStream to be the basis for a new Reader.
	@param	read_limit	The maximum amount to read.
	@throws	PVL_Exception	From <CODE>Set_Reader</CODE>.
	@see	#Set_Reader(InputStream, long)
*/
public Parser
	(
	InputStream	input_stream,
	long		read_limit
	)
	throws PVL_Exception
{Set_Reader (input_stream, read_limit);}

/**	Creates a Parser using an InputStream to create a new Reader as the
	source of PVL statements, with no limit on the amount to read.
<P>
	@param	input_stream	The InputStream to be the basis for a new Reader.
	@throws	PVL_Exception	From <CODE>Set_Reader</CODE>.
	@see	#Set_Reader(InputStream, long)
*/
public Parser
	(
	InputStream	input_stream
	)
	throws PVL_Exception
{Set_Reader (input_stream, NO_READ_LIMIT);}

/**	Creates a Parser with no source of PVL statements.
*/
public Parser () {}


/*------------------------------------------------------------------------------
	String input
*/
/**	Creates a Parser using a String as the source of PVL statements.
<P>
	@param	string	The String to use as the source of characters.
*/
public Parser
	(
	String	string
	)
{super (string);}

/**	Creates a Parser using a character array as the source of PVL
	statements.
<P>
	@param	char_array	The source of characters.
*/
public Parser
	(
	char[]	char_array
	)
{this (String.valueOf (char_array));}

/**	Creates a Parser using a characer array subset as the source of PVL
	statements.
<P>
	@param	char_array	The source of characters.
	@param	offset		The array index where the character source starts.
	@param	length		The number of characters to source.
*/
public Parser
	(
	char[]	char_array,
	int		offset,
	int		length
	)
{this (String.valueOf (char_array, offset, length));}


/*==============================================================================
	Accessors:
*/
/**	Sets the Reader where the Parser will obtain characters.
<P>
	If {@link String_Buffer_Reader#Filter_Input()
	<CODE>Filter_Input</CODE>} is true, then a <CODE>SIZED_RECORDS</CODE>
	{@link #Warning() <CODE>Warning</CODE>} is registered.
<P>
	<B>N.B.</B>: When <CODE>Filter_Input</CODE> is used it will read the
	first two characters from the reader to test them for non-printable
	values, unless the test has already been done on the reader or {@link
	String_Buffer_Reader#Filter_Input(boolean) input filtering has been
	disabled}. If no characters are yet available from the reader - e.g.
	if the reader is backed by a pipe or network socket - then the read
	will block until two characters become available (or an IOException
	occurs). The base String_Buffer_Reader does its own
	<CODE>Filter_Input</CODE> test every time its internal buffer content
	is {@link String_Buffer_Reader#Extend() extended}. Therefore, to
	prevent the Filter_Input() test from blocking when the reader is set
	defer setting the reader until blocking is acceptable, it is known
	that the required input will be available or filtering has been
	disabled. For example:
<P>
<PRE>
Parser parser = new Parser ();
// The reader is certain not to need filtering.
parser.Filter_Input (false);
// Any of the Set_Reader methods may be used.
parser.Set_Reader (reader, read_limit);
</PRE>
<P>
	@param	reader	The Reader source for Parser input. If null,
		then there is no PVL source and thus nothing to be parsed.
	@return	This Parser.
	@throws	PVL_Exception	From <CODE>Set_Reader</CODE>. A
		<CODE>FILE_IO</CODE> form is thrown if there is an IOException
		from <CODE>Filter_Input</CODE>.
	@see	String_Buffer_Reader#Set_Reader(Reader)
	@see	String_Buffer_Reader#Filter_Input()
	@see	String_Buffer_Reader#Read_Limit(long)
*/
public Parser Set_Reader
	(
	Reader		reader,
	long		read_limit
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("Parser.Set_Reader");

/*	Set the Reader before the read limit:
	the read limit will not be changed if there is no Reader.
*/
Set_Reader (reader);
Read_Limit (read_limit);

if (reader != null)
	{
	try
		{	//	Possible Exception from the String_Buffer_Reader.
		if (Filter_Input ())
			{
			if ((DEBUG & DEBUG_SETUP) != 0)
				System.out.println ("Parser.Set_Reader: Input_Is_Filtered");
			Warning
				(
				PVL_Exception.SIZED_RECORDS,
				"Input filtering will be applied.",
				0
				);
			}
		}
	catch (IOException exception)
		{
		//	From the String_Buffer_Reader.
		throw new PVL_Exception
			(
			ID, PVL_Exception.FILE_IO,
			"While testing for input filtering during Set_Reader.\n"
			+ ((exception.getMessage () == null) ?
				"" : "\n" + exception.getMessage ())
			);
		}
	}
return this;
}

/**	Sets the Reader where the Parser will obtain characters by
	constructing an InputStreamReader from the specified InputStream
	and wrapping this in a BufferedReader, the same size as the
	<CODE>Size_Increment</CODE> of the String_Buffer_Reader, for
	efficiency.

	The Parser's {@link #CHARACTER_ENCODING
	<CODE>CHARACTER_ENCODING</CODE>} is used.
<P>
	@param	input_stream	The InputStream source for Parser input. If
		null then there is nothing to read.
	@return	This Parser.
	@throws	PVL_Exception	From <CODE>Set_Reader</CODE>. A
		<CODE>FILE_IO</CODE> form is thrown if there is an
		UnsupportedEncodingException when constructing the
		InputStreamReader.
	@see	String_Buffer_Reader#Size_Increment(int)
*/
public Parser Set_Reader
	(
	InputStream	input_stream,
	long		read_limit
	)
	throws PVL_Exception
{
if (input_stream == null)
	Set_Reader ((Reader)null, read_limit);
else
	{
	try
		{
		Set_Reader
			(
			new BufferedReader
				(
				new InputStreamReader
					(
					input_stream,
					CHARACTER_ENCODING
					),
				Size_Increment ()
				),
			read_limit
			);
		}
	catch (UnsupportedEncodingException exception)
		{
		throw new PVL_Exception
			(
			ID, PVL_Exception.FILE_IO,
			"Unable to create an InputStreamReader with the \""
				+ CHARACTER_ENCODING + "\" character encoding."
			+ ((exception.getMessage () == null) ?
				"" : "\n" + exception.getMessage ())
			);
		}
	}
return this;
}

/**	Sets the Reader where the Parser will obtain characters by
	constructing a FileInputStream from the specified File and passing
	this to the {@link #Set_Reader(InputStream, long) <CODE>Set_Reader
	(InputStream, long)</CODE>} method.
<P>
	@param	file	The File source for Parser input. If null, there
		is nothing to read.
	@return	This Parser.
	@throws	PVL_Exception	A <CODE>FILE_IO</CODE> error condition will
		occur if the File refers to a directory or a file for which
		read permission is not provided. Any other IOException that
		occurs while constructing the FileInputStream will also be
		converted into a <CODE>PVL_Exception.FILE_IO</CODE> exception.
	@see	#Set_Reader(InputStream, long)
*/
public Parser Set_Reader
	(
	File		file,
	long		read_limit
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		(">>> Parser.Set_Reader (File=" + file
		+ ", read_limit=" + read_limit + ") <<<");
if (file == null)
	Set_Reader ((Reader)null, read_limit);
else
	{
	if (file.isDirectory ())
		throw new PVL_Exception
			(
			ID, PVL_Exception.FILE_IO,
			"The pathname is a directory - "
				+ file.getAbsolutePath ()
			);
	if (! file.canRead ())
		throw new PVL_Exception
			(
			ID, PVL_Exception.FILE_IO,
			"Can't read the file - "
				+ file.getAbsolutePath ()
			);
	//	Create an InputStream from the File.
	try {Set_Reader (new FileInputStream (file), read_limit);}
	catch (Exception exception)
		{
		throw new PVL_Exception
			(
			ID, PVL_Exception.FILE_IO,
			"Can't access the file - "
				+ file.getAbsolutePath ()
			+ ((exception.getMessage () == null) ?
				"" : "\n" + exception.getMessage ())
			);
		}
	}
return this;
}


/*------------------------------------------------------------------------------
	Modes:
*/
/**	Enables or disables strict PVL syntax rules in the Parser.
<P>
	Normally the Parser is tolerant. However, the default is controlled by
	the {@link #Strict_Default} value.
<P>
	<B>N.B.</B>: Enabling strict syntax rules will prevent treating
	{@link #All_Values_Strings(boolean) all Values as Strings}.
<P>
	@param	strict	true if strict rules are applied; false otherwise.
	@return	This Parser.
*/
public Parser Strict
	(
	boolean		strict
	)
{
Strict = strict;
return this;
}

/**	Tests if the Parser will enforce strict PVL syntax rules.
<P>
	@return	true if the Parser will enforce strict syntax rules;
		false otherwise.
*/
public boolean Strict ()
{return Strict;}


/**	Enable or disable verbatim quoted strings.
<P>
	With format control (<CODE>Verbatim_Strings</CODE> disabled)
	multi-line quoted strings in PVL statements have white space
	surrounding the line breaks compressed to a single space character -
	except when <CODE>String_Continuation</CODE> is enabled and the last
	non-white space character on the line is a dash ("-"), in which case
	no space is included. This is because output formatting is expected
	to be controlled by embedded format characters which are processed by
	the Write method:
<P>
	<BLOCKQUOTE>
	\n - line break.<BR>
	\t - horizontal tab.<BR>
	\f - form feed (page break).<BR>
	\\ - backslash character.<BR>
	\v - verbatim (no formatting) till the next \v.<BR>
	</BLOCKQUOTE>
<P>
	Without format control (<CODE>Verbatim_Strings</CODE> enabled) all
	STRING Values are taken as-is.
<P>
	By default verbatim strings are disabled. However, the default is
	controlled by the {@link #Verbatim_Strings_Default} value.
<P>
	@param	verbatim	true if quoted strings in PVL statements are to
		be taken verbatim, without format control.
	@return	This Parser.
	@see	#Get_Parameter()
	@see	#Get_Quoted_String()
	@see	#String_Continuation(boolean)
	@see	#Get_Comments()
	@see	#Get_Datum(Value)
	@see	#Get_Units()
*/
public Parser Verbatim_Strings
	(
	boolean		verbatim
	)
{
Verbatim_Strings = verbatim;
return this;
}

/**	Tests if the Parser will handle quoted strings verbatim.
<P>
	@return	true if quoted strings are taken verbatim; false otherwise.
	@see	#Verbatim_Strings(boolean)
*/
public boolean Verbatim_Strings ()
{return Verbatim_Strings;}


/**	Enable or disable string continuation.
<P>
	When string continuation is enabled (the default) and {@link
	#Verbatim_Strings(boolean) verbatim strings} is disabled (the
	default) the occurrance in a quoted string of a {@link
	#STRING_CONTINUATION_DELIMITER} as the last character before the new
	line sequence causes the string continuation delimiter and all
	characters up to the next non-whitespace character to be removed from
	the string; i.e. the string continues on the next line after any
	whitespace.
<P>
	By default string continuation is enabled. However, the default is
	controlled by the {@link #String_Continuation_Default} value.
<P>
	@param	continuation	true if string continuation is to be enabled;
		false otherwise.
	@return	This Parser.
	@see	#Get_Quoted_String()
	@see	#Verbatim_Strings(boolean)
*/
public Parser String_Continuation
	(
	boolean		continuation
	)
{
String_Continuation = continuation;
return this;
}

/**	Tests if the Parser will recognize multi-line string continuation.
<P>
	@return	true if quoted string conintuation will be recognized; false
		otherwise.
	@see	#String_Continuation(boolean)
*/
public boolean String_Continuation ()
{return String_Continuation;}


/**	Enable or disable recognition of crosshatch comments.

	A Crosshatch comment begins with the {@link #CROSSHATCH
	<CODE>CROSSHATCH</CODE>} character and extends to the end of the
	current line (marked by a {@link #LINE_BREAK
	<CODE>LINE_BREAK</CODE>} character). Crosshatch comments are never
	recognized in {@link #Strict(boolean) Strict} mode.
<P>
	<B>Note</B>: Crosshatch comments are not recognized in the PVL
	syntax specification. Because of their common use in configuration
	files, this special extension is provided to accommodate such
	applications. Be default crosshatch comments are not recognized.
<P>
	By default crosshatch comments are enabled. However, the default is
	controlled by the {@link #Crosshatch_Comments_Default} value.
<P>
	@param	crosshatch_comments	true if crosshatch comments are to
		be recognized; false otherwise.
	@return	This Parser.
*/
public Parser Crosshatch_Comments
	(
	boolean		crosshatch_comments
	)
{
Crosshatch_Comments = crosshatch_comments;
return this;
}

/**	Tests if crosshatch comments will be recognized.
<P>
	Regardless of this test, crosshatch comments are never recognized
	in {@link #Strict(boolean) Strict} mode.
<P>
	@return	true if crosshatch comments will be recognized; false otherwise.
	@see	#Crosshatch_Comments(boolean)
*/
public boolean Crosshatch_Comments ()
{return Crosshatch_Comments;}


/**	Enable or disable treating all values as strings.
<P>
	When enabled, and {@link #Strict(boolean) strict} syntax rules are
	not enabled, all PVL parameter values will be parsed as {@link
	Value#STRING Strings}; no other Value {@link Value#Type() Type} will
	be generated. This may be useful in cases where, for example, a
	{@link Value#NUMERIC numeric} Type Value would be generated from a
	PVL value representation that can not be converted to a binary value
	with sufficient precision, or where unquoted PVL value representations
	that would otherwise be expected to be {@link Value#IDENTIFIER
	identifiers} happen to contain all numeric digit characters.
<P>
	By default treating all values as strings is disabled. However the
	default is controlled by the {@link #All_Values_Strings_Default} value.
<P>
	@param	all_values_strings	true if all Values are to be generated
		as {@link Value#STRING} Types; false otherwise.
	@return	This Parser.
*/
public Parser All_Values_Strings
	(
	boolean	all_values_strings
	)
{
All_Values_Strings = all_values_strings;
return this;
}

/**	Tests if the Parser will treat all values as strings.
<P>
	Regardless of this test, treating all values as strings will not
	be enforced if {@link #Strict(boolean) Strict} mode is enabled.
<P>
	@return	true if the Parser will treat all values as strings; false
		otherwise.
	@see	#All_Values_Strings(boolean)
*/
public boolean All_Values_Strings ()
{return All_Values_Strings;}


/**	Gets the current warning status.
<P>
	When conditions are encountered that are unusual enough to warrant
	attention, but not an error condition that would prevent successful
	processing which would cause an exception to be thrown, a warning
	condition is registered. The warning is in the form of a
	PVL_Exception that was not thrown. The current warning status is
	either the {@link #First_Warning(boolean)
	<CODE>First_Warning</CODE>} or the {@link #Last_Warning(boolean)
	<CODE>Last_Warning</CODE>} since a {@link #Reset_Warning()
	<CODE>Reset_Warning</CODE>}.
<P>
	@return	The current warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#First_Warning(boolean)
	@see	#Last_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception Warning ()
{
if (Use_First_Warning)
	return First_Warning;
return Last_Warning;
}

/**	Clears any warning status so that the <CODE>Warning</CODE> method
	will return null until the next warning condition occurs.

	@return	This Parser.
	@see	#Warning()
*/
public Parser Reset_Warning ()
{
First_Warning = null;
Last_Warning = null;
return this;
}

/**	Enables or disables returning the first warning that occurs as the
	current warning status.

	The first warning is one that occurs when the current warning
	status is null.
<P>
	@param	first	true to enable returning the first warning status;
		false to return the last warning that occurred as the current
		warning status.
	@return	This Parser.
	@see	#Warning()
	@see	#First_Warning()
	@see	#Reset_Warning()
*/
public Parser First_Warning
	(
	boolean		first
	)
{
Use_First_Warning = first;
return this;
}

/**	Returns the first warning since the last <CODE>Reset_Warning</CODE>.
<P>
	@return	The first warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#Warning()
	@see	#First_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception First_Warning ()
{return First_Warning;}

/**	Enables or disables returning the last warning that occurs as the
	current warning status.

	The last warning is the most recent one regarless of any previous
	warning conditions that may have occured without an intervening
	<CODE>Reset_Warning</CODE>.
<P>
	@param	last	true to enable returning the last warning status;
		false to return the first warning condition that occurred as
		the current warning status.
	@return	This Parser.
	@see	#Warning()
	@see	#Last_Warning()
	@see	#Reset_Warning()
*/
public Parser Last_Warning
	(
	boolean		last
	)
{
Use_First_Warning = ! last;
return this;
}

/**	Returns the last warning since a <CODE>Reset_Warning</CODE>.
<P>
	@return	The last warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#Warning()
	@see	#Last_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception Last_Warning ()
{return Last_Warning;}


/*==============================================================================
	PVL parser:
*/
/*------------------------------------------------------------------------------
	Parameter
*/
/**	Gets as many Parameters as possible.
<P>
	When the source of PVL statements is a Reader, then an aggregate
	Parameter named {@link #CONTAINER_NAME <CODE>CONTAINER_NAME</CODE>}
	will be provided to contain all the Parameters found (zero or
	more). If this Parser was created with a String as the source of
	PVL statements, then a container Parameter will only be provided if
	more than one Paramter is found: the single Parameter found, or an
	empty <CODE>UNKNOWN</CODE> Parameter if nothing is found.
<P>
	@return	A Parameter containing everything found from the input source.
	@throws	PVL_Exception	If an unrecoverable problem occurred while
		parsing the input source.
	@see	#Add_To(Parameter)
	@see	String_Buffer_Reader#String_Source()
*/
public Parameter Get ()
	throws PVL_Exception
{
if ((DEBUG & DEBUG_AGGREGATE) != 0)
	System.out.println (">>> Parser Get");
Parameter
	The_Parameter = new Parameter (CONTAINER_NAME);

Add_To (The_Parameter);

if (String_Source ())
	{
	/*
		Parameters read from a string only need a new aggregate to
		contain them when the string contains multiple, ungrouped
		parameters to be returned as a group. Otherwise the single
		parameter in the string representation is itself returned. If
		the string does not contain any recognizable parameters an
		unnamed UNKNOWN parameter is returned.

		Parameters read from a file are guaranteed to be returned in a
		new aggregate named CONTAINER_NAME.
	*/
	if ((DEBUG & DEBUG_AGGREGATE) != 0)
		System.out.println ("Parser Get: String_Reader");
	if (The_Parameter.Is_Empty ())
		{
		if ((DEBUG & DEBUG_AGGREGATE) != 0)
			System.out.println ("Parser Get: Empty");
		The_Parameter.Name ("").set_classification (Parameter.UNKNOWN);
		}
	else if (The_Parameter.List_Size () == 1)
		{
		//	Just return the single parameter.
		if ((DEBUG & DEBUG_AGGREGATE) != 0)
			System.out.println ("Parser Get: One parameter");
		The_Parameter = The_Parameter.Remove (0);
		}
	}
if ((DEBUG & DEBUG_AGGREGATE) != 0) System.out.println ("<<< Parser Get");
return The_Parameter;
}


/*------------------------------------------------------------------------------
*/
/**	Adds to a Parameter all Parameters found from the input source.

	Any Paramter added that is itself an <CODE>AGGREGATE</CODE> classification
	will recursively invoke this method on the new aggregate Parameter.
<P>
	While the source of PVL statements is not empty and {@link
	#Get_Parameter() <CODE>Get_Parameter</CODE>} can assemble a
	Parameter from the source, each new Parameter is {@link
	Parameter#Add(Parameter) <CODE>Add</CODE>}ed to the specified
	Parameter's aggregate list. <B>Note</B>: Parameters having
	<CODE>END</CODE> classifications are not added, instead they stop
	the addition of Parameters to the current aggregate list; additions
	will continue with the parent of the current aggregate on returning
	from recursive method invocations. However, if an
	<CODE>END_PVL</CODE> Parameter is encountered no more Parameters
	will be added regardless of the recursion level.
<P>
	@param	The_Aggregate	The aggregate Parameter to which to add
		new Parameters from the input source.
	@return	The_Aggregate Parameter.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>The_Aggregate Parameter is actually an
			<CODE>ASSIGNMENT</CODE> with a non-null Value.
		<DT><CODE>AGGREGATE_CLOSURE_MISMATCH</CODE>
			<DD>In <CODE>Strict</CODE> mode, when an
			<CODE>END_AGGREGATE</CODE> Parameter does not match the
			specific classification of the Parameter containing the
			aggregate list. This is only a <CODE>Warning</CODE> when
			<CODE>Strict</CODE> mode is not enabled.
		<DT>Others are possible during parsing.
		</DL>
	@see	Parameter#AGGREGATE
	@see	String_Buffer_Reader#Is_Empty()
	@see	#Get_Parameter()
	@see	Parameter#Add(Parameter)
	@see	Parameter#END
	@see	Parameter#END_AGGREGATE
	@see	Parameter#END_PVL
*/
public Parameter Add_To
	(
	Parameter	The_Aggregate
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_AGGREGATE) != 0)
	System.out.println (">>> Parser Add_To");
add_to (The_Aggregate);
if ((DEBUG & DEBUG_AGGREGATE) != 0)
	System.out.println ("<<< Parser Add_To");
return The_Aggregate;
}

private boolean add_to
	(
	Parameter	The_Aggregate
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_AGGREGATE) != 0)
	System.out.println
		(">>> add_to: " + The_Aggregate.Name ()
			+ " Next_Location = " + Next_Location () + '\n'
		+ The_Aggregate.Description ());
if (Is_Empty ())
	{
	if ((DEBUG & DEBUG_AGGREGATE) != 0)
		System.out.println
			("add_to: Is_Empty\n"
			+"<<< add_to");
	return false;
	}

//	Ensure the correct parameter classification.
if (The_Aggregate.Is_Assignment () &&
	The_Aggregate.Value () != null)
	throw new PVL_Exception
		(
		ID, PVL_Exception.BAD_ARGUMENT,
		"Can't add to the ASSIGNMENT parameter \""
			+ The_Aggregate.Name () + "\"",
		Next_Location ()
		);
if (! The_Aggregate.Is_Aggregate ())
	The_Aggregate.set_classification (Parameter.GROUP);

//	Collect all the parameters:
boolean
	continue_parsing = true;
while (! Is_Empty ())
	{
	//	Get the next parameter.
	Parameter
		The_Parameter;
	if ((The_Parameter = Get_Parameter ()) == null)
		{
		//	Failed to get a parameter.
		continue_parsing = false;	//	Stop parsing.
		break;
		}
	if ((DEBUG & DEBUG_AGGREGATE) != 0)
		System.out.println
			("add_to: Got " + The_Parameter.Classification_Name ()
				+ " named " + The_Parameter.Name ());

	/*--------------------------------------------------------------------------
		Check for END parameters:
	*/
	if (The_Parameter.Is_End ())
		{
		if (The_Parameter.Is_End_Aggregate ())
			{
			if ((DEBUG & DEBUG_AGGREGATE) != 0)
				System.out.println ("add_to: Is_End_Aggregate");
			if (The_Parameter.Classification () != 
				(The_Aggregate.Classification () | Parameter.END))
				{
				/*	The begin aggregate parameter doesn't match
					the end aggregate parameter.
				*/
				Warning
					(
					PVL_Exception.AGGREGATE_CLOSURE_MISMATCH,
					"For the " + The_Aggregate.Classification_Name ()
						+ " aggregate named \"" + The_Aggregate.Name () + "\"\n"
					+ "  with an " + The_Parameter.Classification_Name ()
						+ " parameter.",
					Next_Location ()
					);
				if (Strict)
					throw Last_Warning;
				}
			}

		if (The_Parameter.Is_End_PVL ())
			{
			//	END input here.
			if ((DEBUG & DEBUG_AGGREGATE) != 0)
				System.out.println
					("add_to: Is_End_PVL");
			continue_parsing = false;
			}
		break;
		}

	//	Add the parameter to the aggregate's list.
	if ((DEBUG & DEBUG_AGGREGATE) != 0)
		System.out.println
			("add_to: Add to " + The_Aggregate.Name () + " -\n"
			+ The_Parameter.Description ());
	The_Aggregate.Add (The_Parameter);

	if (The_Parameter.Is_Aggregate () &&
		//	Recursively add the parameters for this aggregate parameter.
		! (continue_parsing = add_to (The_Parameter)))
		{
		//	Reached the end of input.
		if ((DEBUG & DEBUG_AGGREGATE) != 0)
			System.out.println
				("add_to: End of input -\n"
				+ The_Aggregate.Description ());
		break;
		}
	}

if ((DEBUG & DEBUG_AGGREGATE) != 0)
	System.out.println
		("add_to: The_Aggregate -\n"
		+ The_Aggregate.Description () + "\n"
		+"    continue_parsing - " + continue_parsing + '\n'
		+"<<< add_to: Next_Location = " + Next_Location ());
return continue_parsing;
}


/*------------------------------------------------------------------------------
*/
/**	Gets a Parameter from the source of PVL statements.
<P>
	First, {@link #Get_Comments <CODE>Get_Comments</CODE>} is used to
	to collect any and all comment sequences preceeding the parameter
	proper as the Parameter's <CODE>Comments</CODE>.
<P>
	The next sequence of non-{@link #WHITESPACE
	<CODE>WHITESPACE</CODE>} characters is taken to be the Parameter's
	<CODE>Name</CODE>. If the sequence is quoted (i.e. starts with a
	{@link #TEXT_DELIMITER <CODE>TEXT_DELIMITER</CODE>} or {@link
	#SYMBOL_DELIMITER <CODE>SYMBOL_DELIMITER</CODE>}), then the name is
	all characters within the quotes. Otherwise it is all characters up
	to but not including the next {@link #PARAMETER_NAME_DELIMITER
	<CODE>PARAMETER_NAME_DELIMITER</CODE>}, {@link
	#STATEMENT_END_DELIMITER <CODE>STATEMENT_END_DELIMITER</CODE>},
	whitespace character, or comment sequence. The name is checked for
	any  {@link #Bad_Character(String) <CODE>Bad_Character</CODE>} and
	a <CODE>Warning</CODE> will be registered (exception thrown in
	<CODE>Strict</CODE> mode) if one is found. If
	<CODE>Verbatim_Strings</CODE> is not enabled, then all character
	escape sequences in the name are replaced with their corresponding
	special characters.
<P>
	If the name is associated with a {@link
	#Special_Classification(String)
	<CODE>Special_Classification</CODE>}, then the Parameter is given
	that classification; otherwise it is tentatively classified as a
	<CODE>TOKEN</CODE>.
<P>
	If, after any whitespace or comments following the Parameter name,
	a {@link #PARAMETER_NAME_DELIMITER
	<CODE>PARAMETER_NAME_DELIMITER</CODE>} is found, then the {@link
	#Get_Value() <CODE>Get_Value</CODE>} method is used to obtain the
	expected Parameter Value. If the Parameter had been given the
	<CODE>TOKEN</CODE> classification, then the classification is
	upgraded to <CODE>ASSIGNMENT</CODE>. If, however, the Parameter had
	been given an <CODE>AGGREGATE</CODE> classification as a result of
	its special name, and the Value obtained is a <CODE>STRING</CODE>
	type, then the Parameter's name is changed to the Value String; if,
	in this case, the Value found is not a <CODE>STRING</CODE> type a
	<CODE>Warning</CODE> will be registered (exception thrown in
	<CODE>Strict</CODE> mode).
<P>
	Having assembled a valid Parameter, the <CODE>Next_Location</CODE>
	in the input stream is moved forward past any whitespace or
	<CODE>STATEMENT_END_DELIMITER</CODE>.
<P>
	@return	The next Parameter assembled from the input stream, or null
		if no Parameter can be assembled because the input stream is
		empty.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>ILLEGAL_SYNTAX</CODE>
			<DD><CODE>Strict</CODE> mode is enabled and the Parameter
			name was quoted.
		<DT><CODE>RESERVED_CHARACTER</CODE>
			<DD>The Parameter name contained a bad character and
			<CODE>Strict</CODE> mode is enabled.
		<DT><CODE>GROUP_VALUE</CODE>
			<DD>The Value obtained for an <CODE>AGGREGATE</CODE> Parameter
			is not a <CODE>STRING</CODE> type.
		<DT><CODE>FILE_IO</CODE>
			<DD>If an IOException occurred in the String_Buffer_Reader.
		</DL>
	@see	#Get_Comments()
	@see	Parameter#Comments()
	@see	Parameter#Name()
	@see	#translate_from_escape_sequences(String_Buffer)
	@see	#Special_Classification(String)
	@see	#Get_Value()
*/
public Parameter Get_Parameter ()
	throws PVL_Exception
{
if (Is_Empty ())
	return null;
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		(">>> Get_Parameter: Next_Location = " + Next_Location ());

Parameter
	The_Parameter = new Parameter ();
long
	delimiter,
	location;
int
	index;

/*..............................................................................
	Collect any leading comments before the parameter name:
*/
The_Parameter.Comments (Get_Comments ());

try {	//	Possible IOException from the String_Buffer_Reader.

//	Ignore any statement end delimiter.
if (Is_End (delimiter = Next_Location
		(Skip_Over (Next_Location (), WHITESPACE + STATEMENT_END_DELIMITER))))
	return null;

/*..............................................................................
	Get the parameter name:
*/
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("Get_Parameter: Getting the parameter name starting at " + delimiter);
if (Char_At (delimiter) == TEXT_DELIMITER ||
	Char_At (delimiter) == SYMBOL_DELIMITER)
	{
	//	It's a quoted string.
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Parameter names should not be quoted.",
		delimiter
		);
	if (Strict)
		throw Last_Warning;
	The_Parameter.Name (Get_Quoted_String ());
	}
else
	{
	//	Find the parameter name's trailing delimiter.
	if ((delimiter = Skip_Until
			(Next_Location (), PARAMETER_NAME_DELIMITERS)) < 0)
		delimiter = End_Location ();
	if ((DEBUG & DEBUG_PARAMETER) != 0)
		System.out.println
			("Get_Parameter: Getting the parameter name ending at " + delimiter);

	//	Get the parameter name substring.
	The_Parameter.Name (Substring (Next_Location (), delimiter));
	if ((index = The_Parameter.Name ().indexOf (COMMENT_START_DELIMITERS)) > 0)
		{
		//	Only take the part up to the trailing comment.
		The_Parameter.Name (The_Parameter.Name ().substring (0, index));
		delimiter = Next_Location () + index;
		}

	//	Check for reserved characters in the parameter name.
	if ((index = Bad_Character (The_Parameter.Name ())) >= 0)
		{
		Warning
			(
			PVL_Exception.RESERVED_CHARACTER,
			"At character " + index
				+ " of the parameter name \"" + The_Parameter.Name () + "\"",
			delimiter + index
			);
		if (Strict)
			throw Last_Warning;
		}
	Next_Location (delimiter);
	}
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("Get_Parameter: Name = " + The_Parameter.Name ());
if (! Verbatim_Strings)
	//	Translate escape sequences to special characters.
	The_Parameter.Name
		(translate_from_escape_sequences
			(new String_Buffer (The_Parameter.Name ())).toString ());

/*..............................................................................
	Set the initial parameter classification:
*/
//	Check for special names for special classifications.
int	classification;
if ((classification = Special_Classification (The_Parameter.Name ())) > 0)
	The_Parameter.set_classification (classification);
else
	The_Parameter.set_classification (Parameter.TOKEN);
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("Get_Parameter: Tentative classification = "
		+ The_Parameter.Classification_Name ());

/*..............................................................................
	Get the parameter value(s):
*/
/*	Find the parameter name delimiter character
	that separates the parameter name from the parameter values list.
*/
if (Is_End (delimiter = Next_Location
		(skip_whitespace_and_comments (Next_Location ()))))
	{
	if ((DEBUG & DEBUG_PARAMETER) != 0)
		System.out.println
			("<<< Get_Parameter: End at " + delimiter);
	return The_Parameter;
	}
if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("Get_Parameter: Expecting '" + PARAMETER_NAME_DELIMITER
		+ "', found '" + Char_At (delimiter) + "'");

if (Char_At (delimiter) == PARAMETER_NAME_DELIMITER)
	{
	// This is where the values string starts.
	if ((DEBUG & DEBUG_PARAMETER) != 0)
		System.out.println
			("Get_Parameter: PARAMETER_NAME_DELIMITER at " + delimiter);
	Next_Location (++delimiter);

	/*	Elevate the parameter classification to assignment
		unless it's already been identified with
		one of the special classifications.
	*/
	if (The_Parameter.Is_Token ())
		The_Parameter.set_classification (Parameter.ASSIGNMENT);

	if ((DEBUG & DEBUG_PARAMETER) != 0)
		System.out.println ("Get_Parameter: Getting the parameter values.");

	Value value = Get_Value ();
	if (value != null)
		{
		if ((DEBUG & DEBUG_PARAMETER) != 0)
			System.out.println ("Get_Parameter: Got a value.");
		if (The_Parameter.Is_Aggregate ())
			{
			//	Set the parameter name to the string value.
			The_Parameter.Name (value.String_Data ());
			if (! value.Is_String ())
				{
				//	The value is inappropriate for an aggregate.
				Warning
					(
					PVL_Exception.GROUP_VALUE,
					"Non-STRING value for the "
						+ The_Parameter.Classification_Name () + " parameter.",
					delimiter
					);
				if (Strict)
					throw Last_Warning;
				}
			}
		else
			The_Parameter.set_data (value);
		}
	}

//	Skip any trailing white space and statement end delimiters.
Next_Location (Skip_Over (Next_Location (),
	WHITESPACE + STATEMENT_END_DELIMITER));

}
catch (IOException exception)
	{
	//	From the String_Buffer_Reader.
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"In Get_Parameter.\n"
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}

if ((DEBUG & DEBUG_PARAMETER) != 0)
	System.out.println
		("<<< Get_Parameter: Next_Location = " + Next_Location ());
return The_Parameter;
}


/*------------------------------------------------------------------------------
*/
/**	Gets the next sequence of comments from the source of PVL statements
	as a single comment String.

	After skipping over any whitespace, the next characters must start
	a comment sequence or nothing (null) is returned.
<P>
	A PVL comment uses C-style conventions: It starts after the {@link
	#COMMENT_START_DELIMITERS <CODE>COMMENT_START_DELIMITERS</CODE>}
	and ends before the {@link #COMMENT_END_DELIMITERS
	<CODE>COMMENT_END_DELIMITERS</CODE>}. A comment without the closing
	<CODE>COMMENT_END_DELIMITERS</CODE> will result in a
	<CODE>MISSING_COMMENT_END</CODE> exception in <CODE>Strict</CODE>
	mode; otherwise a <CODE>Warning</CODE> is registered and the next
	line break, {@link #STATEMENT_END_DELIMITER
	<CODE>STATEMENT_END_DELIMITER</CODE>}, or the end of the input
	stream is taken as the end of the comment. <B>Note</B>: Though an
	effort is made to recover from encountering an unending comment,
	this will only be effective when no other normally closed comment
	occurs in the input stream (if a normally closed comment does occur
	after an unclosed comment, the latter will be taken as the end of
	the former), and in this case the input stream will have been read
	into memory until it is empty.
<P>
	Sequential comments, with nothing but white space intervening, are
	accumulated with a single new-line ('\n') chararacter separating
	them in the resulting String that is returned. In
	<CODE>Strict</CODE> mode comments that wrap across line breaks
	cause an exception. When <CODE>Verbatim_Strings</CODE> are not
	enabled whitespace is trimmed from the end of each comment (but not
	the beginning), and escape sequences are translated into their
	corresponding special characters.
<P>
	If any comments are found the <CODE>Next_Location</CODE> of the
	input stream is moved to the position immediately following the
	last comment.
<P>
	@return	A String containing the sequence of comments found, or null
		if no comment occurs before the next PVL item or the end of input.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>ILLEGAL_SYNTAX</CODE>
			<DD><CODE>Strict</CODE> mode is enabled and a comment
			continues on more than one line.
		<DT><CODE>MISSING_COMMENT_END</CODE>
			<DD>A comment does not have
			<CODE>COMMENT_END_DELIMITERS</CODE> and <CODE>Strict</CODE>
			mode is enabled. If <CODE>Strict</CODE> is not enabled and
			a line or statement delimiter can not be found, then the
			exception is thrown.
		<DT><CODE>FILE_IO</CODE>
			<DD>If an IOException occurred in the String_Buffer_Reader.
		</DL>
	@see	#Verbatim_Strings(boolean)
*/
public String Get_Comments ()
	throws PVL_Exception
{
if (Is_Empty ())
	return null;

String_Buffer
	comments = null;
long
	location = Next_Location (),
	comment_start,
	comment_end;
if ((DEBUG & DEBUG_COMMENTS) != 0)
	System.out.println (">>> Get_Comments: Next_Location = " + location);

try {	//	Possible IOException from the String_Buffer_Reader.

//	Accumulate all sequential comments.
while (true)
	{
	//	Find the beginning of the comment string.
	if (Is_End (location = skip_crosshatch_comments (location)) ||
		! Equals (location, COMMENT_START_DELIMITERS))
		break;
	comment_start = location + COMMENT_START_DELIMITERS.length ();
	if ((DEBUG & DEBUG_COMMENTS) != 0)
		System.out.println
			("Get_Comments: comment starts at " + comment_start);

	//	Find the end of the comment string.
	if ((comment_end = Location_Of (location, COMMENT_END_DELIMITERS)) < 0)
		{
		//	No comment end.
		Warning
			(
			PVL_Exception.MISSING_COMMENT_END,
			"For comment starting with \""
				+ Substring (comment_start,
					Math.min (comment_start + 20, End_Location ())) + "\" ...",
			location
			);
		if ((DEBUG & DEBUG_COMMENTS) != 0)
			System.out.println
				("Get_Comments:\n"
				+ Last_Warning.getMessage ());
		if (Strict)
			throw Last_Warning;

		//	Assume it ends at the end of the line.
		if ((comment_end = Skip_Until
				(location, LINE_DELIMITERS + STATEMENT_END_DELIMITER)) < 0)
			location = End_Location ();
		location = Skip_Over
			(comment_end, LINE_DELIMITERS + STATEMENT_END_DELIMITER);
		}
	else
		location = comment_end + COMMENT_END_DELIMITERS.length ();

	String_Buffer
		comment = new String_Buffer
			(Substring (comment_start, comment_end));
	if (Strict && comment.skip_until (0, LINE_BREAK) >= 0)
		throw new PVL_Exception
			(
			ID, PVL_Exception.ILLEGAL_SYNTAX,
			"Multi-line comments are not allowed in Strict mode.\n"
			+ "For comment starting with \""
				+ Substring (comment_start,
					Math.min (comment_start + 20, End_Location ())) + "\" ...",
			comment_start
			);
	//	Clean up the comment String. 
	if (! Verbatim_Strings)
		translate_from_escape_sequences
			(comment.trim_end ());	//	Leave whitespace at the beginning.

	//	Append the comment to the comments.
	if (comments == null)
		comments = comment;
	else
		comments.append ("\n" + comment.toString ());
	if ((DEBUG & DEBUG_COMMENTS) != 0)
		System.out.println
			("Get_Comments: length " + comments.length ()
			+ " - " + comments.toString ());
	}

Next_Location (location);

}
catch (IOException exception)
	{
	//	From the String_Buffer_Reader.
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"In Get_Comments.\n"
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}
if ((DEBUG & DEBUG_COMMENTS) != 0)
	System.out.println
		("<<< Get_Comments: Next_Location = " + Next_Location ());
return (comments == null) ? null : comments.toString ();
}


/*------------------------------------------------------------------------------
	Value
*/
/**	Gets a Value from the source of PVL statements.
<P>
	If the next character, after skipping any whitespace and comments,
	is a {@link #SET_START_DELIMITER <CODE>SET_START_DELIMITER</CODE>}
	or {@link #SEQUENCE_START_DELIMITER
	<CODE>SEQUENCE_START_DELIMITER</CODE>} the Value being assembled is
	typed as a <CODE>SET</CODE> or <CODE>SEQUENCE</CODE>, respectively,
	and the <CODE>Next_Location</CODE> in the input stream is moved
	over the character; otherwise the Value is tentatively typed as a
	<CODE>SET</CODE>, and the <CODE>Next_Location</CODE> is not moved.
<P>
	Now a cycle is entered to obtain as many sequential datum values as
	are available. The first step is to skip any whitespace and
	comments and test the character that is found. If the character is
	a {@link #SET_END_DELIMITER <CODE>SET_END_DELIMITER</CODE>} or
	{@link #SEQUENCE_END_DELIMITER <CODE>SEQUENCE_END_DELIMITER</CODE>},
	then the <CODE>Next_Location</CODE> in the input stream is moved
	over the character and <CODE>Get_Units</CODE> is used to set the
	<CODE>ARRAY</CODE> Value's units description before ending the
	datum cycle. If the character is a {@link #STATEMENT_END_DELIMITER
	<CODE>STATEMENT_END_DELIMITER</CODE>} the datum cycle ends. A
	reserved {@link #PARAMETER_NAME_DELIMITER
	<CODE>PARAMETER_NAME_DELIMITER</CODE>}, {@link
	#PARAMETER_VALUE_DELIMITER <CODE>PARAMETER_VALUE_DELIMITER</CODE>},
	{@link #UNITS_START_DELIMITER <CODE>UNITS_START_DELIMITER</CODE>},
	{@link #UNITS_END_DELIMITER <CODE>UNITS_END_DELIMITER</CODE>}, or
	{@link #NUMBER_BASE_DELIMITER <CODE>NUMBER_BASE_DELIMITER</CODE>}
	character here is an <CODE>ILLEGAL_SYNTAX</CODE> exception. For any
	other character the <CODE>Next_Location</CODE> is moved forward to
	its position as a possible datum. 
<P>
	When the character at this location is a {@link
	#SET_START_DELIMITER <CODE>SET_START_DELIMITER</CODE>} or {@link
	#SEQUENCE_START_DELIMITER <CODE>SEQUENCE_START_DELIMITER</CODE>}
	this method is called recursively to get a subarray as the datum.
	Otherwise the <CODE>Get_Datum</CODE> method is used to get a basic
	value followed by the <CODE>Get_Units</CODE> method to get any
	units description for the new datum. If  no datum is obtained the
	datum cycle ends, otherwise the new datum is added to the Vector of
	Value's being accumulated for this new Value.
<P>
	After skipping any whitespace and comments the next character is
	checked. A reserved {@link #PARAMETER_NAME_DELIMITER
	<CODE>PARAMETER_NAME_DELIMITER</CODE>}, {@link
	#UNITS_START_DELIMITER <CODE>UNITS_START_DELIMITER</CODE>}, {@link
	#UNITS_END_DELIMITER <CODE>UNITS_END_DELIMITER</CODE>}, or {@link
	#NUMBER_BASE_DELIMITER <CODE>NUMBER_BASE_DELIMITER</CODE>}
	character here is an <CODE>ILLEGAL_SYNTAX</CODE> exception. A
	{@link #SET_START_DELIMITER <CODE>SET_START_DELIMITER</CODE>} or
	{@link #SEQUENCE_START_DELIMITER
	<CODE>SEQUENCE_START_DELIMITER</CODE>} character will also generate
	an <CODE>ILLEGAL_SYNTAX</CODE> exception, but if
	<CODE>Strict</CODE> mode is not enabled this will just be
	registered as a <CODE>Warning</CODE> and the
	<CODE>Next_Location</CODE> will be moved to the character's
	position before continuing the datum cycle from the beginning. If
	the character is a {@link #PARAMETER_VALUE_DELIMITER
	<CODE>PARAMETER_VALUE_DELIMITER</CODE>} the
	<CODE>Next_Location</CODE> is moved over the character and the
	datum cycle returns to the beginning. For a {@link
	#SET_END_DELIMITER <CODE>SET_END_DELIMITER</CODE>} or {@link
	#SEQUENCE_END_DELIMITER <CODE>SEQUENCE_END_DELIMITER</CODE>}, if
	the character does not correspond to the delimiter that began the
	array an <CODE>ARRAY_CLOSURE_MISMATCH</CODE> <CODE>Warning</CODE>
	is registered (this exception is thrown in <CODE>Strict</CODE>
	mode). Then the <CODE>Next_Location</CODE> is moved over the
	character and the <CODE>Get_Units</CODE> method is used to set this
	Value's units description before ending the datum cycle. Any other
	character also causes the datum cycle to end.
<P>
	After the datum cycle has collected as many Values as possible, if
	this new Value was not begun with a {@link #SET_START_DELIMITER
	<CODE>SET_START_DELIMITER</CODE>} or {@link
	#SEQUENCE_START_DELIMITER <CODE>SEQUENCE_START_DELIMITER</CODE>}
	and the accumulated Values Vector containins less than two Values
	then the initial tentative <CODE>SET</CODE> type does not apply. In
	this case an empty accumlated Values Vector results in this new
	Value being an <CODE>UNNOWN</CODE> type (i.e. it is empty). When
	only one Value was collected it is the new Value that is returned.
	When two or more Values were collected the Vector containing them
	is set as the data of this <CODE>ARRAY</CODE> Value.
<P>
	@return	The next Value assembled from the input stream, or null if
		no Value can be assembled because the input stream is empty.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>ILLEGAL_SYNTAX</CODE>
			<DD>A misplaced reserved character was found.
		<DT><CODE>ARRAY_CLOSURE_MISMATCH</CODE>
			<DD>The delimiter character ending an array of Values
			does not correspond to the one that began the array.
		<DT><CODE>FILE_IO</CODE>
			<DD>If an IOException occurred in the String_Buffer_Reader.
		</DL>
	@see	Value#set_type(int)
	@see	Value#Units(String)
	@see	#Get_Datum(Value)
*/
public Value Get_Value ()
	throws PVL_Exception
{
if (Is_Empty ())
	return null;
if ((DEBUG & DEBUG_VALUE) != 0)
	System.out.println
		(">>> Get_Value: Next_Location = "+ Next_Location ());

Value
	//	The Value being built.
	The_Value = new Value ();
Vector
	//	The data array for this value.
	The_Array = new Vector ();
Value
	//	Each datum added to the array.
	datum;
long
	delimiter,
	location,
	array_start_location;
int
	start_type = 0,
	end_type = 0;

try {	//	Possible IOException from the String_Buffer_Reader.

if (Is_End (delimiter = Next_Location
		(skip_whitespace_and_comments (Next_Location ()))))
	{
	if ((DEBUG & DEBUG_VALUE) != 0)
		System.out.println
			("Get_Value: " + delimiter
				+ " Is_End\n<<< Get_Value: Next_Loction = " + delimiter);
	return null;
	}

/*..............................................................................
	Check for an initial array start character.
*/
array_start_location = delimiter;
if (Char_At (delimiter) == SET_START_DELIMITER)
	{
	if ((DEBUG & DEBUG_VALUE) != 0)
		System.out.println ("Get_Value: Set detected.");
	The_Value.set_type (start_type = Value.SET);
	}
else if (Char_At (delimiter) == SEQUENCE_START_DELIMITER)
	{
	if ((DEBUG & DEBUG_VALUE) != 0)
		System.out.println ("Get_Value: Sequence detected.");
	The_Value.set_type (start_type = Value.SEQUENCE);
	}
if (start_type != 0)
	Next_Location (++delimiter);
else
	{
	if ((DEBUG & DEBUG_VALUE) != 0)
		System.out.println ("Get_Value: Initial default is set.");
	The_Value.set_type (Value.SET);	//	Tentative default.
	}

/*..............................................................................
	Get as many values as possible.
*/
Get_Values:
while (true)
	{
	//	Find the values string.
	if (Is_End (delimiter = skip_whitespace_and_comments (Next_Location ())))
		break;

	//	Check for a valid values string:

	switch (Char_At (delimiter))
		{
		/*	End of array cases:

			Note: An empty array is valid.
		*/
		case SET_END_DELIMITER:
			end_type = Value.SET;
		case SEQUENCE_END_DELIMITER:
			if (end_type == 0)
				end_type = Value.SEQUENCE;
			if ((DEBUG & DEBUG_VALUE) != 0)
				System.out.println ("Get_Value: End of array at location "
					+ delimiter);
			Next_Location (++delimiter);

			//	Get any units string for this array.
			if ((DEBUG & DEBUG_VALUE) != 0)
				System.out.println ("Get_Value: Getting any units.");
			The_Value.Units (Get_Units ());

		case STATEMENT_END_DELIMITER:
			break Get_Values;

		//	Syntax error cases:
		case PARAMETER_NAME_DELIMITER:
		case PARAMETER_VALUE_DELIMITER:
		case UNITS_START_DELIMITER:
		case UNITS_END_DELIMITER:
		case NUMBER_BASE_DELIMITER:
			/*
				Note: It would be possible here to be VERY tolerant
				and just take these illegal characters as the first
				character of an identifier string, but this seems
				unlikely in any case.
			*/
			throw new PVL_Exception
				(
				ID, PVL_Exception.ILLEGAL_SYNTAX,
				"Expected a value, but found '" + Char_At (delimiter) + "'",
				delimiter
				);

		//	Possible value:
		default:
			Next_Location (delimiter);
			break;
		}

	//	Get the value:

	if (Char_At (delimiter) == SET_START_DELIMITER ||
		Char_At (delimiter) == SEQUENCE_START_DELIMITER)
		{
		//	The value is an array of values.
		if ((DEBUG & DEBUG_VALUE) != 0)
			System.out.println ("Get_Value: Starting a new array.");
		if ((datum = Get_Value ()) == null)
			break Get_Values;
		}
	else
		{
		//	Get the datum.
		if ((DEBUG & DEBUG_VALUE) != 0)
			System.out.println ("Get_Value: Getting a datum.");
		if ((datum = Get_Datum (new Value ())) == null)
			break Get_Values;

		//	Get any units string for this datum.
		if ((DEBUG & DEBUG_VALUE) != 0)
			System.out.println ("Get_Value: Getting any units.");
		datum.Units (Get_Units ());
		}
	//	Add the value to the array.
	if ((DEBUG & DEBUG_VALUE) != 0)
		System.out.println ("Get_Value: Adding the datum to the value array.");
	The_Array.add (datum);

	/*	Find the next word.

		Note: Leave the Next_Location before any comments in case they
		occur as lead-in to the next PVL statement (i.e. this may be
		the end of the current statement). Update the Next_Location
		when the new location is recognized as value syntax.
	*/
	if (Is_End (location = skip_whitespace_and_comments (Next_Location ())))
		break Get_Values;

	//	Check what comes next:

	switch (Char_At (location))
		{
		//	Another datum is expected:

		case PARAMETER_VALUE_DELIMITER:
			Next_Location (++location);
			break;

		case SET_START_DELIMITER:
		case SEQUENCE_START_DELIMITER:
			Warning
				(
				PVL_Exception.ILLEGAL_SYNTAX,
				"Expected another datum, but found '"
					+ Char_At (location) + "'",
				location
				);
			if (Strict)
				throw Last_Warning;
			Next_Location (location);
			break;

		//	End of array cases:
		case SET_END_DELIMITER:
			end_type = Value.SET;

		case SEQUENCE_END_DELIMITER:
			if (end_type == 0)
				end_type = Value.SEQUENCE;
			if ((DEBUG & DEBUG_VALUE) != 0)
				System.out.println
					("Get_Value: End of array at location " + location);
			if (start_type != end_type)
				{
				Warning
					(
					PVL_Exception.ARRAY_CLOSURE_MISMATCH,
					"For Value array starting at location "
						+ array_start_location
						+ " with \""
						+ Substring (array_start_location,
							Math.min (array_start_location + 20,
								End_Location ()))
						+ "\" ...",
					location
					);
				if (Strict)
					throw Last_Warning;
				}
			Next_Location (++location);

			//	Get any units string for this array.
			if ((DEBUG & DEBUG_VALUE) != 0)
				System.out.println
					("Get_Value: Getting any units.");
			The_Value.Units (Get_Units ());
			break Get_Values;

		//	Syntax error cases:
		case PARAMETER_NAME_DELIMITER:
		case UNITS_START_DELIMITER:
		case UNITS_END_DELIMITER:
		case NUMBER_BASE_DELIMITER:
			throw new PVL_Exception
				(
				ID, PVL_Exception.ILLEGAL_SYNTAX,
				"Expected another datum, but found '" + Char_At (location) + "'",
				location
				);

		default:
			//	Not a recognized value syntax.
			break Get_Values;
		}
	}

}
catch (IOException exception)
	{
	//	From the String_Buffer_Reader.
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"In Get_Value.\n"
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}

/*..............................................................................
	Check the results:
*/
if (start_type == 0 &&
	end_type == 0 &&
	The_Array.size () <= 1)
	{
	if (The_Array.size () == 0)
		//	Didn't get anything.
		The_Value.set_type (Value.UNKNOWN);
	else
		{
		/*
			The_Array contains a single, non-array (parsed from an
			undecorated string) value. Make this The_Value's datum.

			Note that this case can only occur at the root level for
			a PVL statement with a single non-array value.
		*/
		if ((DEBUG & DEBUG_VALUE) != 0)
			System.out.println
				("Get_Value: Single value returned.");
		The_Value = (Value)The_Array.firstElement ();
		}
	}
else
	The_Value.set_data (The_Array);

if ((DEBUG & DEBUG_VALUE) != 0)
	System.out.println
		("<<< Get_Value: Next_Location = " + Next_Location ());
return The_Value;
}


/*------------------------------------------------------------------------------
*/
/**	Gets a datum from the source of PVL statements.
<P>
	After skipping any whitespace or comments, the next character is
	checked to determine the type of datum to parse. For a {@link
	#STATEMENT_END_DELIMITER <CODE>STATEMENT_END_DELIMITER</CODE>}
	nothing happens. A reserved  {@link #PARAMETER_NAME_DELIMITER
	<CODE>PARAMETER_NAME_DELIMITER</CODE>}, {@link
	#PARAMETER_VALUE_DELIMITER <CODE>PARAMETER_VALUE_DELIMITER</CODE>},
	{@link #SET_START_DELIMITER <CODE>SET_START_DELIMITER</CODE>},
	{@link #SET_END_DELIMITER <CODE>SET_END_DELIMITER</CODE>}, {@link
	#SEQUENCE_START_DELIMITER <CODE>SEQUENCE_START_DELIMITER</CODE>},
	{@link #SEQUENCE_END_DELIMITER
	<CODE>SEQUENCE_END_DELIMITER</CODE>}, {@link #UNITS_START_DELIMITER
	<CODE>UNITS_START_DELIMITER</CODE>}, {@link #UNITS_END_DELIMITER
	<CODE>UNITS_END_DELIMITER</CODE>}, or {@link #NUMBER_BASE_DELIMITER
	<CODE>NUMBER_BASE_DELIMITER</CODE>} character here is an
	<CODE>ILLEGAL_SYNTAX</CODE> exception. For a {@link #TEXT_DELIMITER
	<CODE>TEXT_DELIMITER</CODE>} or {@link #SYMBOL_DELIMITER
	<CODE>SYMBOL_DELIMITER</CODE>} the <CODE>Get_Quoted_String</CODE>
	method is used to set the datum of the Value and its type is set to
	<CODE>TEXT</CODE> or <CODE>SYMBOL</CODE> respectively.
<P>
	For any ordinary character the substring up to, but not including,
	the next {@link #WHITESPACE <CODE>WHITESPACE</CODE>}, {@link
	#PARAMETER_VALUE_DELIMITER <CODE>PARAMETER_VALUE_DELIMITER</CODE>},
	{@link #STATEMENT_END_DELIMITER <CODE>STATEMENT_END_DELIMITER</CODE>},
	{@link #COMMENT_START_DELIMITERS <CODE>COMMENT_START_DELIMITERS</CODE>},
	any of the SET/SEQUENCE/UNITS START/END delimiters, or the end of
	the input stream is used for parsing a datum. If
	<CODE>Verbatim_Strings</CODE> is not enabled, then all escape
	sequences in the substring are converted to their special character
	equivalents.
<P>
	The datum substring is first assumed to represent a number. If the
	substring contains a {@link #NUMBER_BASE_DELIMITER
	<CODE>NUMBER_BASE_DELIMITER</CODE>} ('#') the number is presumed to
	be in radix base notation:
<P>
	<BLOCKQUOTE>
	[<B><I>sign</I></B>]<B><I>base</I>#<I>value</I>#</B>
	</BLOCKQUOTE>
<P>
	In this case the initial base integer is obtained using the
	<CODE>Integer.parseInt</CODE> method and becomes the Value's
	<CODE>Base</CODE>, and the value number is obtained using the
	<CODE>Long.parseLong</CODE> method with the base argument specified
	and becomes the Value's datum. The sign is applied to the value
	number. The Value becomes type <CODE>INTEGER</CODE>. Without the
	<CODE>NUMBER_BASE_DELIMITER</CODE> the datum substring is taken to
	be in decimal notation. If this number conversion fails, then the
	<CODE>Double.valueOf</CODE> method is tried on the datum substring
	to produce a type <CODE>REAL</CODE> Value.
<P>
	<B>N.B.</B>: If treating {@link #All_Values_Strings(boolean) all values
	as strings} has been enabled, and {@link #Strict(boolean) Strict} mode
	is not enabled, then the value is not assumed to be a number; it is
	always taken as string.
<P>
	If parsing the datum substring as a number fails, then the Value is
	a <CODE>STRING</CODE> and its datum is the substring. If the
	substring contains one of the {@link #DATE_TIME_DELIMITERS
	<CODE>DATE_TIME_DELIMITERS</CODE>} the Value is given the
	<CODE>DATE_TIME</CODE> type. Otherwise it is the
	<CODE>IDENTIFIER</CODE> type. The datum substring is also checked
	for a <CODE>Bad_Character</CODE> with a <CODE>Warning</CODE> being
	registered (exception thrown in <CODE>Strict</CODE> mode) if one is
	found.
<P>
	Once a datum has been given to the Value the
	<CODE>Next_Location</CODE> in the input stream is moved to the
	position immediately following the datum substring.
<P>
	@param	The_Value	The Value to which the next datum is to be applied.
	@return	The_Value, or null if the input stream is empty.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>ILLEGAL_SYNTAX</CODE>
			<DD>A misplaced reserved character was found.
		<DT><CODE>RESERVED_CHARACTER</CODE>
			<DD>A <CODE>STRING</CODE> Value contains a reserved
			character.
		<DT><CODE>FILE_IO</CODE>
			<DD>If an IOException occurred in the String_Buffer_Reader.
		</DL>
	@see	Integer#parseInt(String)
	@see	Long#parseLong(String, int)
	@see	Double#valueOf(String)
	@see	Value#set_data(Object)
	@see	Value#Base(int)
	@see	Value#set_type(int)
	@see	#translate_from_escape_sequences(String_Buffer)
*/
public Value Get_Datum
	(
	Value		The_Value
	)
	throws PVL_Exception
{
if (Is_Empty ())
	return null;
if ((DEBUG & DEBUG_DATUM) != 0)
	System.out.println
		(">>> Get_Datum: Next_Location = " + Next_Location ());

long
	delimiter;
int
	end,
	character = 0,
	type = 0;


try {	//	Possible IOException from the String_Buffer_Reader.

//	Find the beginning of the datum string.
if (Is_End (delimiter = Next_Location
		(skip_whitespace_and_comments (Next_Location ()))))
	{
	if ((DEBUG & DEBUG_DATUM) != 0)
		System.out.println
			("Get_Value: " + delimiter
				+ " Is_End\n<<< Get_Datum: Next_Loction = " + delimiter);
	return null;
	}
if ((DEBUG & DEBUG_DATUM) != 0)
	System.out.println
		("Get_Datum: Datum starts at location " + delimiter);

//	Check the beginning of the datum string.
switch (Char_At (delimiter))
	{
	//	End of PVL statement cases:
	case STATEMENT_END_DELIMITER:
		//	Nothing to get.
		break;

	//	Syntax error cases:
	case PARAMETER_NAME_DELIMITER:
	case PARAMETER_VALUE_DELIMITER:
	case SET_START_DELIMITER:
	case SET_END_DELIMITER:
	case SEQUENCE_START_DELIMITER:
	case SEQUENCE_END_DELIMITER:
	case UNITS_START_DELIMITER:
	case UNITS_END_DELIMITER:
	case NUMBER_BASE_DELIMITER:
		/*
			Note: It would be possible here to be VERY tolerant
			and just take these illegal characters as the first
			character of an identifier string, but this seems
			unlikely in any case.
		*/
		throw new PVL_Exception
			(
			ID, PVL_Exception.ILLEGAL_SYNTAX,
			"When a value datum was expected.",
			delimiter
			);

	//	Quoted string cases:
	case TEXT_DELIMITER:
		type = Value.TEXT;
	case SYMBOL_DELIMITER:
		if (type == 0)
			type = Value.SYMBOL;
		if ((DEBUG & DEBUG_DATUM) != 0)
			System.out.println
				("Get_Datum: Getting a quoted string.");
		String
			//	CAUTION: Get_Quoted_String will update the Next_Location.
			string = Get_Quoted_String ();
		The_Value.set_data (string);
		The_Value.set_type (type);
		return The_Value;

	//	Numeric value or identifier:
	default:
		//	Find the value string delimiter.
		if ((delimiter =
				Skip_Until (Next_Location (), PARAMETER_VALUE_DELIMITERS)) < 0)
			//	Hit the end of input
			delimiter = End_Location ();
		if ((DEBUG & DEBUG_DATUM) != 0)
			System.out.println
				("Get_Datum: Numeric value or symbol ends at location "
					+ delimiter);

		String
			value = Substring (Next_Location (), delimiter);
		if ((end = value.indexOf (COMMENT_START_DELIMITERS)) > 0)
			{
			//	Only take the part up to the comments.
			value = value.substring (0, end);
			delimiter = Next_Location () + end;
			}
		if ((DEBUG & DEBUG_DATUM) != 0)
			System.out.println ("Get_Datum: value string - " + value);
		if (! Verbatim_Strings)
			//	Convert escape sequences to special characters.
			value = translate_from_escape_sequences
				(new String_Buffer (value)).toString ();

		if (Strict ||
			! All_Values_Strings)
			{
			/*..................................................................
				Try for a number.
			*/
			if ((end = value.indexOf (NUMBER_BASE_DELIMITER)) > 0)
				{
				/*..............................................................
					Probable base notation value ([sign]Base#value#).
				*/
				if ((DEBUG & DEBUG_DATUM) != 0)
					System.out.println
						("Get_Datum: Probable base notation value.");
				try
					{
					int
						base = Integer.parseInt
							(value.substring (0, end));
					if ((character = ++end) >= value.length ())
						throw new NumberFormatException ();
					if ((end = value.indexOf (NUMBER_BASE_DELIMITER, end)) < 0)
						end = value.length ();
					long
						number = Long.parseLong
							(value.substring (character, end), Math.abs (base));
					if (base < 0)
						{
						base = -base;
						number = -number;
						}
					The_Value.set_data (new Long (number));
					The_Value.Base (base);
					The_Value.set_type (Value.INTEGER);
					if ((DEBUG & DEBUG_DATUM) != 0)
						System.out.println
							("Get_Datum: Integer = "
								+ number + " with base " + base);
					break;
					}
				catch (NumberFormatException exception)
					{/* Must be a symbol */}
				}
			else
				{
				/*..............................................................
					Try for an integer.
				*/
				try
					{
					The_Value.set_data (Long.valueOf (value));
					The_Value.Base (10);
					The_Value.set_type (Value.INTEGER);
					if ((DEBUG & DEBUG_DATUM) != 0)
						System.out.println
							("Get_Datum: Integer = "
								+ (Long)The_Value.Data ());
					break;
					}
				catch (NumberFormatException exception)
					{
					/*..........................................................
						Try for a real number.
					*/
					try
						{
						The_Value.set_data (Double.valueOf (value));
						The_Value.set_type (Value.REAL);
						if ((DEBUG & DEBUG_DATUM) != 0)
							System.out.println
								("Get_Datum: Real = "
									+ (Double)The_Value.Data ());
						break;
						}
					catch (NumberFormatException accept)
						{/* Must be a symbol */}
					}
				}
			}
		/*......................................................................
			Couldn't make a number. It's a string.
		*/
		The_Value.set_data (value);
		if ((DEBUG & DEBUG_DATUM) != 0)
			System.out.println
				("Get_Datum: String = " + (String)The_Value.Data ());

		/*	Check for possible date and time string.
			>>> WARNING <<< This is a cursory check,
			it's not determinate.
		*/
		if (new String_Buffer (value).skip_until (0, DATE_TIME_DELIMITERS) > 0)
			The_Value.set_type (Value.DATE_TIME);
		else
			The_Value.set_type (Value.IDENTIFIER);

		//	Check for reserved characters.
		if ((character = Bad_Character (value)) >= 0)
			{
			Warning
				(
				PVL_Exception.RESERVED_CHARACTER,
				"At character " + character
					+ " of the datum \""
					+ new String_Buffer (value).special_to_escape () + "\"",
				Next_Location () + character
				);
			if (Strict)
				throw Last_Warning;
			}
	}

if ((DEBUG & DEBUG_DATUM) != 0)
	System.out.println ("<<< Get_Datum: Next_Location = " + delimiter);
Next_Location (delimiter);

}
catch (IOException exception)
	{
	//	From the String_Buffer_Reader.
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"In Get_Datum.\n"
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}

return The_Value;
}


/*------------------------------------------------------------------------------
*/
/**	Gets a units description String from the source of PVL statements.
<P>

	After skipping over any whitespace, the next character must start a
	units description sequence or nothing (null) is returned. A units
	description sequence starts after a {@link #UNITS_START_DELIMITER
	<CODE>UNITS_START_DELIMITER</CODE>} and ends before a {@link
	#UNITS_END_DELIMITER <CODE>UNITS_END_DELIMITER</CODE>}. A units
	description sequence without the closing
	<CODE>UNITS_END_DELIMITER</CODE> will result in a
	<CODE>MISSING_UNITS_END</CODE> exception in <CODE>Strict</CODE>
	mode; otherwise a <CODE>Warning</CODE> is registered and the next
	{@link #PARAMETER_VALUE_DELIMITER
	<CODE>PARAMETER_VALUE_DELIMITER</CODE>}, any of the
	SET/SEQUENCE/UNITS START/END delimiters, {@link
	#STATEMENT_END_DELIMITER <CODE>STATEMENT_END_DELIMITER</CODE>}, or
	the end of the input stream is taken as the end of the units
	description. <B>Note</B>: Though an effort is made to recover from
	encountering an unending units description, this will only be
	effective when no other normally closed units descripiton occurs in
	the input stream (if a normally closed units descripiton does occur
	after an unclosed units description, the latter will be taken as
	the end of the former), and in this case the input stream will have
	been read into memory until it is empty.
<P>
	Sequential comments, with nothing but white space intervening, are
	accumulated with a single new-line ('\n') chararacter separating
	them in the resulting String that is returned. In
	<CODE>Strict</CODE> mode comments that wrap across line breaks
	cause an exception. When <CODE>Verbatim_Strings</CODE> are not
	enabled whitespace is trimmed from the end of each comment (but not
	the beginning), and escape sequences are translated into their
	corresponding special characters.
<P>
	If a units descripiton is found the <CODE>Next_Location</CODE> of the
	input stream is moved to the position immediately following it. The
	units description is trimmed of leading and trailing whitespace.
	If <CODE>Verbatim_Strings</CODE> is not enabled, then all comment
	sequences are removed from the units description String, all whitespace
	sequences are collapsed to a single space (' ') character, and
	escape sequences are substituted for their corresponding special
	characters.
<P>
	@return	A units description String, or null if no units description
		occurs before the next PVL item or the end of input.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>MISSING_UNITS_END</CODE>
			<DD>A units description does not have a
			<CODE>UNITS_END_DELIMITER</CODE>.
		<DT><CODE>FILE_IO</CODE>
			<DD>If an IOException occurred in the String_Buffer_Reader.
		</DL>
	@see	#Verbatim_Strings(boolean)
	@see	#translate_from_escape_sequences(String_Buffer)
*/
public String Get_Units ()
	throws PVL_Exception
{
if (Is_Empty ())
	return null;
if ((DEBUG & DEBUG_UNITS) != 0)
	System.out.println
		(">>> Get_Units: Next_Location = " + Next_Location ());

String_Buffer
	units = null;
long
	delimiter,
	end;

//	Find the beginning of the units string.
if (Is_End (delimiter = skip_whitespace_and_comments (Next_Location ())) ||
	Char_At (delimiter) != UNITS_START_DELIMITER)
	{
	if ((DEBUG & DEBUG_UNITS) != 0)
		System.out.println
			("<<< Get_Units: Next_Location = " + Next_Location ());
	return null;
	}
delimiter++;

try {	//	Possible IOException from the String_Buffer_Reader.

//	Find the end of the units string.
if ((end = Location_Of (delimiter, UNITS_END_DELIMITER)) < 0)
	{
	Warning
		(
		PVL_Exception.MISSING_UNITS_END,
		"For value units starting with \""
			+ Substring (delimiter - 1,
				Math.min (delimiter + 19, End_Location ())) + "\" ...",
		delimiter - 1
		);
	if (Strict)
		throw Last_Warning;

	/*	Lacking a formal units string end marker
		just find the next non-white_space parameter value delimiter.
	*/
	for (end = delimiter;
		(end = Skip_Until (end, PARAMETER_VALUE_DELIMITERS)) >= 0;
		 end = Skip_Over  (end, WHITESPACE))
		if (end != Skip_Until (end, WHITESPACE))
			break;
	if (end < 0)
		end = End_Location ();

	//	Get the units string.
	units = new String_Buffer (Substring (delimiter, end));
	}
else
	units = new String_Buffer (Substring (delimiter, end++));
if ((DEBUG & DEBUG_UNITS) != 0)
	System.out.println ("Get_Units: " + units.toString ());

//	Move the Next_Location to after the end of the units string.
Next_Location (end);

}
catch (IOException exception)
	{
	//	From the String_Buffer_Reader.
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"In Get_Units.\n"
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}

units.trim ();	//	Remove leading and trailing whitespace.
if (! Verbatim_Strings)
	{
	//	Collapse each comments sequence to a single space.
	int
		first,
		last;
	while ((first = units.skip_until (0, COMMENT_START_DELIMITERS)) >= 0)
		{
		if ((last = units.skip_until (first, COMMENT_END_DELIMITERS)) < 0)
			last = units.length ();
		else
			last += COMMENT_END_DELIMITERS.length ();
		units.replace (first, last, " ");
		}
	/*
		Replace all whitespace sequences with a single space,
		and convert all escape sequences to their special characters.

		Note: The escape sequences are converted last so any
		resulting whitespace will not be collapsed away.
	*/
	translate_from_escape_sequences
		(units.replace_span (0, WHITESPACE, " "));
	}

if ((DEBUG & DEBUG_UNITS) != 0)
	System.out.println
		("<<< Get_Units: Next_Location = " + Next_Location ());
return (units == null) ? null : units.toString ();
}


/*------------------------------------------------------------------------------
*/
/**	Gets a quoted String from the source of PVL statements.
<P>
	The next non-whitespace character is taken to be the "quote"
	character. The characters following the first quote character up to
	but not including the next, non-escaped (not preceeded by a
	backslash, '\') quote character are the quoted string. If the
	closing quote character can not be found, then a
	<CODE>MISSING_QUOTE_END</CODE> <CODE>Warning</CODE> will be
	registered (the exception will be thrown in <CODE>Strict</CODE>
	mode) and the quoted string will end at the end of the input
	stream. <B>Note</B>: The lack of a closing quote character will
	cause the entire input stream to be read into memory until it is
	emtpy. The <CODE>Next_Location</CODE> is moved to the position
	immediately following the last quote character.
<P>
	If <CODE>Verbatim_Strings</CODE> is not enabled then line break
	sequences (one or more sequential line breaks) and any surrounding
	whitespace (whitespace ending the last line and beginning the next
	line) are replaced with a single space (' ') character. If, however,
	<CODE>String_Continuation</CODE> is enabled and the last
	non-whitespace character before the line break sequence is a {@link
	#STRING_CONTINUATION_DELIMITER
	<CODE>STRING_CONTINUATION_DELIMITER</CODE>} then no space remains
	(i.e. the string ending with the last non-whitespace character on the
	last line is continued with the first non-whitspace character on the
	next line). In addition, escape sequences are translated to their
	corresponding special characters. Sequences of characters bracketed
	by {@link #VERBATIM_STRING_DELIMITERS
	<CODE>VERBATIM_STRING_DELIMITERS</CODE>} are taken verbatim; they are
	subject to neither end of line treatment nor escape sequence
	translation.
<P>
	@return	A String, or null if no non-whitespace character occurs
		before the end of input.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>MISSING_QUOTE_END</CODE>
			<DD>A closing quote was not found in the input stream.
		<DT><CODE>FILE_IO</CODE>
			<DD>If an IOException occurred in the String_Buffer_Reader.
		</DL>
	@see	#Verbatim_Strings(boolean)
	@see	#String_Continuation(boolean)
	@see	String_Buffer#escape_to_special()

*/
public String Get_Quoted_String ()
	throws PVL_Exception
{
if (Is_Empty ())
	return null;
if ((DEBUG & DEBUG_QUOTED_STRING) != 0)
	System.out.println (">>> Get_Quoted_String");

long
	location,
	start,
	end;

String_Buffer
	quote = null;

try {	//	Possible IOException from the String_Buffer_Reader.

//	The first non-whitespace character is the quotation mark.
if (Is_End (start = location = Next_Location
		(Skip_Over (Next_Location (), WHITESPACE))))
	{
	if ((DEBUG & DEBUG_QUOTED_STRING) != 0)
		System.out.println
			("Get_Quoted_String: " + start
				+ " Is_End\n<<< Get_Datum: Next_Loction = " + start);
	return null;
	}
if ((DEBUG & DEBUG_QUOTED_STRING) != 0)
	System.out.println
		("Get_Quoted_String: Quote character is >"
		+ Char_At (start) + "< at location " + start);

//	Find the matching closing quote.
while ((location = Location_Of (++location, Char_At (start))) >= 0)
	//	Allow for escaped quotation mark.
	if (Char_At (location - 1) != '\\')
		break;
if (location < 0)
	{
	//	No end quote.
	Warning
		(
		PVL_Exception.MISSING_QUOTE_END,
		"For the quoted string starting with "
			+ Substring (start, Math.min (start + 20, End_Location ()))
			+ " ...",
		start
		);
	if (Strict)
		throw Last_Warning;
	location = End_Location ();
	}

//	Make a duplicate of the quoted string.
quote = new String_Buffer (Substring (++start, location));
if ((DEBUG & DEBUG_QUOTED_STRING) != 0)
	System.out.println ("Get_Quoted_String: " + quote.toString ());

//	Move the Next_Location past the quoted string.
Next_Location (location + 1);

}
catch (IOException exception)
	{
	//	From the String_Buffer_Reader.
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"In Get_Quoted_String.\n"
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}

if (! Verbatim_Strings)
	{
	/*	Compress out line wrap effects.
		Sections between VERBATIM_STRING_DELIMITERS are not modified.
	*/
	int
		first = 0,
		last,
		delimiters_length = Parser.VERBATIM_STRING_DELIMITERS.length ();
	String_Buffer
		section;

	for (last = quote.index_of (first, VERBATIM_STRING_DELIMITERS);
		 first < quote.length ();
		 last = quote.index_of (first = last, VERBATIM_STRING_DELIMITERS))
		{
		if (last < 0)
			//	This section extends to the end of the string.
			last = quote.length ();
		section = new String_Buffer (quote.substring (first, last));

		int
			begin = 0,
			index;
		while ((index = section.skip_until (begin, LINE_DELIMITERS)) >= 0)
			{
			//	Backup over any trailing white space.
			if ((begin = section.skip_back_over (index, WHITESPACE)) >= 0)
				{
				if (! String_Continuation ||
					section.charAt (begin) != STRING_CONTINUATION_DELIMITER)
					{
					if (section.charAt (begin) != STRING_CONTINUATION_DELIMITER ||
						begin == 0 ||
						section.charAt (begin - 1) == ' ')
						//	Allow one space.
						section.setCharAt (++begin, ' ');
					begin++;
					}
				}
			else
				begin = 0;

			/*	Skip over any whitespace leading the next line.

				Note that skipping over white space here will
				also skip over any redundant line breaks.
			*/
			index = section.skip_over (index, WHITESPACE);

			//	Delete the extra whitespace.
			section.delete (begin, index);
			}

		//	Convert escape sequences to special characters.
		section.escape_to_special ();

		//	Replace the section in the string.
		quote.replace (first, last, section.toString ());
		last = first + section.length ();

		//	Skip the next, verbatim, section.
		if ((last += delimiters_length) == quote.length () ||
			(last = quote.index_of (last, VERBATIM_STRING_DELIMITERS)) < 0 ||
			(last += delimiters_length) == quote.length ())
			//	No more sections.
			break;
		}
	}

if ((DEBUG & DEBUG_QUOTED_STRING) != 0)
	System.out.println
		("<<< Get_Quoted_String: Next_Location = " + Next_Location ());
return quote.toString ();
}


/*==============================================================================
	Utility functions:
*/
/**	Gets the next location in the PVL source stream following any
	sequence of whitespace and/or comments.

	A {@link #STATEMENT_CONTINUATION_DELIMITER
	<CODE>STATEMENT_CONTINUATION_DELIMITER</CODE>} and any {@link
	#Crosshatch_Comments(boolean) Crosshatch_Comments} (if enabled) are
	included in the whitespace category. <B>Note</B>: As with the
	<CODE>Get_Comments</CODE> method, a comment without a closing
	sequence is taken to end and the next line break,  {@link
	#STATEMENT_END_DELIMITER <CODE>STATEMENT_END_DELIMITER</CODE>}, or
	the end of the input stream; but this condition will cause the
	entire input stream to be read into memory.
<P>
	@param	location	The starting location from which to skip over
		whitespace and comments.
	@return	The location of the next character after any whitespace or
		comments.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>MISSING_COMMENT_END</CODE>
			<DD>A comment does not have
			<CODE>COMMENT_END_DELIMITERS</CODE> and <CODE>Strict</CODE>
			mode is enabled.
		<DT><CODE>FILE_IO</CODE>
			<DD>If an IOException occurred in the String_Buffer_Reader.
		</DL>
*/
public long skip_whitespace_and_comments
	(
	long		location
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println
		(">>> Parser.skip_whitespace_and_comments: " + location);
long
	comment_end;

try {	//	Possible IOException from the String_Buffer_Reader.
while (true)
	{
	//	Find the beginning of the comment string.
	if (Is_End (location = skip_crosshatch_comments (location)) ||
		! Equals (location, COMMENT_START_DELIMITERS))
		break;
	if ((DEBUG & DEBUG_UTILITIES) != 0)
		System.out.println
			("  Comment at " + location);
	//	Find the end of the comment string.
	location += COMMENT_START_DELIMITERS.length ();
	if ((comment_end = Location_Of (location, COMMENT_END_DELIMITERS)) < 0)
		{
		//	No comment end.
		Warning
			(
			PVL_Exception.MISSING_COMMENT_END,
			"For comment starting with \""
				+ Substring (location,
					Math.min (location + 20, End_Location ())) + "\" ...",
			location
			);
		if ((DEBUG & DEBUG_UTILITIES) != 0)
			System.out.println
				("Parser.skip_whitespace_and_comments:\n"
				+ Last_Warning.getMessage ());
		if (Strict)
			throw Last_Warning;

		//	Assume it ends at the end of the line.
		if ((comment_end = Skip_Until
				(location, LINE_DELIMITERS + STATEMENT_END_DELIMITER)) < 0)
			location = End_Location ();

		location = Skip_Over
			(comment_end, LINE_DELIMITERS + STATEMENT_END_DELIMITER);
		}
	else
		location = comment_end + COMMENT_END_DELIMITERS.length ();
	if ((DEBUG & DEBUG_UTILITIES) != 0)
		System.out.println ("    ends at " + location);
	}
}
catch (IOException exception)
	{
	//	From the String_Buffer_Reader.
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"In skip_whitespace_and_comments.\n"
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}
if ((DEBUG & DEBUG_UTILITIES) != 0)
	System.out.println
		("<<< Parser.skip_whitespace_and_comments: " + location);
return location;
}

private long skip_crosshatch_comments
	(
	long	location
	)
	throws PVL_Exception
{
try {	//	Possible IOException from the String_Buffer_Reader.
if (Strict || ! Crosshatch_Comments)
	//	Just skip whitespace.
	location = Skip_Over
		(location, WHITESPACE + STATEMENT_CONTINUATION_DELIMITER);
else
	{
	if ((DEBUG & DEBUG_UTILITIES) != 0)
		System.out.println
			(">>> Parser.skip_crosshatch_comments: " + location);
	while (true)
		{
		//	Find the beginning of the comment string.
		if (Is_End (location = Skip_Over
				(location, WHITESPACE + STATEMENT_CONTINUATION_DELIMITER)) ||
			Char_At (location) != CROSSHATCH)
			break;
		//	A crosshatch comment extends to the end of the line.
		if ((DEBUG & DEBUG_UTILITIES) != 0)
			System.out.println ("  Crosshatch comment at " + location);
		location = Skip_Until (location, LINE_BREAK);
		if (location < 0)
			location = End_Location ();
		if ((DEBUG & DEBUG_UTILITIES) != 0)
			System.out.println ("    ends at " + location);
		}
	if ((DEBUG & DEBUG_UTILITIES) != 0)
		System.out.println
			("<<< Parser.skip_crosshatch_comments: " + location);
	}
}
catch (IOException exception)
	{
	//	From the String_Buffer_Reader.
	throw new PVL_Exception
		(
		ID, PVL_Exception.FILE_IO,
		"In skip_whitespace_and_comments.\n"
		+ ((exception.getMessage () == null) ?
			"" : "\n" + exception.getMessage ())
		);
	}
return location;
}


/**	Translates escape sequences in a String_Buffer to their
	corresponding special characters.

	The {@link String_Buffer#escape_to_special()
	<CODE>escape_to_special</CODE>} method is used to translate escape
	sequences to special characters. However the occurance of {@link
	Parser#VERBATIM_STRING_DELIMITERS
	<CODE>VERBATIM_STRING_DELIMITERS</CODE>} starts a sequence of
	characters that are taken verbatim (they are not translated) up to
	the next <CODE>VERBATIM_STRING_DELIMITERS</CODE> or the end of the
	string (the <CODE>VERBATIM_STRING_DELIMITERS</CODE> are dropped).
<P>
	@param	string	The String_Buffer to be translated.
	@return	The translated String_Buffer.
	@see	String_Buffer#escape_to_special()
*/
public static String_Buffer translate_from_escape_sequences
	(
	String_Buffer	string
	)
{
if (string == null)
	return string;

int
	first = 0,
	last,
	length,
	delimiters_length = VERBATIM_STRING_DELIMITERS.length ();
String_Buffer
	section;

//	Find sections delimited by VERBATIM_STRING_DELIMITERS.
for (last = string.index_of (first, VERBATIM_STRING_DELIMITERS);
	 first < string.length ();
	 last = string.index_of (first = last, VERBATIM_STRING_DELIMITERS))
	{
	if (last < 0)
		//	This section extends to the end of the string.
		last = string.length ();
	section = new String_Buffer (string.substring (first, last));
	length = section.length ();

	//	Convert escape sequences to special characters.
	section.escape_to_special ();
	if (section.length () != length)
		{
		//	The section changed; replace it in the string.
		string.replace (first, last, section.toString ());
		last = first + section.length ();
		}

	//	Skip the next, verbatim, section.
	if ((last += delimiters_length) >= string.length () ||
		(last = string.index_of (last, VERBATIM_STRING_DELIMITERS)) < 0 ||
		(last += delimiters_length) == string.length ())
		//	No more sections.
		break;
	}
string.replace (0, VERBATIM_STRING_DELIMITERS, "");
return string;
}


/**	Checks a String for any bad character.

	A bad character is one of the {@link #RESERVED_CHARACTERS
	<CODE>RESERVED_CHARACTERS</CODE>} or a non-printable character.
<P>
	@param	string	The String to check.
	@return	The index of the first bad character found in the string,
		or -1 if no bad characters were found.
*/
public static int Bad_Character
	(
	String	string
	)
{
if (string != null)
	{
	for (int index = 0;
			 index < string.length ();
			 index++)
		{
		char character = string.charAt (index);
		if (! isprint (character) ||
			RESERVED_CHARACTERS.indexOf (character) >= 0)
			return index;
    	}
	}
return -1;
}


/**	Tests if a character is printable: in the ASCII range from the
	space character (' ') to the tilde character ('~') inclusive.
<P>
	@param	character	The char to test.
	@return	true if the character is printable; false otherwise.
*/
public static boolean isprint
	(
	char	character
	)
{
return character >= ' ' && character < 0x7F;
}


/*------------------------------------------------------------------------------
*/
/**	Gets the Parameter classification code corresponding to the
	specified special Parameter name String.

	The special Parameter names and their classification codes are:
<P>
	<I>Begin_Object</I> or <I>BeginObject</I> - <CODE>BEGIN_OBJECT</CODE><BR>
	<I>Object</I> - <CODE>OBJECT</CODE><BR>
	<I>End_Object</I> or <I>EndObject</I> - <CODE>END_OBJECT</CODE><BR>
	<I>Begin_Group</I> or <I>BeginGroup</I> - <CODE>BEGIN_GROUP</CODE><BR>
	<I>Group</I> - <CODE>GROUP</CODE><BR>
	<I>End_Group</I> or <I>EndGroup</I> - <CODE>END_GROUP</CODE><BR>
	<I>End</I> - <CODE>END_PVL</CODE><BR>
<P>
	The names are not case sensitive.
<P>
	@param	name	A String that may be a special Parameter name.
	@return	the Parameter classification code associated with the
		special Parameter name; or -1 if the name is not a special
		Parameter name.
	@see	Parameter
*/
public static int Special_Classification
	(
	String		name
	)
{
if (name != null)
	{
	for (int element = 0;
		 element < SPECIAL_NAMES.length;
		 element++)
		if (name.equalsIgnoreCase (SPECIAL_NAMES[element]))
			return SPECIAL_CLASSIFICATIONS[element];
	}
return -1;
}

/**	Gets the special Parameter name String for the specified Parameter
	classification code.
<P>
	@param	classification	A Parameter classification code int.
	@return	The special Parameter name String associated with the
		classification code; or null if the classification code is not
		associated with a special Parameter name.
	@see	#Special_Classification(String)
*/
public static String Special_Name
	(
	int			classification
	)
{
for (int element = 0;
	 element < SPECIAL_CLASSIFICATIONS.length;
	 element++)
	if (classification == SPECIAL_CLASSIFICATIONS[element])
		return SPECIAL_NAMES[element];
return null;
}


/*------------------------------------------------------------------------------
*/
private void Warning
	(
	String		description,
	String		explanation,
	long		location
	)
{
Last_Warning =
	new PVL_Exception
		(
		ID,
		description,
		explanation,
		location
		);
if ((DEBUG & DEBUG_WARNINGS) != 0)
	System.out.println
		("Parser.Warning:\n" + Last_Warning);

if (First_Warning == null)
	First_Warning = Last_Warning;
}


}	//	class Parser
