/*	String_Buffer_Reader

PIRL CVS ID: String_Buffer_Reader.java,v 1.16 2012/04/16 06:15:35 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Strings;

import	java.lang.StringBuffer;
import	java.lang.String;
import	java.lang.Math;
import	java.lang.IllegalStateException;
import	java.io.Reader;
import	java.io.IOException;
import	java.text.CharacterIterator;

/**	A <I>String_Buffer_Reader</I> provides methods to manipulate a
	character stream as if it were a <I>String_Buffer</I> by backing it
	with a Reader to provide the characters for the String_Buffer.
<P>
	The intended use is to consume a character stream, in a
	semi-sequential manner, processing <I>statements</I> as they are
	encountered by moving along the stream from statement to statement. A
	statement is composed of a contiguous sequence of characters of
	variable length.
<P>
	The character stream may be sourced from a Reader or a String. A
	String source is a convenience that allows a String_Buffer to be used
	transparently in the same context where a Reader might also be used,
	but instead a "pre-read" source of characters is provided directly (a
	StringReader could also be used).
<P>
	For a Reader source the character stream is buffered using a sliding
	window. The source contents are referenced by virtual <I>Location</I>
	values that act as indexes into the entire character stream; the
	first character of the stream is at location 0. The buffer window is
	automatically extended to contain characters from any location
	available in the input stream.
<P>
	To avoid a large number of read operations on the input stream, the
	buffer is always extended by a specified <I>Size_Increment</I>. In
	addition, a <I>Read_Limit</I> may be specified to force input
	termination at any location in the character stream. Input may also
	be terminated when a threshold of non-text bytes (more than a
	specified sequential limit) occur in the stream, as would typically
	occur after the initial label area of an image file.
<P>
	When statements from the character stream have been processed and are
	no longer needed the <I>Next_Location</I> is updated. That part of
	the character stream in the buffer before the Next_Location is
	considered to be disposable so that the next time the buffer needs to
	be extended to a new stream location the portion of the buffer
	containing characters from before the Next_Location are deleted,
	which slides the buffer forward. Thus instead of simply extending
	indefinitely, the buffer is reused by moving the contents to cover
	the section of the character stream being processed and removing
	contents no longer needed. In this way character streams of
	indefinite length may be processed using a relatively small buffer
	size.
<P>
	Modifications may be made to the contents of the buffer using any of
	the String_Buffer methods (which includes all of the StringBuffer
	methods). When the buffer needs to be extended, because a reference
	is made to a location beyond the end of the current buffer contents,
	input characters are always appended to the end of the existing
	buffer contents regardless of changes that have been made that might
	have altered the number of characters it contains. However, the
	<CODE>Next_Location</CODE> will not change unless specifically
	updated by the user, and all characters before the
	<CODE>Next_Location</CODE> will be deleted whenever the buffer
	contents are extended, so characters before the
	<CODE>Next_Location</CODE> should be considered consumed. Thus the
	String_Buffer_Reader effectively offers String_Buffer (StringBuffer)
	access to all characters obtained from the Reader start at the
	<CODE>Next_Location</CODE> and extending on. If the
	<CODE>Next_Location</CODE> is never moved forward (it can be moved
	backwards, but not before the current <CODE>Buffer_Location</CODE>)
	then all characters from the Reader are always available (this is
	always the case for a String source of characters), but sliding the
	buffer forward past consumed statements is what provides efficient
	use of buffer memory for indefinately long input streams.
<P>
	<B>Note</B>: Those String_Buffer methods which search forward for
	character patterns will extend the buffer to the end of input if the
	pattern is not found. To avoid unnecessarily large buffer extensions
	the <CODE>Read_Limit</CODE> can be set to some location downstream
	that the application considers to be beyond where the pattern could
	reasonably be found.
<P>
	When the character source is a String, nothing is ever read.
	Nevertheless, the contents of the buffer may still be appended with
	additional characters and the <CODE>Next_Location</CODE> may still be
	moved forward. When buffer manipulation methods attempt to extend the
	buffer, it will still be slid forward by deleting characters before
	the <CODE>Next_Location</CODE>. In this case virtual location values
	may be thought of as the number of charaters consumed (slid over)
	plus the number of characters in the buffer.
<P>
	A special feature of this String_Buffer_Reader is its ability to
	handle character streams with binary sized records (as produced by
	DEC VMS systems). Each record in such a stream has the form:
<P>
	<BLOCKQUOTE>
	<B><I>Size Characters</I></B> [ <B><I>Pad</I></B> ] 
	</BLOCKQUOTE>
<P>
	Where the <I>Size</I> is a two byte, LSB first, binary integer value
	specifying the number of bytes of <I>Characters</I> to follow. And
	<I>Pad</I> is a byte of value 0 that is present when the Size is odd.
	The Size bytes will be replaced with a LINE_BREAK (CR-LF) sequence,
	and any Pad byte will be replaced with a space character, thus
	providing a consistent character stream. <B>Note</B>: This special
	filtering will only be applied when the source of characters is a
	Reader.
<P>
	@see		String_Buffer

	@author		Bradford Castalia, UA/PIRL
	@version	1.16 
*/
public class String_Buffer_Reader
	extends String_Buffer
{
public static final String
	ID = "PIRL.Strings.String_Buffer_Reader (1.16 2012/04/16 06:15:35)";

/*------------------------------------------------------------------------------
	String buffer management:
*/
/**	The amount to increase the size of the string buffer when it needs to
	be enlarged while reading a file.
<P>
	It should be at least 8k to allow for the largest possible sized
	record on first read.
*/
public static final int
	DEFAULT_SIZE_INCREMENT		= 16384;
private int
	Size_Increment				= DEFAULT_SIZE_INCREMENT;

/*	Input characters array.

	This array holds the characters input from the Character_Reader. The
	characters may be scanned in the array before being appended to the
	String_Buffer base class. This is, indeed, inefficient: it would be
	much better to read directly into the String_Buffer storage, but
	the Reader's read method will only accept a char[], and the
	String_Buffer (i.e. its underlying StringBuffer) does not provide
	direct access to its character storage in this form.

	Note that input data bytes are being read as char arrays which are
	16-bit data types in Java. This is inefficient in terms of space,
	but efficient for compatibility with the standard String data type
	being processed.
*/
private char[]
	Input_Array					= new char[Size_Increment];
//	The amount of valid characters in the array (from 0).
private int
	Input_Array_Amount			= 0;

/*	A Reader source of characters.

	This will remain null if the source is a String.
*/
private Reader
	Character_Reader			= null;

/*	String_Buffer contents locations:

		Location - relative to the input stream.
		Index    - relative to the current String_Buffer contents.
*/
private long
	//	Location of the first character in the buffer.
	Buffer_Location				= 0,
	//	Location of the next statement
	//		(characters before this location are disposable).
	Next_Location				= 0;

/**	The default {@link #Read_Limit(long) read limit}.
<P>
	This class was designed for use with the PVL Parser (which is a
	subclass) for processing PVL syntax statements. The read limit sets
	the maximum amount of a file to read when working through a file
	label. When the limit is reached it is presumed that there are no
	more statements to be processed. Set it to -1 for no limit.
<P>
	<B>N.B.</B>: Without a recognizable end-of-label marker it is quite
	possible for non-label file data to be interpreted as PVL
	statements. Thus it is advisable to set a reasonable limit on the
	amount of file data to read. It is typical for a PVL label to
	contain as one of its parameters the size of the label. This
	suggests a strategy of using a reasonably small limit (enough to
	ensure that the label size parameter, usually near the start of the
	PVL statements, will be read), finding this parameter, rewinding
	the file, and ingesting the PVL statements again using the
	parameter value as the limiting value. A less "intelligent" (but
	likely to be easier) approach is to use the default limit and
	simply check the validity of parameters, since applications are
	likely to know what parameters are valid for them.
*/
public static final long
	DEFAULT_READ_LIMIT			= 16 * DEFAULT_SIZE_INCREMENT;
private long
	//	Maximum number of characters to read.
	Read_Limit					= DEFAULT_READ_LIMIT,
	//	Total number of characters read so far.
	Total_Read					= 0;

/**	The <CODE>Read_Limit</CODE> size value that means what it says.
<P>
	@see	#Read_Limit(long)
*/
public static final long
	NO_READ_LIMIT				= -1;


/*	The number of non-text chars that will end data input.

	To allow for possible binary record size bytes and padding
	this value must be at least 4.
*/
private int
	Non_Text_Limit				= 4,
	Non_Text_Count				= 0;

private static String
	NL							= System.getProperty ("line.separator");


//	DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_DATA		= 1 << 0,
	DEBUG_BUFFER	= 1 << 1,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors:
*/
/*------------------------------------------------------------------------------
	File input
*/
/**	Creates a String_Buffer_Reader with the reader as the source of
	characters up to the specified limit.
<P>
	@param	reader	The Reader to use as the source of input.
	@param	limit	The maximum number of character bytes to read.
	@see	#Read_Limit(long)
*/
public String_Buffer_Reader
	(
	Reader		reader,
	long		limit
	)
{Set_Reader (reader).Read_Limit (limit);}

/**	Creates a String_Buffer_Reader with the reader as the unlimited
	source of characters.
<P>
	@param	reader	The Reader to use as the source of input.
*/
public String_Buffer_Reader (Reader reader)
{this (reader, 0);}

/**	Creates a String_Buffer_Reader with no character source.
<P>
	@see	#Set_Reader(Reader)
*/
public String_Buffer_Reader () {}


/*------------------------------------------------------------------------------
	String input
*/
/**	Creates a String_Buffer_Reader with a String source of characters.
<P>
	@param	string	The String to use as the source of input.
*/
public String_Buffer_Reader
	(
	String		string
	)
{
append (string);
Total_Read = string.length ();
}


/*==============================================================================
	Accessors.
*/
/**	Sets the Reader to use as the source of characters.
<P>
	Changing the Reader in mid stream may have unexpected side effects.
	Any data still being held in the internal character array pending
	further processing before transfer to the charactr buffer will
	remain in the virtual input stream before data read from the new
	Reader (note that for a {@link #String_Source()
	String source of characters} the internal character array is
	not used). The <CODE>Total_Read</CODE> will not be reset by a change
	of Reader; i.e. a single virtual character input stream is seen.
<P>
	<B>Note</B>: If the reader is set to null, then input has, by
	definition, ended, but the current contents of the buffer remain
	available. However, any data still pending processing in the
	internal character array is dropped. If this object was created
	with a String character source a Reader may be set to supplement
	the initial String. In this case the <CODE>Total_Read</CODE> will
	include the length of the initial String.
<P>
	@param	reader	The reader to associate with this object.
	@return	This Buffer_String_Reader.
	@see	#Total_Read()
	@see	#String_Source()
	@see	#Extend()
*/
public String_Buffer_Reader Set_Reader
	(
	Reader		reader
	)
{
Character_Reader = reader;
if (String_Source ())
	Input_Array_Amount = 0;
return this;
}

/**	Gets the Reader used as the source of characters.
<P>
	@return	The Reader associated with this object; will be null
		if the object currently has no Reader (e.g. it was created
		with a String character source).
	@see	#Set_Reader(Reader)
*/
public Reader Get_Reader ()
{return Character_Reader;}


/**	Tests if the source of characters is a Reader.
<P>
	@return	true if the source of characters is a Reader.
*/
public boolean Reader_Source ()
{return (Character_Reader != null);}

/**	Tests if the source of characters is a String (there is no Reader).
<P>
	@return	true if the source of characters is a String.
*/
public boolean String_Source ()
{return (Character_Reader == null);}

/**	Reset as if (almost) nothing happened.
<P>
	The {@link #Buffer_Location() buffer location} and {@link
	#Next_Location() next location} in the stream are reset to zero, the
	internal character buffer is marked as empty. If a {@link
	#Reader_Source() Reader source} (rather than {@link #String_Source()
	String source}) is being used the {@link #Total_Read() total read} is
	set to zero, the {@link #Read_Limit() read limit} set to {@link
	#NO_READ_LIMIT} and the base String_Buffer is cleared. <n>N.B.</b>:
	The internal character buffer {@link #Size_Increment() size
	increment} and {@link #Non_Text_Limit() non-text data threshold} are
	not changed.
<P>
	If {@link #Filter_Input() input filtering} is enabled it is reset
	so the source stream will be tested again.
*/
public String_Buffer_Reader Reset ()
{
Buffer_Location = 0;
Next_Location = 0;
Input_Array_Amount = 0;
if (Reader_Source ())
	{
	Total_Read = 0;
	Read_Limit = NO_READ_LIMIT;
	clear ();
	}
if (Sized_Record != NO_FILTERING)
	{
	Sized_Record = CHECK_FOR_SIZED_RECORDS;
	Padding = 0;
	LSB_Character = 0;
	}
return this;
}

/**	Gets the location in the input stream of the index in the buffer.
<P>
	Location values must be used when manipulating the contents
	of the buffer using String_Buffer_Reader methods.
<P>
	Location values are virtual with respect to the entire character
	input stream. Location 0 corresponds to the first character
	consumed after the buffer has slid forward, or the first character
	currently in the buffer if nothing has yet been consumed. If the
	number of characters in the buffer is only changed when it is
	{@link #Extend() <CODE>Extend</CODE>}ed, then location values are
	relative to all characters in the input stream. However, since the
	number of characters in the buffer may be changed by various
	methods, location values are actually relative to all characters
	that the buffer has slid over (been consumed) plus the current
	contents of the buffer; i.e. the virtual input stream.
<P>
	<B>Note</B>: Location values are long; index values are int.
<P>
	@param	index	A buffer index value.
	@return	The corresponding input stream location.
	@see	#Extend()
*/
public long Location (int index)
{return Buffer_Location + index;}

/**	Gets the index in the buffer for the location in the input stream.
<P>
	Index values are relative to the current contents of the buffer;
	index 0 corresponds to the character currently at the beginning of
	the buffer. Index values must be used when menipulating the
	contents of the buffer with String_Buffer (StingBuffer) methods.
<P>
	<B>Note</B>: Location values are long; index values are int.
<P>
	@param	location	A location in the input stream.
	@return	The corresponding buffer index relative to the
		current location of the beginning of the buffer in the
		input stream.
*/
public int Index (long location)
{return (int)(location - Buffer_Location);}

/**	Gets the current location of the beginning of the buffer in the
	virtual stream.
<P>
	@return	The location of the first character in the buffer.
*/
public long Buffer_Location ()
{return Buffer_Location;}

/**	Resets the virtual stream location values.
<p>
	The current {@link #Buffer_Location() character buffer location} in
	the source of charcters is subtracted from the {@link
	#Next_Location() next location} and the buffer location is set to
	zero. The {@link #Total_Read() total read} is also set to zero if
	the source of characters is a {@link #Reader_Source () stream reader},
	not a String.
<P>
	A reset has the effect of making it appear as if the character
	buffer had been loaded for the first time, but without affecting
	the current relative location of the next character to be read.
<P>
	Because the location values are long integers they can be exepected
	to remain valid even for a reader that is a coninuously generated
	stream (such as a network socket). Nevertheless, a reset when the
	next location exceeds some (very large) threshold will ensure that
	a stream of unlimited length can be continuously processed.
*/
public String_Buffer_Reader Reset_Location ()
{
Flush (Next_Index ());
Next_Location -= Buffer_Location;
Buffer_Location = 0;
if (Reader_Source ())
	Total_Read = 0;
return this;
}

/**	Sets the next location to position the buffer when sliding it
	forward.
<P>
	If the location is beyond the end of the current buffer contents
	the buffer is extended to include the location, if possible.
<P>
	@param	location	The next location to use for the beginning of
		the buffer when the buffer is slid forward.
	@return	The new value of the next location. The only case where
		this will be different from the specified location is when the
		end of input has been reached before the specified location
		could be reached; in which case the return value is the final
		end location.
	@throws	IndexOutOfBoundsException	If the location is before the
				beginning of the buffer.
	@throws	IOException	If there was a problem extending the buffer.
	@see	#End_Location()
	@see	#Extend()
*/
public long Next_Location
	(
	long		location
	)
	throws IndexOutOfBoundsException, IOException
{return (Next_Location = Location (get_index (location)));}

/**	Gets the current value of the next location.
<P>
	@return	The current value of the next location.
	@see	#Next_Location(long)
*/
public long Next_Location ()
{return Next_Location;}

/**	Sets the <CODE>Next_Location</CODE> using a buffer index value.
<P>
	@param	index	The buffer index value for the next location.
	@return	The index value of the new next location.
	@throws	IndexOutOfBoundsException	If the index is before the
				beginning of the buffer.
	@throws	IOException	If there was a problem extending the buffer.
	@see	#Next_Location(long)
*/
public int Next_Index (int index)
	throws IndexOutOfBoundsException, IOException
{
int
	 next_index = get_index (Location (index));
Next_Location = Location (next_index);
return next_index;
}

/**	Gets the index in the buffer of the <CODE>Next_Location</CODE>.
<P>
	@return	The index in the buffer of the next location.
	@see	#Next_Location()
*/
public int Next_Index ()
{return (int)(Next_Location - Buffer_Location);}

/**	Gets the location immediately following the last character in the
	buffer.
<P>
	@return	The location immediately following the last character in
		the buffer. 
*/
public long End_Location ()
{return Buffer_Location + length ();}

/**	Gets the buffer index of the last character in the buffer. This is
	the same as the number of characters currently in the buffer.
<P>
	@return	The buffer index of the last character in the buffer.
*/
public int End_Index ()
{return length ();}


/**	Sets the location where character input is to stop; the
	maximum number of characters to be obtained from the source.
<P>
	If the limit is 0, then the <CODE>DEFAULT_READ_LIMIT</CODE> (256
	KB) is used. If the limit is less than 0, then
	<CODE>NO_READ_LIMIT</CODE> will be used, so no upper limit will be
	imposed on the number of characters to obtain from the source.
<P>
	<B>Note</B>: The read limit is reset to the total number of
	characters read if a Reader encounters the end of the input stream.
<P>
	<B>Note</B>: The read limit will have no effect {@link
	#String_Source() when the character source is a String}. In this case
	nothing is ever read so the read limit is never changed.
<P>
	<B>Note</B>: When the {@link #Non_Text_Limit(int) threshold for
	sequential non-text data} has been reached while extending the
	buffer, then the read limit is automatically set to the number of
	characters read before the non-text sequence. Once this condition has
	been encountered, the read limit can not be changed without first
	resetting the non-text data threshold up to a higher value.
<P>
	@param	limit	The location where character input is to stop.
	@return	This String_Buffer_Reader.
	@see	#NO_READ_LIMIT
*/
public String_Buffer_Reader Read_Limit
	(
	long		limit
	)
{
if (Reader_Source () &&
	Non_Text_Count < Non_Text_Limit)
	{
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			(">>> String_Buffer_Reader.Read_limit:" + limit);
	if (limit > 0)
		Read_Limit = limit;
	else if (limit == 0)
		Read_Limit = DEFAULT_READ_LIMIT;
	else
		Read_Limit = NO_READ_LIMIT;
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("<<< String_Buffer_Reader.Read_limit: " + Read_Limit);
	}
return this;
}

/**	Sets the read limit to <CODE>NO_READ_LIMIT</CODE>.
<P>
	@return	This String_Buffer_Reader.
	@see	#Read_Limit(long)
*/
public String_Buffer_Reader No_Read_Limit ()
{return Read_Limit (NO_READ_LIMIT);}

/**	Gets the current limit where reading characters is to stop;
	the maximum number of characters to input from the Reader.
<P>
	This will be <CODE>NO_READ_LIMIT</CODE> if there is no limit to
	the number of characters to read from the source.
<P>
	@return	The current read limit.
*/
public long Read_Limit ()
{return Read_Limit;}

/**	Gets the total number of characters read so far. This value
	is the actual number of characters input from the Reader
	regardless of any subsequent use of the data.
<P>
	@return	The total number of characters read.
*/
public long Total_Read ()
{return Total_Read;}


/**	Tests if the end of input has been reached.
<P>
	The end of input occurs when the read limit has been reached. When
	the input source is a String, then it has always ended (i.e. there
	is nothing to read).
<P>
	@return	true if the character source has ended.
*/
public boolean Ended ()
{
return (Read_Limit != NO_READ_LIMIT && Total_Read >= Read_Limit) ||
		String_Source ();
}

/**	Tests if there are no more characters at the
	<CODE>Next_Location</CODE>; the source of characters is effectively
	empty.
<P>
	There will be no more characters at the <CODE>Next_Location</CODE>
	if the input has eneded and the <CODE>Next_Location</CODE> is at,
	or beyond, the <CODE>End_Location</CODE>.
<P>
	@return	true if there are no more statements available.
	@see	#Ended()
	@see	#End_Location()
*/
public boolean Is_Empty ()
{return (Ended () && Next_Location >= End_Location ());}

/**	Tests if the specified location is at or beyond the
	<CODE>End_Location</CODE>.
<P>
	<B>Note</B>: Unlike <CODE>Ended</CODE>, <CODE>Is_End</CODE> only
	indicates that the location is beyond the current buffer contents.
	There may still be more source characters available.
<P>
	@param	location	The location to test.
	@return	true if the location is at or beyond the
		<CODE>End_Location</CODE>.
	@see	#End_Location()
*/
public boolean Is_End (long location)
{return location >= End_Location ();}


/**	Sets the size increment by which the input buffer will be
	extended when it is slid forward in the Reader's input stream.
<P>
	If the amount is less than or equal to 0, then the default size
	increment (16 KB) will be applied. If the amount is less than or
	equal to the <CODE>Non_Text_Limit</CODE> threshold, then it will
	be increased by that amount. If the current internal character
	array contains more input data that the size increment, then it
	will be set to the amount of data being held. If the new size
	increment is different from the length of the current internal
	character array, then a new array will be allocated and any data in
	the old array will be copied into it.
<P>
	@param	amount	The new size increment.
	@return	This String_Buffer_Reader.
	@see	#Non_Text_Limit(int)
	@see	#Extend()
*/
public String_Buffer_Reader Size_Increment
	(
	int		amount
	)
{
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		(">>> String_Buffer_Reader.Size_Increment: " + amount);
if (amount <= 0)
	Size_Increment = DEFAULT_SIZE_INCREMENT;
else
	Size_Increment = amount;
if (Size_Increment <= Non_Text_Limit)
	Size_Increment += Non_Text_Limit;
if (Size_Increment < Input_Array_Amount)
	Size_Increment = Input_Array_Amount;

if (Input_Array.length != Size_Increment)
	{
	//	Re-allocate the array for input characters.
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("    Reallocating Input_Array: " + Size_Increment);
	char[]
		new_array = new char[Size_Increment];
	//	Copy the input data to the new array.
	for	(int index = 0;
			 index < Input_Array_Amount;
			 index++)
		new_array[index] = Input_Array[index];
	Input_Array = new_array;
	}
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		("<<< String_Buffer_Reader.Size_Incrment: " + Size_Increment);
return this;
}

/**	Gets the size increment by which the input buffer will be
	extended when it is slid forward in the Reader's input stream.
<P>
	@return	The current size increment.
	@see	#Size_Increment(int)
*/
public int Size_Increment ()
{return Size_Increment;}


/**	Sets the length of a non-text data sequence that will cause input
	from the Reader's character stream to the the buffer to stop.
<P>
	Text characters are recognized by the <CODE>Is_Text</CODE> method;
	characters that fail this test are non-text.
<P>
	Since this class is managing a String buffer, it is assumed that
	non-text data from the input should not be transferred into the
	buffer; i.e. the data are not valid String characters. However, to
	allow for possible input filter requirements (e.g. binary record size
	values), a sequence of non-text data less than the limit is allowed
	into the character buffer. When a sequence of non-text data in the
	input stream reaches the limit the <CODE>Read_Limit</CODE> is set to
	the amount read up to, but not including, the sequence of non-text
	data. A limit of 0 has the same effect as a limit of 1 (so a limit of
	zero will be forced to 1); i.e. no non-text data are acceptable.
	Specifying a negative limit disables non-text data checking; i.e. any
	non-text data is acceptable.
<P>
	<B>Note</B>: No sequence of non-text data as long as the limit amount
	is allowed into the object's character buffer; shorter sequences are
	allowed to pass. And since the <CODE>Read_Limit</CODE> is set to the
	location in the input stream immediately preceeding a limiting
	non-text sequence, the buffer will not be {@link #Extend()
	<CODE>Extend</CODE>}ed beyond this location. However, the
	<CODE>Non_Text_Limit</CODE> may subsequently be increased and the
	<CODE>Read_limit</CODE> lifted (in that order; the
	<CODE>Read_limit</CODE> can not be changed while a
	<CODE>Non_Text_Limit</CODE> block is in effect) to allow further
	processing of the input stream (data read from the Reader but not
	transferred to the character buffer is never lost). Nevertheless, any
	non-text data sequence outstanding will remain in effect and will be
	included in counting the length of the next non-text sequence; the
	non-text sequence length is only reset to 0 when a text character is
	seen in the input stream.
<P>
	<B>Warning</B>: The <CODE>Non_Text_Limit</CODE> may affect the
	operation of an input filter. The input filter provided with this
	class requires, if {@link #Filter_Input(boolean) input filtering is
	enabled}, a <CODE>Non_Text_Limit</CODE> of a least 4.
<P>
	@param	limit	The maximum allowed sequence of non-text input data.
	@return	This String_Buffer_Reader.
	@see	#Is_Text(char)
	@see	#Read_Limit(long)
	@see	#Extend()
*/
public String_Buffer_Reader Non_Text_Limit
	(
	int		limit
	)
{
if (limit == 0)
	//	Accept no non-text data (stop on first occurance).
	limit = 1;
else
if (limit < 0)
	//	Accept all non-text data bytes.
	limit = -1;
Non_Text_Limit = limit;
return this;
}

/**	Get the current limit for non-text data input.
<P>
	@return	The non-text data limit.
	@see	#Non_Text_Limit(int)
*/
public int Non_Text_Limit ()
{return Non_Text_Limit;}


/*==============================================================================
	Buffer Management.
*/
/**	Extend the string buffer with additional input characters.
<P>
	The procedure to extend the buffer has several steps:
<P><DL>
	<DT><B>Test if input filtering is enabled.</B>
		<DD>The {@link #Filter_Input() <CODE>Filter_Input</CODE>} test
		is done before reading any data so the Reader input position is
		not altered. The test method may need to read characters from
		the source at its current position.
	<DT><B>Slide the buffer forward.</B>
		<DD>If the {@link #Next_Location(long)
		<CODE>Next_Location</CODE>} is beyond the {@link
		#Buffer_Location() <CODE>Buffer_Location</CODE>} then the
		consumed contents of the buffer - i.e. from the beginning up
		to, but not including, the <CODE>Next_Index</CODE> - are
		deleted.
	<DT><B>Check for end of input.</B>
		<DD>The end of input occurs when the end of the Reader's data
		stream has been reached, or the amount of input has reached the
		{@link #Read_Limit(long) <CODE>Read_Limit</CODE>}. When this
		object was constructed from a String the end of input has been
		reached by definition. When the end of input has been reached
		the buffer can not be extended so the method returns false.
	<DT><B>Determine how much to extend the buffer.</B>
		<DD>The buffer will be extended by the lesser of the amount of
		free space in the internal character array or the amount from
		the current {@link #Total_Read() <CODE>Total_Read</CODE>} up to
		the <CODE>Read_Limit</CODE>. Of course, if there is no read
		limit then the former is always used. The internal character
		array, where characters read from the Reader are stored for
		checking before being transferred to the object's buffer, has a
		length of {@link #Size_Increment(int)
		<CODE>Size_Increment</CODE>}. However, during each read cycle
		less than the entire character array contents may be
		transferred to the buffer; the remainder is carried over in the
		array where new input data is added in the next read cycle.
		Thus the buffer will be extended by no more than
		<CODE>Size_Increment</CODE> characters.
	<DT><B>Read characters.</B>
		<DD>Characters are read from the Reader into an internal
		character array. Here they may be scanned for non-text data
		before being appended to the object's character buffer. This
		cycle continues until the amount to extend the buffer has been
		read or the end of the input stream is encounterd.
	<DT><B>Check for non-text data.</B>
		<DD>Characters read into the internal storage array may be
		scanned for a sequence of non-text data of the current maximum
		length. This check is only done if the {@link
		#Non_Text_Limit(int) non-text data threshold} is non-negative.
		Counting of sequential non-text data is continuous across buffer
		extensions and is only reset to zero when a text character is
		found. If the non-text data threshold is reached, then only the
		data preceeding the sequence is appended to the object's
		character buffer and the <CODE>Read_Limit</CODE> is reset to the
		corresponding position in the data stream. Shorter sequences of
		non-text data at the end of scanned input are also omitted from
		the transfer to the buffer, against the possibility of more
		non-text data from the next read, but the <CODE>Read_Limit</CODE>
		is not changed. Data not transferred to the buffer is retained in
		the internal array where it is added to in the next read cycle.
	<DT><B>Transfer characters to the buffer.</B>
		<DD>The data in the character array, from the beginning up to
		but not including any trailing non-text data (if non-text data
		checking is enabled), is appended to the object's character
		buffer.
	<DT><B>Filter the new characters.</B>
		<DD>If input filtering is enabled, the {@link
		#Filter_Input(int) <CODE>Filter_Input</CODE>} method is invoked
		on the new characters.
</DL><P>
	<B>Note</B>: Altering the Reader's input position in the character
	stream may have have unpredictable consequences. The impact on the
	operation of the <CODE>Filter_Input</CODE> method in particular
	could be fatal. Without input filtering, however, it is quite
	possible to move the Reader's input position (e.g. by
	<CODE>reset</CODE>, <CODE>skip</CODE>, or <CODE>read</CODE>
	operations) to suit application needs. It is important to keep in
	mind that any such alterations will go undetected, and the value of
	the <CODE>Total_Read</CODE> and source stream locations will
	continue to indicate the amount of data actually read and consumed
	from a virtually sequential input stream.
<P>
	@return true if more input is available; false if the end of
		input was reached.
	@throws	IOException	From the <CODE>read</CODE> method of the Reader.
	@throws	IndexOutOfBoundsException	If the buffer indexes are invalid.
	@see	#Filter_Input()
	@see	#Next_Location(long)
	@see	#Buffer_Location()
	@see	#Next_Index()
	@see	#Ended()
	@see	#Read_Limit(long)
	@see	#Total_Read()
	@see	#Size_Increment(int)
	@see	#Is_Text(char)
	@see	#Non_Text_Limit(int)
	@see	#Filter_Input(int)
*/
public boolean Extend ()
	throws IndexOutOfBoundsException, IOException
{
if ((DEBUG & DEBUG_BUFFER) != 0)
	System.out.println (">>> String_Buffer_Reader.Extend");

/*	>>> WARNING <<< Do the Filter_Input test BEFORE reading any
	data so the Reader input position is not altered for the first
	test (which, for the default test method, will read the next
	two input characters the first time it is used).
*/
boolean
	filter_input = Filter_Input ();

int
	next_index = Math.min (Next_Index (), End_Index ());
if (next_index > 0)
	{
	//	Free the consumed data.
	try
		{
		delete (0, next_index);
		}
	catch (StringIndexOutOfBoundsException exception)
		{
		throw new IndexOutOfBoundsException (Exception_Message
			(
			"Invalid Next_Index during buffer Extend!" + NL
			+ "  Buffer_Location = " + Buffer_Location + NL
			+ "    Next_Location = " + Next_Location
				+ " (index " + next_index + ')' + NL
			+ "     End_Location = " + End_Location ()
				+ " (index " + End_Index () + ')' + NL
			+ ((exception.getMessage () == null) ?
				"" : NL + exception.getMessage ()),
			Total_Read
			));
		}
	//	Update the location of the first character in the String_Buffer.
	Buffer_Location += next_index;
	}

if (Ended () ||
	Non_Text_Count >= Non_Text_Limit)
	{
	/*	The end of input has been reached.

		When the non-text data input threshold is first encountered
		after reading input (below) the read limit is set to the input
		location before the non-text data. So this condition suggests
		that the read limit must have been changed (user set the max
		up, the limit up, then the max down?). In any case, this is an
		end of input condition.
	*/
	if ((DEBUG & DEBUG_BUFFER) != 0)
		System.out.println
			("    Ended" + NL
			+"<<< String_Buffer_Reader.Extend");
	return false;
	}

/*	The curent end index before refilling the buffer
	where the new characters will start.
*/
int
	start_index = End_Index ();

//	Read more data.
int
	read_amount,
	amount_read;

/*	Determine how far to extend the buffer:
    the lesser of the free space in the Input_Array
	or the amount from the Total_Read up to the Read_Limit.

	The Input_Array.length should be the Size_Increment.
*/
read_amount = Input_Array.length - Input_Array_Amount;
if (Read_Limit != NO_READ_LIMIT)
	read_amount =
		(int)Math.min ((long)read_amount, Read_Limit - Total_Read);
if ((DEBUG & DEBUG_BUFFER) != 0)
	System.out.println
		("    String_Buffer_Reader.Extend: read amount = " + read_amount);

Read_Data:
while (read_amount > 0)
	{
	try
		{
		//	Append to the end of any remaining data.
		if ((amount_read = Character_Reader.read
				(Input_Array, Input_Array_Amount, read_amount)) < 0)
			{
			//	End of input encounterd.
			Read_Limit = Total_Read;
			if ((DEBUG & DEBUG_BUFFER) != 0)
				System.out.println
					("    Ended" + NL
					+"<<< String_Buffer_Reader.Extend");
			break;
			}
		}
	catch (IOException exception)
		{
		throw new IOException (Exception_Message
			(
			"While reading " + read_amount + " characters."
			+ ((exception.getMessage () == null) ?
				"" : NL + exception.getMessage ()),
			Total_Read
			));
		}
	//	Update the totals.
	Total_Read += amount_read;
	Input_Array_Amount += amount_read;
	if ((DEBUG & DEBUG_BUFFER) != 0)
		System.out.println
			("    String_Buffer_Reader.Extend:"
			+"    read amount = " + read_amount + NL
			+"    amount read = " + amount_read + NL
			+"       buffered = " + Input_Array_Amount + NL
			+"     total read = " + Total_Read);
	read_amount -= amount_read;	//	The amount left to read.

	if (Non_Text_Limit >= 0)
		{
		/*	Input stops when a sequence of Non_Text_Limit data bytes are
			detected.

			Since this class is managing a String buffer, it is assumed
			that non-text data from the input should not be transferred
			into the buffer; i.e. the data is not valid String
			characters. However, to allow for possible input filter
			requirements (e.g. binary record size values), a sequence of
			non-text data less than Non_Text_Limit is allowed into the
			character buffer. Note that the sequence of non-text data may
			start before the end of the array, so the count of non-text
			data is maintained across buffer extends. The accumulator of
			non-text data bytes encountered will not be reset unless a
			text data byte is seen.

			Note that a Non_Text_Limit value of 0 means that any non-text
			data will qualifies as the end of input.

			Trailing non-text data is not transferred from the internal
			character array into the object buffer. This ensures that a
			max non-text sequence will never be placed in the buffer.
		*/
		next_index = Non_Text_Count;	//	Skip previously scanned data.
		while (next_index < Input_Array_Amount)
			{
			if (Is_Text (Input_Array[next_index++]))
				Non_Text_Count = 0;
			else if (++Non_Text_Count >= Non_Text_Limit)
				{
				//	Max non-text data; stop before this sequence.
				Read_Limit =
					Total_Read - Input_Array_Amount
						+ next_index - Non_Text_Count;
				read_amount = 0;
				if ((DEBUG & DEBUG_BUFFER) != 0)
					System.out.println
						("    String_Buffer_Reader.Extend: max non-text ("
							+ Non_Text_Limit
							+ ") at index " + (next_index - 1) + NL
						+"    Read limit = " + Read_Limit);
				break;
				}
			}
		//	Leave trailing non-text data for the next buffer extend.
		next_index -= Non_Text_Count;
		}
	else
		next_index = Input_Array_Amount;

	if ((DEBUG & DEBUG_BUFFER) != 0)
		System.out.println
			("    String_Buffer_Reader.Extend: append (Input_Array, 0, "
				+ next_index + ");");
	append (Input_Array, 0, next_index);
	Input_Array_Amount -= next_index;
	//	Move the remaining data to the front of the buffer.
	Flush (next_index);
	}

if (filter_input && start_index < End_Index ())
	//	Apply the input filter to the new data.
	Filter_Input (start_index);

if ((DEBUG & DEBUG_BUFFER) != 0)
	System.out.println
		("    Buffer_Location = " + Buffer_Location + NL
		+"      Next_Location = " + Next_Location + NL
		+"       End_Location = " + End_Location () + NL
		+"         Read_Limit = " + Read_Limit + NL
		+"         total read = " + (End_Index () - start_index) + NL
		+"         Total_Read = " + Total_Read + NL
		+"              Ended = " + Ended () + NL
		+"<<< String_Buffer_Reader.Extend");
return (! Ended ());
}

/**	Flush data from the front of the Input_Array.
<p>
	The Input_Array_Amount of data from the start index is copied to
	the front of the Input_Array.
<p>
	@param	start	The index of the first valid datum in the Input_Array.
*/
private void Flush
	(
	int		start
	)
{
if (Input_Array_Amount <= 0 ||
	start <= 0 ||
	start >= Input_Array.length)
	return;
int
	end = start + Input_Array_Amount;
if (end > Input_Array.length)
	end = Input_Array.length;

if ((DEBUG & DEBUG_BUFFER) != 0)
	System.out.println
		(">-< String_Buffer_Reader.Flush: Moving " + (end - start)
			+ " characters from " + start + " to the buffer front.");
int
	index = 0;
while (start < end)
	Input_Array[index++] = Input_Array[start++];
}


/**	Tests if a character is considered to be text.
<P>
	Text is evaluated against the ASCII code set and includes all
	printable characters - values from 0x20 (' ') to 0x7E ('~')
	inclusive - plus the usual format control characters - HT, LF, VT,
	FF, and CR (0x9 to 0xD inclusive).
<P>
	@param	character	The character to be tested.
	@return	true if the character is text; false otherwise.
*/
protected boolean Is_Text
	(
	char		character
	)
{
return ((character >= 0x20 && character <  0x7F) ||	//	Printable text
		(character >= 0x9  && character <= 0xD));	//	HT,LF,VT,FF,CR
}


//	Special variables for handling sized record format files:

/**	The "sized record" problem:
<P>
	Some files use a record structure which is composed of a leading
	size value (16-bits, LSB first) followed by size bytes of data.
	This is like Pascal strings (and just as cumbersome). Unfortunately,
	sized records add another layer of processing logic for the
	tolerant design goal.
<P>
	The solution, here, is to first test for the size value in the
	first two bytes read and, if present, flag this condition so that
	string buffer refills will be cleaned up. The test for the presence
	of the size value depends on two assumptions: 1) the file is
	currently positioned at the beginning of a size value, and 2) the
	MSB of the size value maps to an unprintable character (i.e. < 32).
	The first assumption depends on the user. The second assumption
	depends on the first record having less than 8k bytes.
<P>
	The string buffer will continue to be processed as usual, but the
	string buffer will be cleaned up after each read by replacing the
	record size values with LINE_BREAK characters. To keep track of
	where the record size values are located in the string buffer, the
	offset from "last" to the next size value is recorded in
	"Sized_Record". Since this is also used as a flag for the sized
	records format of the data, this value must not be zero. Therefore,
	if this value would be 0, it is set to -1 instead. It's possible
	for a size value to be split across a buffer read. In this case the
	recorded value is set to -2, the first (LSB) byte of the size value
	is saved in the String_Buffer "LSB_Character" entry, and the first
	byte of the next read is known to be the value of the next (MSB) byte
	(see the padded record exception below). Thus the interpretation of
	Sized_Record is as follows:
<P>
<DL>
<DT>&npsp;0 - Not a sized record file.
<DT>>0 - Offset from last to next size value (see padded exception).
<DT>-1 - Offset is 0.
<DT>-2 - Next char after last is second byte (MSB) of size value.
	<DD>The value of the first byte is stored in "LSB_Character".
<DT>-3 - Check for sized records hasn't been done yet.
</DL>
<P>
	While one might think that because all sized records are guaranteed
	to be of even size - by padding, when needed, which adds it's own
	problems - that an even sized buffer would prevent the possibility
	of a split size value, the unpredictable nature of the statements
	and the manner of processing in a sliding window buffer could
	result in the buffer becoming non-aligned with respect to file
	records. Therefore, the possibility of split size values must be
	taken into account.
<P>
	The record size value is the number of characters in the next
	record (which many, of course, not be a complete statement). This
	value does not include the two chars of the size value itself, or a
	possible zero valued char that is appended to the record whenever
	its size is odd. This pad byte presents the problem of prematurely
	terminating the string presumed to contain all of the statements.
	Therefore it is plugged with a space character whenever it occurs.
	To keep track of odd-sized, and thus padded records, across buffer
	refills, a "padded" flag entry is set accordingly at the end of
	cleaning up each new buffer refill. This value (0 if no pad byte, 1
	otherwise) allows the Sized_Record value to be adjusted
	accordingly.
*/
private int
	Sized_Record				= CHECK_FOR_SIZED_RECORDS,
	Padding						= 0;
private char
	LSB_Character				= 0;
private static final int
	NO_FILTERING				= 0,
	OFFSET_ZERO					= -1,
	NEXT_AFTER_LAST_IS_MSB		= -2,
	CHECK_FOR_SIZED_RECORDS		= -3;
private static final char[]
	LINE_BREAK					= {'\r', '\n'};


/**	Tests if the character source will be filtered on input.
<P>
	Some files - notably binary data files produced by DEC VMS systems
	- use a record structure composed of a leading binary size value
	(16-bits, LSB first) followed by size bytes of data. This is like
	Pascal strings (and just as cumbersome). In addition, if the size
	is odd then a zero-valued pad byte will be appended to the record
	(to force word alignment for all size values). Binary record size
	values are detected by testing the second (MSB) character obtained
	from the Reader (String character sources are not tested) for a
	value less than 32 (' ') but not 9 (HT), 10 (NL) or 13 (CR). This
	test for the presence of the binary size value depends on a few
	assumptions: 1) the Reader is positioned at the beginning of a
	potential binary size value (this is probably the beginng of the
	file or stream), 2) the second character of normal (unfiltered)
	input will not be unprintable, and 3) the first binary size value
	is less than 8k and outside the ranges (2304-2815) and (3328-3583).
	<B>N.B.</B>: This last assumption is rather risky, of course; but
	these records are very likely to be less than 2k in size. Of
	course, this also requires that any character encoding preserve the
	values of bytes in the stream.
<P>
	During character input a source that has sized records will be
	filtered: The record size bytes will be replaced with a LINE_BREAK
	(CR-LF) sequence, and any pad bytes will be replaced with a space
	character, thus providing a consistent character stream.
<P>
	When a String_Buffer_Reader is created its input filtering status
	is untested. If the input filtering status is untested when this
	method is used then the next two characters are read (if the source
	is a Reader) unconditionally - i.e. regardless if the source has
	already been read or characters otherwise put in the buffer - and
	they are tested. The results of the input source test update the
	input filtering status to either filtered or unfiltered, so
	subsequent use of this method will not repeat the input source test
	unless the input filtering status is reset to the initial untested
	state (which may be done by the {@link #Filter_Input(boolean)
	<CODE>Filter_Input (boolean)</CODE>} method.
<P>
	<B>Note</B>: To accommodate binary record size values the
	<CODE>Non_Text_Limit</CODE> threshold must be greater than 3
	(for a possible pad byte followed by two record size bytes).
<P>
	@return	true if input filtering will be applied, false otherwise.
	@throws	IOException	If the Reader could not be read during the
		initial check.
	@see	#Filter_Input(boolean)
	@see	#Filter_Input(int)
	@see	#Non_Text_Limit(int)
*/
public boolean Filter_Input ()
	throws IOException
{
if (Sized_Record == CHECK_FOR_SIZED_RECORDS)
	{
	if (String_Source ())
		return false;
	if ((DEBUG & DEBUG_BUFFER) != 0)
		System.out.println (">>> String_Buffer_Reader.Filter_Input");
	int
		count = 2,
		amount = 0;
	while (amount < 2)
		{
		try
			{
			count = 2 - amount;
			if ((count = Character_Reader.read
					(Input_Array, Input_Array_Amount + amount, count)) < 0)
				{
				//	End of input encounterd.
				Read_Limit = Total_Read + amount;
				break;
				}
			}
		catch (IOException exception)
			{
			throw new IOException (Exception_Message
				(
				"While reading " + count + " characters."
				+ ((exception.getMessage () == null) ?
					"" : NL + exception.getMessage ()),
				Total_Read + amount
				));
			}
		amount += count;
		}
	Total_Read += amount;

	char
		character = Input_Array[Input_Array_Amount + 1];
	if (amount >= 2 &&
		Input_Array[Input_Array_Amount + 1] < 0x20 &&	// ' '
		Input_Array[Input_Array_Amount + 1] != 0x09 &&	// <HT>
		Input_Array[Input_Array_Amount + 1] != 0x0a &&	// <NL>
		Input_Array[Input_Array_Amount + 1] != 0x0d		// <CR>
		)
		{
		//	Sized records detected.
		if ((DEBUG & DEBUG_BUFFER) != 0)
			System.out.println
				("    Initial check - true");
		//	Set the offset to the location of this binary record size data.
		if ((Sized_Record = Input_Array_Amount) == 0)
			Sized_Record = OFFSET_ZERO;
		}
	else
		{
		//	Sized records were not detected.
		if ((DEBUG & DEBUG_BUFFER) != 0)
			System.out.println
				("    Initial check - false");
		Sized_Record = NO_FILTERING;
		}
	//	Update the amount of data in the array.
	Input_Array_Amount += amount;
	if ((DEBUG & DEBUG_BUFFER) != 0)
		System.out.println ("<<< String_Buffer_Reader.Filter_Input: "
			+ (Sized_Record != NO_FILTERING));
	}
return Sized_Record != NO_FILTERING;
}

/**	Enable or disable input filtering.
<P>
	If input filtering is to be allowed and it is currently not enabled, the
	input filtering status is reset to untested. If input filtering is not
	to be allowed it is unconditionally disabled.
<P>
	@param	allow	true if input filtering is allowed; false otherwise.
	@return	This String_Buffer_Reader.
	@see	#Filter_Input()
	@see	#Filter_Input(int)
*/
public String_Buffer_Reader Filter_Input
	(
	boolean		allow
	)
{
if (allow)
	{
	if (Sized_Record == NO_FILTERING)
		Sized_Record = CHECK_FOR_SIZED_RECORDS;	//	Input test required.
	}
else
	Sized_Record = NO_FILTERING;
return this;
}

/**	Filters the current contents of the character buffer, starting at the
	specified index and continuing to the end of the buffer contents.
<P>
	The details of the filtering implemented here is described by the
	{@link #Filter_Input() <CODE>Filter_Input</CODE>} test method.
<P>
	@param	index	The index in the character buffer where filtering
		is to start. Nothing is done if the index is invalid (i.e. < 0
		or >= the end of characters in the buffer).
	@throws	IndexOutOfBoundsException	An index for record size bytes
		or padding is invalid. Assuming that the filtering algorithm is
		bug free (8^]), then the input stream has been corrupted,
		possible due to inappropriate changes to its position by the
		application. It is very likely in this case that previous
		character buffer contents have been mangled as well.
	@throws	IOException	This implmentation doesn't do anything that
		could generate an IOException. However, the method is marked
		this way so sbuclass implementations that need to use the
		Reader may throw this exception.
	@see	#Filter_Input()
	@see	#End_Index()
	@see	#Record_Size(char, char)
*/
protected void Filter_Input
	(
	int			index	//	Previous end index.
	)
	throws IndexOutOfBoundsException, IOException
{
if ((DEBUG & DEBUG_BUFFER) != 0)
	System.out.println
		(">>> String_Buffer_Reader.Filter_Input: from index " + index);
if (index < 0 ||
	index >= End_Index ())
	return;
int
	record_size = 0;

try
	{
	//	Pick up where we left off.
	if (Sized_Record > 0)
		record_size = Sized_Record;
	else if (Sized_Record == OFFSET_ZERO)
		record_size = 0;
	else if (Sized_Record == NEXT_AFTER_LAST_IS_MSB)
		{
		/*	Split record size value.
			Reassemble it and fill the MSB hole.
		*/
		record_size = Record_Size (LSB_Character, charAt (index));
		setCharAt (index, LINE_BREAK[1]);
		}

	//	Plug the holes in the sized records.
	for (index += record_size;	//	Next (pad and) record size bytes.
		(index + Padding + 1) < End_Index ();
		 index += record_size)
		{
		if (Padding > 0)
			{
			//	Patch the null pad char with a space.
			if ((DEBUG & DEBUG_BUFFER) != 0)
				System.out.println
					("    String_Buffer_Reader.Filter_Input: pad at " + index);
			setCharAt (index++, ' ');
			}
		record_size = Record_Size (index);
		if ((DEBUG & DEBUG_BUFFER) != 0)
			System.out.println
				("    String_Buffer_Reader.Filter_Input: record size "
					+ record_size + " at " + index
					+" (next size at "
					+ (index + 2 + record_size + (record_size % 2)) + ")");
		Padding = record_size % 2;	//	Is there a pad?
		//	Replace the 2 record size chars with LINE_BREAK chars.
		setCharAt (index++, LINE_BREAK[0]);
		setCharAt (index++, LINE_BREAK[1]);
		}

	//	Check for possible trailing pad byte.
	if (index < End_Index () &&
		Padding > 0)
		{
		if ((DEBUG & DEBUG_BUFFER) != 0)
			System.out.println
				("    String_Buffer_Reader.Filter_Input: trailing pad at "
					+ index);
		setCharAt (index++, ' ');
		}

	//	Check for split size value.
	if ((index + 1) == End_Index ())
		{
		Sized_Record = NEXT_AFTER_LAST_IS_MSB;
		LSB_Character = charAt (index);
		if ((DEBUG & DEBUG_BUFFER) != 0)
			System.out.println
				("    String_Buffer_ReaderFilter_Input: split size at "
					+ index);
		setCharAt (index, LINE_BREAK[0]);
		}

	/*	Set the offset from the end index to where this process
		will pick up again after the next buffer refill.
	*/
	else if ((Sized_Record = index - End_Index ()) == 0)
		Sized_Record = OFFSET_ZERO;
	}
catch (IndexOutOfBoundsException exception)
	{
	throw new IndexOutOfBoundsException (Exception_Message
		(
		"Lost record sizing bytes for Filter_Input!" + NL
		+ "         Location = " + Location (index)
			+ " (index " + index + ')' + NL
		+ "      Record size = " + record_size + NL
		+ "          Padding = " + Padding + NL
		+ "  Buffer location = " + Buffer_Location + NL
		+ "    Buffer length = " + length () + NL
		+ "  Buffer capacity = " + capacity () + NL
		+ ((exception.getMessage () == null) ?
			"" : NL + exception.getMessage ()),
		Total_Read
		));
	}
if ((DEBUG & DEBUG_BUFFER) != 0)
	System.out.println
		("<<< Filter_Input: Sized_Record = " + Sized_Record);
}


/**	Converts a 16-bit (in 2 sequential chars), LSB-first value at the
	buffer index to a record size value.
<P>
	This method is used by the input filter.
<P>
	@param	index	The index of the binary record size bytes in the buffer.
	@return	A record size value.
	@see	#Record_Size(char, char)
*/
protected int Record_Size
	(
	int		index
	)
{
return Record_Size
	(
	charAt(index),
	charAt(index + 1)
	);
}

/**	Converts two char values to a single int value, assuming that the
	first char value is the LSB and the second the MSB of a 16-bit
	integer.
<P>
	Bits 0-7 of the MSB are shifted left 8 bits and ORed with bits 0-7
	of the LSB to form the integer value.
<P>
	@param	LSB	The Least Significant Byte of the 16-bit record size value.
	@param	MSB	The Most Significant Byte of the 16-bit record size value.
	@return	A record size value.
	@see	#Filter_Input()
*/
protected static int Record_Size
	(
	char	LSB,
	char	MSB
	)
{
return (int)(((MSB & 0xFF) << 8) | (LSB & 0xFF));
}


/*==============================================================================
	Utility functions:
*/
/*	String_Buffer-like methods using virtual stream locations.

	The location arguments are not limited to the current contents of
	the character buffer; the buffer will be extended as needed to
	bring the specified location into the buffer, until the end of
	input data is reached.

	>>> NOTE <<<
	Methods that search will ingest all available data if the search is
	not satisfied. There is thus a big memory consumption penalty for
	failed searches.
*/
/**	A character that should not occur in any valid String.
<P>
	@see	CharacterIterator#DONE
*/
public static final char
	INVALID_CHARACTER			= CharacterIterator.DONE;

/**	Gets the character at the specified location in the virual stream.
<P>
	The current contents of the character buffer will be automatically
	extended to the specified location.
<P>
	@param	location	The location from which to get a character.
	@return	The char found at the location. If the location is invalid
		for any reason then the <CODE>INVALID_CHARACTER</CODE> is
		returned.
*/
public char Char_At
	(
	long		location
	)
{
try {return charAt (get_index (location));}
catch (IndexOutOfBoundsException exception) {return INVALID_CHARACTER;}
catch (IOException exception) {return INVALID_CHARACTER;}
}


/**	Gets the substring including the characters from the start location
	up to, but not including, the end location in the virtual stream.
<P>
	The current contents of the character buffer will be automatically
	extended to include the specified locations. If the substring
	extends beyond the end of input location, then that portion up to
	the end of input will be returned. If both start and end locations
	are beyond the end of input, then an empty String will be returned.
<P>
	@param	start	The location of the first character of the substring.
	@param	end		The location of end of the substring (the location
		immediately following the last character in the substring).
	@return	The String from the start location, inclusive, to the end
		location, exclusive.
	@throws	IndexOutOfBoundsException	A location is before the
		<CODE>Buffer_Location</CODE>, or start is greater than end.
	@throws	IOException	While reading characters to extend the buffer.
*/
public String Substring
	(
	long		start,
	long		end
	)
	throws IndexOutOfBoundsException, IOException
{
int
	first,
	last;
/*	Make sure the end of the string is in the buffer first,
	in case this causes the start location move in the buffer
	as a result of sliding forward.
*/
last = get_index (end);
first = get_index (start);
return substring (first, last);
}


/**	Skips over a character set in the virtual stream.
<P>
	Starting at the specified location find the location of the next
	character that is not in the skip string. The skip string is not a
	pattern; i.e. a character in the virtual stream that matches any
	character in the skip string is skipped during the search. The
	return location will be for a character that does not occur in the
	skip string.
<P>
	The character buffer will be extended until a non-skip character is
	found. <B>Note</B>: The character buffer will be extended to
	include all available input if all characters from the beginning
	location on are in the skip String.
<P>
	@param	location	The location from which to start the search.
	@param	skip		The String containing characters to be skipped.
	@return	The location of the next character not in the skip String.
		This will be the <CODE>End_Location</CODE> if the end of input
		data is reached (i.e. all characters from the start location on
		are in the skip String).
	@throws	IndexOutOfBoundsException	The location is before the
		<CODE>Buffer_Location</CODE>.
	@throws	IOException	While reading characters to extend the buffer.
*/
public long Skip_Over
	(
	long		location,
	String		skip
	)
	throws IndexOutOfBoundsException, IOException
{
while (Is_End (location = Location (skip_over (get_index (location), skip))) &&
		! Ended ());
return location;
}

/**	Skips until a member of the character set is found in the virtual
	stream.
<P>
	Starting at the specified location find the location of the next
	character that is in the find string. The find string is not a
	pattern; i.e. a character in the virtual stream that matches any
	character in the find string satisfies the search. The return
	location will be for a character that occurs in the find string.
<P>
	The character buffer will be extended until a find character is
	found. <B>Note</B>: The character buffer will be extended to
	include all available input if all characters from the beginning
	location on are not in the find String.
<P>
	@param	location	The location from which to start the search.
	@param	find		The String containing characters to be found.
	@return	The location of the next character also in the find String.
		If the end of input data is reached (i.e. all characters from
		the start location on are not in the find String), then -1 is
		returned.
	@throws	IndexOutOfBoundsException	The location is before the
		<CODE>Buffer_Location</CODE>.
	@throws	IOException	While reading characters to extend the buffer.
*/
public long Skip_Until
	(
	long		location,
	String		find
	)
	throws IndexOutOfBoundsException, IOException
{
while ((location = Location (skip_until (get_index (location), find)))
			< Buffer_Location &&
		! Ended ())
		location = End_Location ();
return (location < Buffer_Location) ? -1 : location;
}


/**	Finds the next location of the pattern String in the virtual stream.
<P>
	Starting at the specified location find the location of the next
	occurance of the substring that matches the pattern String.
<P>
	The character buffer will be extended until the pattern substring
	is found. <B>Note</B>: The character buffer will be extended to
	include all available input if the pattern can not be found.
<P>
	@param	location	The location from which to start the search.
	@param	pattern		The String to be found as a substring of the
		virtual stream.
	@return	The location of the beginning of the pattern substring. If
		the end of input data is reached without finding the pattern
		String, then -1 is returned.
	@throws	IndexOutOfBoundsException	The location is before the
		<CODE>Buffer_Location</CODE>.
	@throws	IOException	While reading characters to extend the buffer.
*/
public long Location_Of
	(
	long		location,
	String		pattern
	)
	throws IndexOutOfBoundsException, IOException
{
while ((location = Location (index_of (get_index (location), pattern)))
			< Buffer_Location &&
		! Ended ())
	{
	//	Move back to allow for a pattern match over the previous buffer end.
	location =
		Math.max (End_Location () - pattern.length () + 1, 0);
	Extend ();
	}
return (location < Buffer_Location) ? -1 : location;
}

/**	Finds the next location of the character in the virtual stream.
<P>
	Starting at the specified location find the next location of the
	specified character.
<P>
	The character buffer will be extended until the specified character
	is found. <B>Note</B>: The character buffer will be extended to
	include all available input if the character can not be found.
<P>
	@param	location	The location from which to start the search.
	@param	character	The character to be found.
	@return	The location of the next occurance of the character in the
		virtual stream. If the end of input data is reached without
		finding the character, then -1 is returned.
	@throws	IndexOutOfBoundsException	The location is before the
		<CODE>Buffer_Location</CODE>.
	@throws	IOException	While reading characters to extend the buffer.
*/
public long Location_Of
	(
	long		location,
	char		character
	)
	throws IndexOutOfBoundsException, IOException
{
while ((location = Location (index_of (get_index (location), character)))
			< Buffer_Location &&
		! Ended ())
	 location = End_Location ();
return (location < Buffer_Location) ? -1 : location;
}


/**	Tests if the pattern equals the substring starting at the location.
<P>
	The character buffer will be extended as needed to include a
	substring starting at the specified location that is as long as the
	pattern String.
<P>
	@param	location	The location of the substring to compare.
	@param	pattern		The String to compare against the substring.
	@return	true if the substring equals the pattern String; false
		otherwise.
	@throws	IndexOutOfBoundsException	The location is before the
		<CODE>Buffer_Location</CODE>.
	@throws	IOException	While reading characters to extend the buffer.
*/
public boolean Equals
	(
	long		location,
	String		pattern
	)
	throws IndexOutOfBoundsException, IOException
{
//	Make sure the pattern doesn't overlap the end of the buffer.
long
	end_of_pattern = location + pattern.length () - 1;
while (end_of_pattern >= End_Location () &&
		Extend ());
if (end_of_pattern >= End_Location ())
	return false;
return equals (get_index (location), pattern);
}


//------------------------------------------------------------------------------
/**	Gets the index in the buffer for the location.
<P>
	Makes sure the location has been read into the buffer. A location
	at or beyond the end of data input is set to the end location.
<P>
	Note: As a special case a location of -1 is taken to mean the
	location of the next input character (i.e. End_Location ()). This
	accommodates sequential searches on the buffer where the return
	value is -1 when the search fails.
<P>
	@param	location	The virtual stream location to get into the buffer.
	@return	The buffer index of the effective location.
*/
private int get_index
	(
	long		location
	)
	throws IndexOutOfBoundsException, IOException
{
if (location < Buffer_Location)
	throw new IndexOutOfBoundsException (Exception_Message
		(
		"String_Buffer_Reader: Can't get to location " + location
		+ " with the String_Buffer located at " + Buffer_Location + ".",
		location
		));
if (location == -1)
	location = End_Location ();
while (location >= End_Location () && Extend ());
if (location >= End_Location ())
	return Index (End_Location ());
return Index (location);
}


/*------------------------------------------------------------------------------
	Exception message
*/
/**	Provide a standard exception message.
<P>
	The first line of the result message will be the class (@link #ID}.
	The second line will be the user specified message. The third line
	will be "At data input location <location>.", where <location> is
	the value of the specified location.
<P>
	@param	message	The user message String.
	@param	location	The virtual stream location where the
		exception occurred.
	@return	A standard exception message String.
*/
private static String Exception_Message
	(
	String		message,
	long		location
	)
{
return
	ID + NL
	+ message + NL
	+ "At data input location " + location + ".";
}


}	//	End of class
