/*	Memory_Chart

PIRL CVS ID: Memory_Chart.java,v 1.5 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import	javax.swing.JComponent;
import	javax.swing.UIManager;
import	javax.swing.event.ChangeEvent;
import	javax.swing.event.ChangeListener;
import	java.awt.Font;
import	java.awt.Dimension;
import	java.awt.geom.Rectangle2D;
import	java.awt.Graphics;


/**	A <i>Memory_Chart</i> provides a self-updating graphical display of
	the Java runtime environment (JRE) memory usage.
<p>
	The JRE reports three categories of memory: the total amount of
	{@link #Available_Memory() available memory} in the JRE heap space,
	the amount of {@link #Allocated_Memory allocated memory} used by
	objects, and the amount of {@link #Free_Memory() free memory} that is
	no longer in use by objects which can be garbage collected.
<p>
	Memory use over time is stored in a Memory_History that samples the
	JRE memory categories at a user specified rate and stores the values
	in history arrays up to some maximum number of values. When the
	maximum number of samples has been accumulated the next sample causes
	the oldest value to be dropped before the new value is added.
	Obtaining a sample also results in the component display to be
	updated by displaying all current memory history values as a
	horizontally time-ordered (oldest on the left) chart with one
	vertical bar per sample. Each sample bar is divided into three
	sections; from top to bottom: the amount of unallocated memory
	({@link #Available_Memory() available memory} minus {@link
	#Allocated_Memory() allocated memory}) in blue, the amount of {@link
	#Free_Memory() free memory} in green, and the amount of memory in use
	(@link #Allocated_Memory() allocated memory} minus {@link
	#Free_Memory() free memory}) in red. The visual effect is a memory
	use chart that automatically scrolls from right to left as new samples
	are acquired.
<p>
	Below the memory use bars a {@link #Seconds_Scale(boolean) seconds
	scale} is optionally displayed with short tick marks every fifth bar
	and long tick marks every tenth bar. Also, if the sampling rate is
	more often than every 20 seconds a thick long mark is provided for
	each minute. The total duration, in seconds, of the history chart is
	written at the left end of the seconds scale. If annotations are
	enabled and the sampling rate is not once per second the scale
	multiplication factor is shown.
<p>
	To the right of the chart {@link #Annotation(boolean) annotation} may
	be optionally displayed. This will list the most recent memory sample
	using the same values and colors as used in the chart bars.
<p>
	Memory sampling may be stopped and restarted. When sampling is
	restarted the memory history is cleared. The sampling rate may be
	changed at any time which will also clear the current memory history.
<p>
	The component will automatically manage the Memory_History by
	ensuring that there is always enough memory history for the chart
	display width.
<p>
	<b>N.B.</b>: The Memory_History sampling can not be guaranteed to
	always occur at the selected rate; other activity on the system may
	prevent a sampling timer event from occurring at the intended time.
	Nevertheless, there will still be the correct number of samples
	generated for the overall amount of elapsed time. To achieve this
	there may be occasional bursts  of catch-up sampling events that
	will result in memory values that have not been sampled at the
	expected time; but no sampling events will be dropped. Thus overall
	the memory chart will be a reasonable reflection of the JRE memory use
	profile with any particular sample occurring as close to the expected
	time as the system allows.
<p>
	@author		Bradford Castalia - idaeim
	@version	1.5
	@see	Memory_History
*/
public class Memory_Chart
	extends JComponent
	implements ChangeListener
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Viewers.Memory_Chart (1.5 2012/04/16 06:22:58)";

/**	The initial size of the component.
*/
public static final Dimension
	DEFAULT_SIZE			= new Dimension (200, 70);

/**	The text font that will be used to provide chart annotations.
*/
public static final Font
	ANNOTATION_FONT			= new Font ("Monospaced", Font.PLAIN, 10);

//!	Should the memory values annotations be displayed?
private boolean
	Annotation				= true;
private int
	Annotation_Area_Width	= 0,
	Annotation_Text_Height	= 0;

//!	Working values until the real values can be determined.
private static final int
	DEFAULT_ANNOTATION_AREA_WIDTH	= 33,
	DEFAULT_ANNOTATION_TEXT_HEIGHT	= 11;

//!	Should the seconds scale be displayed?
private boolean
	Seconds_Scale			= true;
private int
	Seconds_Scale_Height	= 0;

//!	Working value until the real value can be determined.
private static final int
	DEFAULT_SECONDS_SCALE_HEIGHT	= DEFAULT_ANNOTATION_TEXT_HEIGHT;

//!	The memory usage history model.
private Memory_History
	History					= null;


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_UI			= 1 << 1,
	DEBUG_MODEL			= 1 << 2,
	DEBUG_GEOMETRY		= 1 << 3,
	DEBUG_ALL			= -1,

	DEBUG		    	= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Memory_Chart that samples at a specified rate.
<p>
	@param	sampling_rate	The number of seconds between memory samples.
		A sampling rate of zero constructs an inactive chart.
*/
public Memory_Chart
	(
	int		sampling_rate
	)
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println (">>> Memory_Chart.Constructor");
UIManager.put ("Memory_ChartUI", "Memory_Chart_UI");
if (sampling_rate < 0)
	sampling_rate = 0;
History = new Memory_History
	(DEFAULT_SIZE.width - Annotation_Area_Width (), sampling_rate);
Dimension
	dimension = new Dimension (DEFAULT_SIZE);
if (Annotation_Text_Height != 0)
	//	Allow for four annotation lines and the seconds scale.
	dimension.height = Annotation_Text_Height * 5;
History.addChangeListener (this);
setMinimumSize   (dimension);
setPreferredSize (dimension);
setUI (new Memory_Chart_UI ());
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println ("<<< Memory_Chart.Constructor");
}

/**	Construct an idle Memory_Chart.
<p>
	A {@link #Rate(int) sampling rate} must be set before the chart will
	become active.
*/
public Memory_Chart ()
{this (0);}

/*==============================================================================
	Geometry
*/
/**	Enable or disable the display of memory values annotation.
<p>
	When enabled an area on the right side of the memory chart will be
	used to display, from top to bottom: the total amount of {@link
	#Available_Memory() available memory} in black, the amount of
	unallocated memory ({@link #Available_Memory() available memory}
	minus {@link #Allocated_Memory() allocated memory}) in blue, the
	amount of {@link #Free_Memory() free memory} ({@link
	#Allocated_Memory() allocated} but not used by any objects; i.e.
	subject to garbage collection) in green, and the amount of memory in
	use (@link #Allocated_Memory() allocated memory} minus {@link
	#Free_Memory() free memory}) in red. Each value is in the range
	0-1023 with a magnitude suffix character - "K" for kilobytes,
	"M" for megabytes, "G" for gigabytes - as appropriate.
<p>
	@param	enabled	If true (the default) the annotation display will be
		enabled. If false, annoation will not be displayed.
	@return	This Memory_Chart.
	@see	#Annotation_Area_Width()
*/
public Memory_Chart Annotation
	(
	boolean	enabled
	)
{
if (Annotation != enabled)
	{
	boolean
		annotation = Annotation;
	Annotation = enabled;
	firePropertyChange ("Annotation", annotation, Annotation);
	repaint ();
	}
return this;
}

/**	Check if the annotation display is enabled.
<p>
	@return	true if the annotation display is enabled; false otherwise.
*/
public boolean Annotation ()
{return Annotation;}

/**	Get the width of the annotation display area.
<p>
	The annotation display occupies the right end of the component area.
	The value will be zero if the {@link #Annotation(boolean) annotation}
	display is not enabled. If the appropriate size can not be determined
	from the {@link #ANNOTATION_FONT annotation font} (because a
	component rendering graphics device has not yet been assigned) the
	width will be a guesstimate based on a typical display device of
	approximately 100 pixels per inch.
<p>
	@return	The width, in pixels, of the a annotation display area,
		or zero if annotation is disabled.
*/
public int Annotation_Area_Width ()
{
int
	value = 0;
if (Annotation)
	{
	if (Annotation_Area_Width == 0)
		{
		Reset_Geometry ();
		value = (Annotation_Area_Width == 0) ?
			DEFAULT_ANNOTATION_AREA_WIDTH :
			Annotation_Area_Width;
		}
	else
		value = Annotation_Area_Width;
	}
return value;
}

/**	Get the height of the annotation text.
<p>
	If the appropriate size can not be determined from the {@link
	#ANNOTATION_FONT annotation font} (because a component rendering
	graphics device has not yet been assigned) the text height will be a
	guesstimate based on a typical display device of approximately 100
	pixels per inch.
<p>
	@return	The height, in pixels, of the a annotation text.
*/
public int Annotation_Text_Height ()
{
if (Annotation_Text_Height == 0)
	Reset_Geometry ();
return (Annotation_Text_Height == 0) ?
	DEFAULT_ANNOTATION_TEXT_HEIGHT :
	Annotation_Text_Height;
}

/**	Enable or disable the display of the seconds scale.
<p>
	When enabled the seconds scale an area at the bottom of the memory
	chart will be used to display tick marks to aid in counting the
	number of memory samples in the chart. Short tick marks will occur
	every fifth sample measured from the most recent sample on the right
	end of the chart. Long tick marks will occur every tenth sample. If
	the sampling rate is more often than every 20 seconds a thick long
	mark is provided for each minute in the sampling history. The total
	duration, in seconds, of the history chart is written at the left end
	of the seconds scale when there is sufficient space. If {@link
	#Annotation(boolean) annotation} is enabled "secs" will be written to
	the right of the seconds scale in the annotation area. However, if
	the sampling rate is not once per second the annotation is the scale
	multiplication factor preceeded by "x" and followed by " s".
<p>
	@param	enabled	If true (the default) the seconds scale display will be
		enabled. If false, the seconds scale will not be displayed.
	@return	This Memory_Chart.
	@see	#Seconds_Scale_Height()
*/
public Memory_Chart Seconds_Scale
	(
	boolean	enabled
	)
{
if (Seconds_Scale != enabled)
	{
	boolean
		seconds_scale = Seconds_Scale;
	Seconds_Scale = enabled;
	firePropertyChange ("Seconds_Scale", seconds_scale, Seconds_Scale);
	repaint ();
	}
return this;
}

/**	Check if the seconds scale display is enabled.
<p>
	@return	true if the seconds scale display is enabled; false otherwise.
*/
public boolean Seconds_Scale ()
{return Seconds_Scale;}

/**	Get the height of the seconds scale display area.
<p>
	If the appropriate size can not be determined from the {@link
	#ANNOTATION_FONT annotation font} (because a component rendering
	graphics device has not yet been assigned) the height will be a
	guesstimate based on a typical display device of approximately 100
	pixels per inch.
<p>
	@return	The height, in pixels, of the seonds scale display area.
*/
public int Seconds_Scale_Height ()
{
int
	value = 0;
if (Seconds_Scale)
	{
	if (Seconds_Scale_Height == 0)
		{
		Reset_Geometry ();
		value = (Seconds_Scale_Height == 0) ?
			DEFAULT_SECONDS_SCALE_HEIGHT :
			Seconds_Scale_Height;
		}
	else
		value = Seconds_Scale_Height;
	}
return value;
}

/**	Determine the annotation and seconds scale sizes.
<p>
	If a graphics display device can be obtained for the component the
	annotation and seconds scale sizes are determined from the {@link
	#ANNOTATION_FONT} metrics. The annotation area width is the rendered
	width of the string "1055M" plus 3; the seconds scale height is the
	same as the annotation text height. If, however, a graphics display
	device for the component is not currently known then all values are
	set to zero.
*/
private void Reset_Geometry ()
{
if ((DEBUG & DEBUG_GEOMETRY) != 0)
	System.out.println (">>> Memory_Chart.Reset_Geometry");
Graphics
	graphics = getGraphics ();
if (graphics != null)
	{
	Rectangle2D
		bounds = getFontMetrics (ANNOTATION_FONT)
			.getStringBounds ("1055M", graphics);
	Annotation_Area_Width  = (int)(bounds.getWidth ()) + 3;
	Annotation_Text_Height = (int)(bounds.getHeight ());
	Seconds_Scale_Height   = Annotation_Text_Height;
	}
else
	{
	Annotation_Area_Width  =
	Annotation_Text_Height =
	Seconds_Scale_Height   = 0;
	}
if ((DEBUG & DEBUG_GEOMETRY) != 0)
	System.out.println
		("    Annotation_Area_Width  = " + Annotation_Area_Width + '\n'
		+"    Annotation_Text_Height = " + Annotation_Text_Height + '\n'
		+"    Seconds_Scale_Height   = " + Seconds_Scale_Height + '\n'
		+"<<< Memory_Chart.Reset_Geometry");
}

/*==============================================================================
	Model
*/
/**	Set a new Memory_History model for the component.
<p>
	@param	model	A Memory_History object.
*/
public void setModel
	(
	Memory_History	model
	)
{
Memory_History
	previous = History;
if (model == null)
	model = new Memory_History (getWidth (), 1);
History = model;
History.addChangeListener (this);
firePropertyChange ("model", previous, model);
}

/**	Get the Memory_History model bound to the component.
<p>
	@return	A Memory_History object.
*/
public Memory_History getModel ()
{return History;}

/**	Set the memory history sampling rate.
<p>
	If the new samping rate is different than the current sampling rate
	the {@link Memory_History#Rate(int) rate} is changed and a "rate"
	property change event is fired. The {@link #Previous_Rate() previous rate},
	if non-zero, is remembered.
<p>
	<b>N.B.</b>: If the rate is less than or equal to zero the
	chart will be {@link #Stop() stopped}.
<p>
	@param	rate	The memory history sampling rate to be used.
	@return	This Memory_Chart.
*/
public Memory_Chart Rate
	(
	int		rate
	)
{
int
	previous_rate = History.Rate ();
if (previous_rate != rate)
	{
	History.Rate (rate);
	firePropertyChange ("rate", previous_rate, rate);
	}
return this;
}

/**	Stop the memory chart.
<p>
	Memory sampling will be stopped. This is the same as setting the
	{@link #Rate(int) sampling rate} to zero.
*/
public void Stop ()
{History.Rate (0);}

/**	Start the chart memory sampling.
<p>
	If the current {@link #Rate() sampling rate} is zero and the {@link
	#Previous_Rate() previous rate} was non-zero, the memory history is
	{@link Memory_History#Clear() cleared} and its sampling rate is reset
	to the previous rate.
*/
public void Start ()
{
if (History.Rate () == 0 &&
	History.Previous_Rate () != 0)
	{
	History.Clear ();
	History.Rate (History.Previous_Rate ());
	}
}

/**	Get the memory sampling rate.
<p>
	@return	The current memory history sampling rate.
*/
public int Rate ()
{return History.Rate ();}

/**	Get the previous memory sampling rate.
<p>
	@return	The previous memory history sampling rate. This will be
		zero only if there was no non-zero previous sampling rate.
*/
public int Previous_Rate ()
{return History.Previous_Rate ();}

/**	The ChangeListener interface method.
<p>
	The action is to repaint the component.
*/
public void stateChanged
	(
	ChangeEvent	event
	)
{repaint ();}

/**	Get the amount of memory allocated by the Java runtime environment
	but not currently in use.
<p>
	@return	The amount, in bytes, of allocated but free memory.
	@see	Memory_History#Available_Memory()
*/
public static long Free_Memory ()
{return Memory_History.Free_Memory ();}

/**	Get the amount of memory currently in use by the Java runtime
	environment.
<p>
	@return	The amount, in bytes, of allocated memory.
	@see	Memory_History#Available_Memory()
*/
public static long Allocated_Memory ()
{return Memory_History.Allocated_Memory ();}

/**	Get the total amount of memory available to the Java runtime
	environment.
<p>
	@return	The total amount of memory, in bytes, available to the Java
		runtime environment.
	@see	Memory_History#Available_Memory()
*/
public static long Available_Memory ()
{return Memory_History.Available_Memory ();}

/*==============================================================================
	View
*/
/**	Set the graphical user interface (UI) delegate for the component.
<p>
	@param	UI	A Memory_ChartUI object that will paint the component
		and handle other user interface events.
	@see	JComponent#setUI(ComponentUI)
*/
public void setUI
	(
	Memory_ChartUI	UI
	)
{super.setUI (UI);}

/**	Set the user interface delegate to the ComponentUI found by the
	UIManger for this component.
<p>
	This component is invalidated.
<p>
	@see	#getUIClassID()
*/
public void updateUI ()
{
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println (">>> Memory_Chart.updateUI");
setUI ((Memory_ChartUI)UIManager.getUI (this));
invalidate ();
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println ("<<< Memory_Chart.updateUI");
}

/**	Get the ComponentUI class identification for this component.
<p>
	@return	The String "Memory_ChartUI".
*/
public String getUIClassID ()
{
if ((DEBUG & DEBUG_UI) != 0)
	System.out.println (">-< Memory_Chart.getUIClassID: Memory_ChartUI");
return "Memory_ChartUI";
}

}	//	End of Memory_Chart class.
