/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <glib.h>
#include <glib-object.h>
#include <hybris/properties/properties.h>

#include "powerd-internal.h"
#include "device-config.h"
#include "log.h"

/*
 * Android defines various device-specific parameters in files named
 * config.xml. The general structure of the tags is:
 *
 *   <type name="config_foo">...</type>
 *
 * This will define a config option named "foo" of type "type". Types
 * for arrays are included. For example, the following defines an
 * array of integers:
 *
 *   <integer-array name="config_myIntArray">
 *     <item>0</item>
 *     <item>1</item>
 *   </integer-array>
 *
 * In powerd we are only interested in a few specific variables. The
 * code here offers fairly general parsing of the config files, but
 * handling of only a few specific types is present.
 */

enum elem_type {
    ELEM_TYPE_NONE,
    ELEM_TYPE_BOOL,
    ELEM_TYPE_INT,
    ELEM_TYPE_STRING,
    ELEM_TYPE_FRACTION,
    ELEM_TYPE_DIMENSION,
    ELEM_TYPE_INT_ARRAY,
    ELEM_TYPE_STRING_ARRAY,
    ELEM_TYPE_ITEM,

    NUM_ELEM_TYPES
};

static const char *elem_strs[NUM_ELEM_TYPES] = {
    [ELEM_TYPE_BOOL]          = "bool",
    [ELEM_TYPE_INT]           = "integer",
    [ELEM_TYPE_STRING]        = "string",
    [ELEM_TYPE_FRACTION]      = "fraction",
    [ELEM_TYPE_DIMENSION]     = "dimen",
    [ELEM_TYPE_INT_ARRAY]     = "integer-array",
    [ELEM_TYPE_STRING_ARRAY]  = "string-array",
    [ELEM_TYPE_ITEM]          = "item",
};

struct parser_state {
    enum elem_type type;
    gboolean in_item;
    gchar *name;
    GValue value;
};

static struct parser_state state = {
    .value = G_VALUE_INIT,
};
static GHashTable *config_hash;

static enum elem_type get_elem_type(const gchar *name)
{
    int i;
    for (i = 0; i < NUM_ELEM_TYPES; i++) {
        if (elem_strs[i] && !strcmp(name, elem_strs[i]))
            break;
    }
    if (i >= NUM_ELEM_TYPES)
        return ELEM_TYPE_NONE;
    return i;
}

static const gchar *get_elem_name(const gchar **attr_names,
                                  const gchar **attr_values)
{
    int i;
    for (i = 0; attr_names[i]; i++) {
        if (!strcmp(attr_names[i], "name"))
            break;
    }
    if (!attr_names[i])
        return NULL;
    return attr_values[i];
}

static void on_start_elem(GMarkupParseContext *context, const gchar *name,
                          const gchar **attr_names, const gchar **attr_values,
                          gpointer user_data, GError **error)
{
    const gchar *config_name;
    enum elem_type type;

    type = get_elem_type(name);
    if (type == ELEM_TYPE_NONE)
        return;
    if (type == ELEM_TYPE_ITEM) {
        if (state.type == ELEM_TYPE_INT_ARRAY ||
            state.type == ELEM_TYPE_STRING_ARRAY)
            state.in_item = TRUE;
        return;
    }
    if (state.type != ELEM_TYPE_NONE) {
        g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                    "Nested elements not supported");
        return;
    }

    config_name = get_elem_name(attr_names, attr_values);
    if (!config_name)
        return;
    /* Strip leading "config_" if present, not all vars have it */
    if (!strncmp(config_name, "config_", 7))
        config_name += 7;

    switch (type) {
    case ELEM_TYPE_BOOL:
        g_value_init(&state.value, G_TYPE_BOOLEAN);
        break;
    case ELEM_TYPE_INT:
        g_value_init(&state.value, G_TYPE_UINT);
        break;
    case ELEM_TYPE_INT_ARRAY:
        {
            GArray *a;
            g_value_init(&state.value, G_TYPE_ARRAY);
            a = g_array_new(FALSE, FALSE, sizeof(guint32));
            g_value_set_boxed(&state.value, a);
            break;
        }
    default:
        /* Type not supported at this time */
        return;
    }

    state.type = type;
    state.in_item = FALSE;
    state.name = g_strdup(config_name);
}

static void on_end_elem(GMarkupParseContext *context, const gchar *name,
                        gpointer user_data, GError **error)
{
    enum elem_type type;
    GValue *value;

    if (state.type == ELEM_TYPE_NONE)
        return;

    type = get_elem_type(name);
    if (type == ELEM_TYPE_NONE)
        return;
    if (type == ELEM_TYPE_ITEM) {
        state.in_item = FALSE;
        return;
    }

    if (type == state.type) {
        value = g_new0(GValue, 1);
        g_value_init(value, G_VALUE_TYPE(&state.value));
        g_value_copy(&state.value, value);
        g_hash_table_replace(config_hash, state.name, value);
        /* Note: state.name is not freed as it is used for hash table key */
    } else {
        g_set_error(error, G_MARKUP_ERROR, G_MARKUP_ERROR_INVALID_CONTENT,
                    "Start and end element types don't match");
        g_free(state.name);
    }

    state.type = ELEM_TYPE_NONE;
    state.name = NULL;
    g_value_unset(&state.value);
}

/* WARNING: pt_text is not nul-terminated */
static void on_text(GMarkupParseContext *context, const gchar *pt_text,
                    gsize text_len, gpointer user_data, GError **error)
{
    gchar *text = g_strndup(pt_text, text_len);
    switch (state.type) {
    case ELEM_TYPE_BOOL:
        {
            gboolean value;
            if (!G_VALUE_HOLDS_BOOLEAN(&state.value)) {
                g_warning("Incorrect GValue type");
                break;
            }
            value = !g_ascii_strcasecmp(text, "true");
            g_value_set_boolean(&state.value, value);
            break;
        }
    case ELEM_TYPE_INT:
        {
            guint value;
            gchar *endp;
            if (!G_VALUE_HOLDS_UINT(&state.value)) {
                g_warning("Incorrect GValue type");
                break;
            }
            value = (guint)g_ascii_strtoll(text, &endp, 0);
            if (endp - text == text_len)
                g_value_set_uint(&state.value, value);
            break;
        }
    case ELEM_TYPE_INT_ARRAY:
        {
            GArray *a;
            guint value;
            gchar *endp;
            if (!state.in_item)
                break;
            if (!G_VALUE_HOLDS_BOXED(&state.value)) {
                g_warning("Incorrect GValue type");
                break;
            }
            value = (guint)g_ascii_strtoll(text, &endp, 0);
            if (endp - text == text_len) {
                a = g_value_get_boxed(&state.value);
                g_array_append_val(a, value);
            }
        }
        break;
    default:
        break;
    }
    g_free(text);
}

static GMarkupParser parser = {
    .start_element  = on_start_elem,
    .end_element    = on_end_elem,
    .text           = on_text,
};

static void config_hash_destroy_key(gpointer data)
{
    g_free(data);
}

static void config_hash_destroy_value(gpointer data)
{
    GValue *v = data;
    if (G_VALUE_HOLDS_BOXED(v)) {
        /* Currently only boxed data is arrays */
        GArray *a = g_value_get_boxed(v);
        g_array_free(a, TRUE);
    }
    g_value_unset(v);
    g_free(v);
}

static int process_config(const char *fname)
{
    int fd, ret = 0;
    gchar *buf = NULL;
    long page_size;
    int buf_len;
    ssize_t len;
    GMarkupParseContext *context = NULL;
    GError *error;

    fd = open(fname, O_RDONLY);
    if (fd == -1) {
        powerd_warn("Could not open device config %s: %s",
                    fname, strerror(errno));
        return -errno;
    }

    buf_len = 4096;
    page_size = sysconf(_SC_PAGESIZE);
    if (page_size > 0)
        buf_len = (int)page_size;
    buf = malloc(buf_len);
    if (!buf) {
        ret = -ENOMEM;
        goto out;
    }

    context = g_markup_parse_context_new(&parser, 0, NULL, NULL);
    do {
        len = read(fd, buf, buf_len);
        if (len == 0)
            break;
        if (len == -1) {
            if (errno == EINTR)
                continue;
            powerd_warn("Error reading %s: %s", fname, strerror(errno));
            ret = -errno;
            break;
        }

        error = NULL;
        if (!g_markup_parse_context_parse(context, buf, len, &error)) {
            powerd_warn("Failed parsing device config %s\n", fname);
            if (error) {
                powerd_warn("%s", error->message);
                g_error_free(error);
            }
            ret = -EIO;
        }
    } while (!ret);

out:
    if (context)
        g_markup_parse_context_free(context);
    if (buf)
        free(buf);
    close(fd);
    return ret;
}

/*
 * Get the device config option specified by @name. The leading
 * "config_" characters should be stripped from the name if present.
 * Returns 0 if the given variable is found.
 *
 * @value should be an initialized GValue. On success, the value of
 * the variable will be copied to the GValue, and the caller is
 * responsible for freeing the memory. E.g.:
 *
 *   GValue v = G_VALUE_INIT;
 *   if (!device_config_get("myVar", &v)) {
 *       do_something_with_value(&v);
 *       g_value_unset(&v);
 *   }
 */
int device_config_get(const char *name, GValue *value)
{
    GValue *v;

    if (!config_hash)
        return -ENODEV;

    v = g_hash_table_lookup(config_hash, name);
    if (!v)
        return -ENOENT;

    g_value_init(value, G_VALUE_TYPE(v));
    g_value_copy(v, value);
    return 0;
}

void device_config_init(void)
{
    char device[PROP_VALUE_MAX];
    char *xml_path;

    if (!property_get("ro.product.device", device, NULL)) {
        powerd_warn("Could not determine device, running without config");
        return;
    }
    powerd_info("Running on %s\n", device);

    config_hash = g_hash_table_new_full(g_str_hash, g_str_equal,
                                        config_hash_destroy_key,
                                        config_hash_destroy_value);

    /*
     * Always start with config-default.xml for defaults, then
     * the device-specific config
     */
    if (process_config(POWERD_DEVICE_CONFIGS_PATH "/config-default.xml"))
        goto error;
    xml_path = g_strdup_printf("%s/config-%s.xml",
                               POWERD_DEVICE_CONFIGS_PATH, device);
    if (process_config(xml_path))
        goto error;
    g_free(xml_path);
    return;

error:
    device_config_deinit();
}

void device_config_deinit(void)
{
    if (config_hash) {
        g_hash_table_destroy(config_hash);
        config_hash = NULL;
    }
}
