#!/usr/bin/env python3

import filecmp
import os
import shutil
import subprocess
import tempfile


VERSIONFILE = "px/version.py"


def main():
    """Update px/version.py with the current git version."""
    git_version = (
        subprocess.check_output(["git", "describe", "--dirty"]).decode("utf-8").strip()
    )

    with tempfile.NamedTemporaryFile(suffix=".py", delete=False) as tmp:
        tmp.write(b"# NOTE: Auto generated by update_version_py.py, no touchie!\n")
        tmp.write(b'VERSION = "%s"\n' % bytearray(git_version, "utf_8"))

        # Flushing is required for filecmp.cmp() to work (below)
        tmp.flush()

        if not os.path.isfile(VERSIONFILE):
            # No version file found
            shutil.move(tmp.name, VERSIONFILE)
        elif not filecmp.cmp(tmp.name, VERSIONFILE):
            # Version file needs updating
            shutil.move(tmp.name, VERSIONFILE)
        else:
            # VERSIONFILE was already up to date. If we touch it in this
            # case, it will have its file timestamp updated, which will
            # force the slow px_integration_test.py tests to get rerun.
            #
            # Just clean up our tempfile and be merry.
            os.remove(tmp.name)


if __name__ == "__main__":
    main()
