#' Drawing graphs
#'
#' The common bits of the three plotting functions `plot.igraph`,
#' `tkplot` and `rglplot` are discussed in this manual page.
#'
#' There are currently three different functions in the igraph package which
#' can draw graph in various ways:
#'
#' `plot.igraph` does simple non-interactive 2D plotting to R devices.
#' Actually it is an implementation of the [graphics::plot()] generic
#' function, so you can write `plot(graph)` instead of
#' `plot.igraph(graph)`. As it used the standard R devices it supports
#' every output format for which R has an output device. The list is quite
#' impressing: PostScript, PDF files, XFig files, SVG files, JPG, PNG and of
#' course you can plot to the screen as well using the default devices, or the
#' good-looking anti-aliased Cairo device.  See [plot.igraph()] for
#' some more information.
#'
#' [tkplot()] does interactive 2D plotting using the `tcltk`
#' package. It can only handle graphs of moderate size, a thousand vertices is
#' probably already too many. Some parameters of the plotted graph can be
#' changed interactively after issuing the `tkplot` command: the position,
#' color and size of the vertices and the color and width of the edges. See
#' [tkplot()] for details.
#'
#' [rglplot()] is an experimental function to draw graphs in 3D using
#' OpenGL. See [rglplot()] for some more information.
#'
#' Please also check the examples below.
#'
#' @aliases igraph.plotting
#' @section How to specify graphical parameters: There are three ways to give
#' values to the parameters described below, in section 'Parameters'. We give
#' these three ways here in the order of their precedence.
#'
#' The first method is to supply named arguments to the plotting commands:
#' [plot.igraph()], [tkplot()] or rglplot()].
#' Parameters for vertices start with prefix \sQuote{\code{vertex.}},
#' parameters for edges have prefix \sQuote{\code{edge.}}, and global
#' parameters have no prefix. Eg. the color of the vertices can be given via
#' argument `vertex.color`, whereas `edge.color` sets the color of
#' the edges. `layout` gives the layout of the graphs.
#'
#' The second way is to assign vertex, edge and graph attributes to the graph.
#' These attributes have no prefix, ie. the color of the vertices is taken from
#' the `color` vertex attribute and the color of the edges from the
#' `color` edge attribute. The layout of the graph is given by the
#' `layout` graph attribute. (Always assuming that the corresponding
#' command argument is not present.) Setting vertex and edge attributes are
#' handy if you want to assign a given \sQuote{look} to a graph, attributes are
#' saved with the graph is you save it with [base::save()] or in
#' GraphML format with [write_graph()], so the graph will have the
#' same look after loading it again.
#'
#' If a parameter is not given in the command line, and the corresponding
#' vertex/edge/graph attribute is also missing then the general igraph
#' parameters handled by [igraph_options()] are also checked. Vertex
#' parameters have prefix \sQuote{\code{vertex.}}, edge parameters are prefixed
#' with \sQuote{\code{edge.}}, general parameters like `layout` are
#' prefixed with \sQuote{\code{plot}}.  These parameters are useful if you want
#' all or most of your graphs to have the same look, vertex size, vertex color,
#' etc. Then you don't need to set these at every plotting, and you also don't
#' need to assign vertex/edge attributes to every graph.
#'
#' If the value of a parameter is not specified by any of the three ways
#' described here, its default valued is used, as given in the source code.
#'
#' Different parameters can have different type, eg. vertex colors can be given
#' as a character vector with color names, or as an integer vector with the
#' color numbers from the current palette. Different types are valid for
#' different parameters, this is discussed in detail in the next section. It is
#' however always true that the parameter can always be a function object in
#' which it will be called with the graph as its single argument to get the
#' \dQuote{proper} value of the parameter.  (If the function returns another
#' function object that will *not* be called again\dots)
#' @section The list of parameters: Vertex parameters first, note that the
#' \sQuote{\code{vertex.}} prefix needs to be added if they are used as an
#' argument or when setting via [igraph_options()]. The value of the
#' parameter may be scalar valid for every vertex or a vector with a separate
#' value for each vertex. (Shorter vectors are recycled.)  \describe{
#' \item{size}{The size of the vertex, a numeric scalar or vector, in the
#' latter case each vertex sizes may differ. This vertex sizes are scaled in
#' order have about the same size of vertices for a given value for all three
#' plotting commands. It does not need to be an integer number.
#' The default value is 15. This is big enough to place short labels on
#' vertices.}
#' \item{size2}{The \dQuote{other} size of the vertex, for some
#' vertex shapes. For the various rectangle shapes this gives the height of the
#' vertices, whereas `size` gives the width. It is ignored by shapes for
#' which the size can be specified with a single number.
#' The default is 15.}
#' \item{color}{The fill color of the vertex. If it is
#' numeric then the current palette is used, see
#' [grDevices::palette()]. If it is a character vector then it may
#' either contain integer values, named colors or RGB specified colors with
#' three or four bytes. All strings starting with \sQuote{\code{#}} are assumed
#' to be RGB color specifications. It is possible to mix named color and RGB
#' colors. Note that [tkplot()] ignores the fourth byte (alpha
#' channel) in the RGB color specification.
#'
#' For `plot.igraph` and integer values, the default igraph palette is
#' used (see the \sQuote{palette} parameter below. Note that this is different
#' from the R palette.
#'
#' If you don't want (some) vertices to have any color, supply `NA` as the
#' color name.
#'
#' The default value is \dQuote{\code{SkyBlue2}}.}
#' \item{frame.color}{The
#' color of the frame of the vertices, the same formats are allowed as for the
#' fill color.
#'
#' If you don't want vertices to have a frame, supply `NA` as the color
#' name.
#'
#' By default it is \dQuote{black}.  }
#' \item{frame.width}{The width of the
#' frame of the vertices. The default value is 1.
#' }
#' \item{shape}{The shape of the vertex, currently \dQuote{\code{circle}},
#' \dQuote{\code{square}}, \dQuote{\code{csquare}}, \dQuote{\code{rectangle}},
#' \dQuote{\code{crectangle}}, \dQuote{\code{vrectangle}}, \dQuote{\code{pie}}
#' (see [vertex.shape.pie()]), \sQuote{\code{sphere}}, and
#' \dQuote{\code{none}} are supported, and only by the
#' [plot.igraph()] command. \dQuote{\code{none}} does not draw the
#' vertices at all, although vertex label are plotted (if given). See
#' [shapes()] for details about vertex shapes and
#' [vertex.shape.pie()] for using pie charts as vertices.
#'
#' The \dQuote{\code{sphere}} vertex shape plots vertices as 3D ray-traced
#' spheres, in the given color and size. This produces a raster image and it is
#' only supported with some graphics devices. On some devices raster
#' transparency is not supported and the spheres do not have a transparent
#' background. See [dev.capabilities] and the
#' \sQuote{\code{rasterImage}} capability to check that your device is
#' supported.
#'
#' By default vertices are drawn as circles.  }
#' \item{label}{The vertex labels.
#' They will be converted to character. Specify `NA` to omit vertex
#' labels.
#' The default vertex labels are the vertex ids.}
#' \item{label.family}{The
#' font family to be used for vertex labels.  As different plotting commands
#' can used different fonts, they interpret this parameter different ways. The
#' basic notation is, however, understood by both [plot.igraph()] and
#' [tkplot()]. [rglplot()] does not support fonts at all
#' right now, it ignores this parameter completely.
#'
#' For [plot.igraph()] this parameter is simply passed to
#' [graphics::text()] as argument `family`.
#'
#' For [tkplot()] some conversion is performed. If this parameter is
#' the name of an existing Tk font, then that font is used and the
#' `label.font` and `label.cex` parameters are ignored completely. If
#' it is one of the base families (serif, sans, mono) then Times, Helvetica or
#' Courier fonts are used, there are guaranteed to exist on all systems. For
#' the \sQuote{symbol} base family we used the symbol font is available,
#' otherwise the first font which has \sQuote{symbol} in its name. If the
#' parameter is not a name of the base families and it is also not a named Tk
#' font then we pass it to [tcltk::tkfont.create()] and hope the user
#' knows what she is doing. The `label.font` and `label.cex`
#' parameters are also passed to [tcltk::tkfont.create()] in this
#' case.
#'
#' The default value is \sQuote{serif}.  }
#' \item{label.font}{The font within
#' the font family to use for the vertex labels. It is interpreted the same way
#' as the the `font` graphical parameter: 1 is plain text, 2 is bold face,
#' 3 is italic, 4 is bold and italic and 5 specifies the symbol font.
#'
#' For [plot.igraph()] this parameter is simply passed to
#' [graphics::text()].
#'
#' For [tkplot()], if the `label.family` parameter is not the
#' name of a Tk font then this parameter is used to set whether the newly
#' created font should be italic and/or boldface. Otherwise it is ignored.
#'
#' For [rglplot()] it is ignored.
#'
#' The default value is 1.  }
#' \item{label.cex}{The font size for vertex labels.
#' It is interpreted as a multiplication factor of some device-dependent base
#' font size.
#'
#' For [plot.igraph()] it is simply passed to
#' [graphics::text()] as argument `cex`.
#'
#' For [tkplot()] it is multiplied by 12 and then used as the
#' `size` argument for [tcltk::tkfont.create()].  The base font
#' is thus 12 for tkplot.
#'
#' For [rglplot()] it is ignored.
#'
#' The default value is 1.  }
#' \item{label.dist}{ The distance of the label from
#' the center of the vertex. If it is 0 then the label is centered on the
#' vertex. If it is 1 then the label is displayed beside the vertex.
#'
#' The default value is 0.  }
#' \item{label.degree}{ It defines the position of
#' the vertex labels, relative to the center of the vertices. It is interpreted
#' as an angle in radians, zero means \sQuote{to the right}, and
#' \sQuote{\code{pi}} means to the left, up is `-pi/2` and down is
#' `pi/2`.
#'
#' The default value is `-pi/4`.  }
#' \item{label.color}{The color of the
#' labels, see the `color` vertex parameter discussed earlier for the
#' possible values.
#'
#' The default value is `black`.  } }
#'
#' Edge parameters require to add the \sQuote{\code{edge.}} prefix when used as
#' arguments or set by [igraph_options()]. The edge parameters:
#' \describe{
#' \item{color}{The color of the edges, see the `color` vertex
#' parameter for the possible values.
#' By default this parameter is `darkgrey`.  }
#' \item{width}{The width of
#' the edges.
#' The default value is 1.  }
#' \item{arrow.size}{The size of the arrows.
#' Currently this is a constant, so it is the same for every edge. If a vector
#' is submitted then only the first element is used, ie. if this is taken from
#' an edge attribute then only the attribute of the first edge is used for all
#' arrows. This will likely change in the future.
#'
#' The default value is 1.  }
#' \item{arrow.width}{The width of the arrows.
#' Currently this is a constant, so it is the same for every edge. If a vector
#' is submitted then only the first element is used, ie. if this is taken from
#' an edge attribute then only the attribute of the first edge is used for all
#' arrows. This will likely change in the future.
#'
#' This argument is currently only used by [plot.igraph()].
#'
#' The default value is 1, which gives the same width as before this option
#' appeared in igraph.  }
#' \item{lty}{The line type for the edges. Almost the
#' same format is accepted as for the standard graphics
#' [graphics::par()], 0 and \dQuote{blank} mean no edges, 1 and
#' \dQuote{solid} are for solid lines, the other possible values are: 2
#' (\dQuote{dashed}), 3 (\dQuote{dotted}), 4 (\dQuote{dotdash}), 5
#' (\dQuote{longdash}), 6 (\dQuote{twodash}).
#'
#' [tkplot()] also accepts standard Tk line type strings, it does not
#' however support \dQuote{blank} lines, instead of type \sQuote{0} type
#' \sQuote{1}, ie. solid lines will be drawn.
#'
#' This argument is ignored for [rglplot()].
#'
#' The default value is type 1, a solid line.  }
#' \item{label}{The edge labels.
#' They will be converted to character. Specify `NA` to omit edge labels.
#'
#' Edge labels are omitted by default.}
#' \item{label.family}{Font family of the
#' edge labels. See the vertex parameter with the same name for the details.}
#' \item{label.font}{The font for the edge labels. See the corresponding vertex
#' parameter discussed earlier for details.}
#' \item{label.cex}{The font size for
#' the edge labels, see the corresponding vertex parameter for details.}
#' \item{label.color}{The color of the edge labels, see the `color` vertex
#' parameters on how to specify colors. }
#' \item{label.x}{The horizontal
#' `NA` elements will be replaced by automatically calculated coordinates.
#' If `NULL`, then all edge horizontal coordinates are calculated
#' automatically. This parameter is only supported by `plot.igraph`.}
#' \item{label.y}{The same as `label.x`, but for vertical coordinates.}
#' \item{curved}{Specifies whether to draw curved edges, or not. This can be a
#' logical or a numeric vector or scalar.
#'
#' First the vector is replicated to have the same length as the number of
#' edges in the graph. Then it is interpreted for each edge separately. A
#' numeric value specifies the curvature of the edge; zero curvature means
#' straight edges, negative values means the edge bends clockwise, positive
#' values the opposite. `TRUE` means curvature 0.5, `FALSE` means
#' curvature zero.
#'
#' By default the vector specifying the curvature is calculated via a call to
#' the [curve_multiple()] function. This function makes sure that
#' multiple edges are curved and are all visible. This parameter is ignored for
#' loop edges.
#'
#' The default value is `FALSE`.
#'
#' This parameter is currently ignored by [rglplot()].}
#' \item{arrow.mode}{This parameter can be used to specify for which edges
#' should arrows be drawn. If this parameter is given by the user (in either of
#' the three ways) then it specifies which edges will have forward, backward
#' arrows, or both, or no arrows at all.  As usual, this parameter can be a
#' vector or a scalar value.  It can be an integer or character type. If it is
#' integer then 0 means no arrows, 1 means backward arrows, 2 is for forward
#' arrows and 3 for both. If it is a character vector then \dQuote{<} and
#' \dQuote{<-} specify backward, \dQuote{>} and \dQuote{->} forward arrows and
#' \dQuote{<>} and \dQuote{<->} stands for both arrows.  All other values mean
#' no arrows, perhaps you should use \dQuote{-} or \dQuote{--} to specify no
#' arrows.
#'
#' Hint: this parameter can be used as a \sQuote{cheap} solution for drawing
#' \dQuote{mixed} graphs: graphs in which some edges are directed some are not.
#' If you want do this, then please create a *directed* graph, because as
#' of version 0.4 the vertex pairs in the edge lists can be swapped in
#' undirected graphs.
#'
#' By default, no arrows will be drawn for undirected graphs, and for directed
#' graphs, an arrow will be drawn for each edge, according to its direction.
#' This is not very surprising, it is the expected behavior.  }
#' \item{loop.angle}{Gives the angle in radians for plotting loop edges. See
#' the `label.dist` vertex parameter to see how this is interpreted.
#'
#' The default value is 0.  }
#' \item{loop.angle2}{Gives the second angle in
#' radians for plotting loop edges. This is only used in 3D, `loop.angle`
#' is enough in 2D.
#'
#' The default value is 0.  } }
#'
#' Other parameters: \describe{
#' \item{layout}{ Either a function or a numeric
#' matrix. It specifies how the vertices will be placed on the plot.
#'
#' If it is a numeric matrix, then the matrix has to have one line for each
#' vertex, specifying its coordinates. The matrix should have at least two
#' columns, for the `x` and `y` coordinates, and it can also have
#' third column, this will be the `z` coordinate for 3D plots and it is
#' ignored for 2D plots.
#'
#' If a two column matrix is given for the 3D plotting function
#' [rglplot()] then the third column is assumed to be 1 for each
#' vertex.
#'
#' If `layout` is a function, this function will be called with the
#' `graph` as the single parameter to determine the actual coordinates.
#' The function should return a matrix with two or three columns. For the 2D
#' plots the third column is ignored.
#' The default value is `layout_nicely`, a smart function that chooses a
#' layout based on the graph.}
#' \item{margin}{The amount of empty space below,
#' over, at the left and right of the plot, it is a numeric vector of length
#' four. Usually values between 0 and 0.5 are meaningful, but negative values
#' are also possible, that will make the plot zoom in to a part of the graph.
#' If it is shorter than four then it is recycled.
#' [rglplot()] does not support this parameter, as it can zoom in and
#' out the graph in a more flexible way.
#' Its default value is 0.  }
#' \item{palette}{The color palette to use for
#' vertex color.  The default is \code{\link{categorical_pal}}, which is a
#' color-blind friendly categorical palette. See its manual page for details
#' and other palettes. This parameters is only supported by `plot`, and
#' not by `tkplot` and `rglplot`.  }
#' \item{rescale}{Logical constant,
#' whether to rescale the coordinates to the `[-1,1]x[-1,1](x[-1,1])` interval.
#' This parameter is not implemented for `tkplot`.
#' Defaults to `TRUE`, the layout will be rescaled.  }
#' \item{asp}{A
#' numeric constant, it gives the `asp` parameter for [plot()],
#' the aspect ratio. Supply 0 here if you don't want to give an aspect ratio.
#' It is ignored by `tkplot` and `rglplot`.
#' Defaults to 1.  }
#' \item{frame}{Boolean, whether to plot a frame around the
#' graph. It is ignored by `tkplot` and `rglplot`.
#' Defaults to `FALSE`.  }
#' \item{main}{Overall title for the main plot.
#' The default is empty if the `annotate.plot` igraph option is
#' `FALSE`, and the graph's `name` attribute otherwise. See the same
#' argument of the base `plot` function. Only supported by `plot`.}
#' \item{sub}{Subtitle of the main plot, the default is empty. Only supported
#' by `plot`.}
#' \item{xlab}{Title for the x axis, the default is empty if
#' the `annotate.plot` igraph option is `FALSE`, and the number of
#' vertices and edges, if it is `TRUE`. Only supported by `plot`.}
#' \item{ylab}{Title for the y axis, the default is empty. Only supported by
#' `plot`.} }
#' @author Gabor Csardi \email{csardi.gabor@@gmail.com}
#' @seealso [plot.igraph()], [tkplot()],
#' [rglplot()], [igraph_options()]
#' @keywords graphs
#' @examples
#'
#' \dontrun{
#'
#' # plotting a simple ring graph, all default parameters, except the layout
#' g <- make_ring(10)
#' g$layout <- layout_in_circle
#' plot(g)
#' tkplot(g)
#' rglplot(g)
#'
#' # plotting a random graph, set the parameters in the command arguments
#' g <- barabasi.game(100)
#' plot(g, layout=layout_with_fr, vertex.size=4,
#'      vertex.label.dist=0.5, vertex.color="red", edge.arrow.size=0.5)
#'
#' # plot a random graph, different color for each component
#' g <- sample_gnp(100, 1/100)
#' comps <- components(g)$membership
#' colbar <- rainbow(max(comps)+1)
#' V(g)$color <- colbar[comps+1]
#' plot(g, layout=layout_with_fr, vertex.size=5, vertex.label=NA)
#'
#' # plot communities in a graph
#' g <- make_full_graph(5) %du% make_full_graph(5) %du% make_full_graph(5)
#' g <- add_edges(g, c(1,6, 1,11, 6,11))
#' com <- cluster_spinglass(g, spins=5)
#' V(g)$color <- com$membership+1
#' g <- set_graph_attr(g, "layout", layout_with_kk(g))
#' plot(g, vertex.label.dist=1.5)
#'
#' # draw a bunch of trees, fix layout
#' igraph_options(plot.layout=layout_as_tree)
#' plot(make_tree(20, 2))
#' plot(make_tree(50, 3), vertex.size=3, vertex.label=NA)
#' tkplot(make_tree(50, 2, mode="undirected"), vertex.size=10,
#' vertex.color="green")
#' }
#' @name plot.common
#' @rdname plot.common
NULL

#' Optimal edge curvature when plotting graphs
#'
#' @description
#' `r lifecycle::badge("deprecated")`
#'
#' `autocurve.edges()` was renamed to `curve_multiple()` to create a more
#' consistent API.
#' @inheritParams curve_multiple
#' @keywords internal
#' @export
autocurve.edges <- function(graph, start = 0.5) { # nocov start
  lifecycle::deprecate_soft("2.0.0", "autocurve.edges()", "curve_multiple()")
  curve_multiple(graph = graph, start = start)
} # nocov end
#   IGraph R package
#   Copyright (C) 2003-2012  Gabor Csardi <csardi.gabor@gmail.com>
#   334 Harvard street, Cambridge, MA 02139 USA
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc.,  51 Franklin Street, Fifth Floor, Boston, MA
#   02110-1301 USA
#
###################################################################

###################################################################
# Common functions for plot and tkplot
###################################################################

i.parse.plot.params <- function(graph, params) {
  ## store the arguments
  p <- list(vertex = list(), edge = list(), plot = list())
  for (n in names(params)) {
    if (substr(n, 1, 7) == "vertex.") {
      nn <- substring(n, 8)
      p[["vertex"]][[nn]] <- params[[n]]
    } else if (substr(n, 1, 5) == "edge.") {
      nn <- substring(n, 6)
      p[["edge"]][[nn]] <- params[[n]]
    } else {
      p[["plot"]][[n]] <- params[[n]]
    }
  }

  func <- function(type, name, range = NULL, dontcall = FALSE) {
    if (!type %in% names(p)) {
      stop("Invalid plot option type")
    }
    ret <- function() {
      v <- p[[type]][[name]]
      if (is.function(v) && !dontcall) {
        v <- v(graph)
      }
      if (is.null(range)) {
        return(v)
      } else {
        if (length(v) == 1) {
          return(rep(v, length(range)))
        } else {
          return(rep(v, length.out = max(range) + 1)[[range + 1]])
        }
      }
    }
    if (name %in% names(p[[type]])) {
      ## we already have the parameter
      return(ret())
    } else {
      ## we don't have the parameter, check attributes first
      if (type == "vertex" && name %in% vertex_attr_names(graph)) {
        p[[type]][[name]] <- vertex_attr(graph, name)
        return(ret())
      } else if (type == "edge" && name %in% edge_attr_names(graph)) {
        p[[type]][[name]] <- edge_attr(graph, name)
        return(ret())
      } else if (type == "plot" && name %in% graph_attr_names(graph)) {
        p[[type]][[name]] <- graph_attr(graph, name)
        return(ret())
      } else {
        ## no attributes either, check igraph parameters
        n <- paste(sep = "", type, ".", name)
        v <- igraph_opt(n)
        if (!is.null(v)) {
          p[[type]][[name]] <- v
          return(ret())
        }
        ## no igraph parameter either, use default value
        p[[type]][[name]] <- i.default.values[[type]][[name]]
        return(ret())
      }
    }
  }

  return(func)
}

i.get.edge.labels <- function(graph, edge.labels = NULL) {
  if (is.null(edge.labels)) {
    edge.labels <- rep(NA, ecount(graph))
  }

  edge.labels
}

i.get.labels <- function(graph, labels = NULL) {
  if (is.null(labels)) {
    if ("name" %in% vertex_attr_names(graph)) {
      labels <- vertex_attr(graph, "name")
    } else {
      labels <- seq_len(vcount(graph))
    }
  }
  labels
}

i.get.arrow.mode <- function(graph, arrow.mode = NULL) {
  if (is.character(arrow.mode) &&
    length(arrow.mode) == 1 && substr(arrow.mode, 1, 2) == "a:") {
    arrow.mode <- vertex_attr(graph, substring(arrow.mode, 3))
  }

  if (is.character(arrow.mode)) {
    tmp <- numeric(length(arrow.mode))
    tmp[arrow.mode %in% c("<", "<-")] <- 1
    tmp[arrow.mode %in% c(">", "->")] <- 2
    tmp[arrow.mode %in% c("<>", "<->")] <- 3
    arrow.mode <- tmp
  }

  if (is.null(arrow.mode)) {
    if (is_directed(graph)) {
      arrow.mode <- 2
    } else {
      arrow.mode <- 0
    }
  }

  arrow.mode
}

i.get.main <- function(graph) {
  if (igraph_opt("annotate.plot")) {
    n <- graph$name[1]
    n
  } else {
    ""
  }
}

i.get.xlab <- function(graph) {
  if (igraph_opt("annotate.plot")) {
    paste(vcount(graph), "vertices,", ecount(graph), "edges")
  } else {
    ""
  }
}

igraph.check.shapes <- function(x) {
  xx <- unique(x)
  bad.shapes <- !xx %in% ls(.igraph.shapes)
  if (any(bad.shapes)) {
    bs <- paste(xx[bad.shapes], collapse = ", ")
    stop("Bad vertex shape(s): ", bs, ".")
  }
  x
}

i.postprocess.layout <- function(maybe_layout) {
  if ("layout" %in% names(maybe_layout)) {
    # This branch caters for layout_with_sugiyama, which returns multiple
    # things
    layout <- maybe_layout$layout
  } else {
    # This is the normal path for layout functions that return matrices
    layout <- maybe_layout
  }
  layout
}

#' Optimal edge curvature when plotting graphs
#'
#' If graphs have multiple edges, then drawing them as straight lines does not
#' show them when plotting the graphs; they will be on top of each other. One
#' solution is to bend the edges, with diffenent curvature, so that all of them
#' are visible.
#'
#' `curve_multiple()` calculates the optimal `edge.curved` vector for
#' plotting a graph with multiple edges, so that all edges are visible.
#'
#' @param graph The input graph.
#' @param start The curvature at the two extreme edges. All edges will have a
#'   curvature between `-start` and `start`, spaced equally.
#' @return A numeric vector, its length is the number of edges in the graph.
#' @author Gabor Csardi \email{csardi.gabor@@gmail.com}
#' @seealso [igraph.plotting] for all plotting parameters,
#' [plot.igraph()], [tkplot()] and [rglplot()]
#' for plotting functions.
#' @family plot.common
#' @export
#' @importFrom stats ave
#' @keywords graphs
#' @examples
#'
#' g <- make_graph(c(
#'   0, 1, 1, 0, 1, 2, 1, 3, 1, 3, 1, 3,
#'   2, 3, 2, 3, 2, 3, 2, 3, 0, 1
#' ) + 1)
#'
#' curve_multiple(g)
#'
#' set.seed(42)
#' plot(g)
#'
curve_multiple <- function(graph, start = 0.5) {
  el <- apply(as_edgelist(graph, names = FALSE), 1, paste, collapse = ":")
  ave(rep(NA, length(el)), el, FUN = function(x) {
    if (length(x) == 1) {
      return(0)
    } else {
      return(seq(-start, start, length.out = length(x)))
    }
  })
}

.igraph.logo.raster <-
  structure(c(
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 311332508L,
    1217499541L, 1804702102L, 1066570390L, 211129749L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 379033495L, 1334940052L, -2104389227L,
    -1450012011L, -2087546218L, 1368494484L, 412456341L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 41975936L, 1905496981L,
    -141388906L, -7171435L, -7171435L, -7171435L, -325938283L, 1452380564L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    41975936L, 1905496981L, -158166379L, -7171435L, -7171435L, -7171435L,
    -7171435L, -7171435L, -141389163L, 1972540052L, 41975936L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -2037148780L, -7171435L, -24798561L, -12009013L,
    -13250855L, -11616826L, -24340838L, -7171435L, 1586664085L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 311332508L, -963472747L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7236971L, -7171435L, -7171435L,
    -7171435L, -7171435L, -946695531L, 361927314L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 798134930L,
    -40791403L, -25321308L, -16061704L, -16715521L, -16715521L, -16715521L,
    -15408144L, -24471653L, -258829418L, 344755353L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1483500650L, -7171435L, -7171435L, -7824996L, -12858668L,
    -15212050L, -16519427L, -15212050L, -12858668L, -7890531L, -7171435L,
    -7171435L, -1382903147L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 2056426132L, -7171435L, -13643043L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -12139572L,
    -7171435L, 1385337493L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 1452380564L, -7171435L,
    -7171435L, -8936279L, -15800587L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -15865867L, -9132373L, -7171435L, -7171435L,
    1485934996L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1433234795L, -7171435L, -15603981L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -14100510L, -7171435L, -2104389227L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -812412011L, -7171435L, -7432808L, -15080979L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -15277585L, -7498344L, -7171435L, -694971499L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, -1919774060L,
    -7171435L, -14623768L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -13120041L, -7171435L, 1704104597L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 177838489L,
    -74280299L, -7171435L, -10439750L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -10701380L, -7171435L, -40725867L, 211129749L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 1368494484L, -7171435L, -10374471L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16584963L, -9067350L,
    -7171435L, 714248856L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 999527315L, -7171435L, -7171435L,
    -12270386L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -12531503L, -7171435L,
    -7171435L, 1033015958L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 33554431L, -1080913258L, -7171435L, -10701636L, -15277329L,
    -16519427L, -14885141L, -9720911L, -7171435L, -1718381676L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 1217499541L, -7171435L, -7171435L, -12793389L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -13054505L, -7171435L, -7171435L, 1251053972L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    479367826L, -929918315L, -7171435L, -7171435L, -7236971L, -7171435L,
    -7171435L, -1366060139L, 227117469L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 361927314L,
    -7171435L, -7171435L, -10962753L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -11289661L, -7171435L, -7171435L, 412456341L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 1536398230L,
    -7171435L, -778857580L, -1013804395L, -1752067691L, 1334940052L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, -544042347L,
    -7171435L, -8086625L, -16061704L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16126983L, -8217439L,
    -7171435L, -426601835L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, -1097690475L, -23948651L,
    579833750L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 76714645L, 1452446357L, -1986882923L,
    -1785556331L, 1720881813L, 361927317L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, -2070703211L, -7171435L,
    -7171435L, -10570822L, -16649985L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16649985L, -10636101L, -7171435L, -7171435L,
    -2020503147L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 596808338L, -23948651L, -1114467692L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 747803285L, -829255019L, -7171435L, -7171435L, -7171435L,
    -7171435L, -326004074L, 1418891925L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 127046290L, -728591723L, -7171435L, -7171435L,
    -9786446L, -15603981L, -16715521L, -16715521L, -16715521L, -15538958L,
    -9655375L, -7171435L, -7171435L, -661482859L, 144678815L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -2053991786L, -7171435L, 1502778005L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 278041237L, -443444587L,
    -7171435L, -10963009L, -14492954L, -15015956L, -12335666L, -24340839L,
    -40725867L, 999461525L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 848598164L, -225275243L, -7171435L, -7171435L, -7171435L,
    -8347998L, -9720911L, -8348254L, -7171435L, -7171435L, -7171435L,
    -225275243L, 949129878L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 61516458L, -443379051L, -292384107L,
    127046290L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -1835887979L, -7171435L, -12008757L, -16715521L,
    -16715521L, -16715521L, -16715521L, -14492954L, -24013930L, -745368939L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    546279319L, -1114467692L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -7171435L, -7171435L, -1064136043L, 546279319L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 1301451413L, -7171435L, -1835822188L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -795700587L, -24340838L, -16519427L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -9917004L, -7171435L, 361927317L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    33554431L, 1469289365L, -1752067691L, -896363883L, -242052459L,
    -141389163L, -7171435L, -309095531L, 429496729L, 1301451413L,
    -2104389227L, -1215130987L, -879586667L, -1701670251L, 1704104597L,
    798134930L, 75530368L, 16777215L, -1332571499L, -7171435L, 798134930L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -174943595L, -9067350L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -11420476L, -7171435L,
    999461525L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -1986948715L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -158166379L,
    -1517120875L, -74280299L, -879586667L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -812477803L, -24340839L, -16519427L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -9851469L, -7171435L, 328372885L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 261724569L, -1248685419L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -7566182L,
    -8355679L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -1869376618L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, -1902996843L,
    -7171435L, -11681849L, -16715521L, -16715521L, -16715521L, -16715521L,
    -14166045L, -7236714L, -208498027L, 882086803L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 1150456470L, -493710699L, -7171435L, -7171435L, -7303018L,
    -10789959L, -13026608L, -14934812L, -16513548L, -16645131L, -15921426L,
    -14013478L, -11973946L, -8618845L, -7171435L, -7171435L, -23948651L,
    -1768779114L, 144678815L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 227709589L, -544107883L, -7171435L,
    -10570822L, -13969951L, -14492954L, -11943478L, -24210280L, -23948651L,
    -7171435L, -23948651L, -1517186668L, 529831060L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 596808338L, -174943595L,
    -7171435L, -7171435L, -8684636L, -14605855L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16316174L, -11382080L, -7237226L, -7171435L, -7171435L, -1852665195L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 663917205L, -929918315L, -7171435L, -7171435L,
    -7171435L, -7171435L, -393112938L, 1284674197L, 1049661588L,
    -879586667L, -7171435L, -141389163L, -1986948715L, 261724569L,
    16777215L, 16777215L, 16777215L, 41975936L, -1013804395L, -7171435L,
    -7171435L, -11184706L, -16316174L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -14342690L, -8158305L, -7171435L, -23948651L,
    1066570390L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 59937429L, 1234342549L, 2140312213L,
    -1936551275L, 1486000789L, 294818453L, 16777215L, 16777215L,
    33554431L, 1519621014L, -527265131L, -7171435L, -342715755L,
    1821545109L, 93952409L, 16777215L, 1922142614L, -7171435L, -7171435L,
    -9868880L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -14210851L, -7237227L, -7171435L,
    -560819563L, 211129749L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 144678815L, 1989383061L, -258829675L, -7171435L, -644705643L,
    1804767894L, -141389163L, -7171435L, -7829349L, -15658261L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -11184706L, -7171435L, -7171435L, -1785622123L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 344755353L, -1835822188L, -91057515L, -7171435L, -7171435L,
    -7171435L, -13289772L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16250383L, -8421470L, -7171435L, -292384107L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 647271572L, -409824619L, -7171435L, -7566183L, -16513548L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -11447872L,
    -7171435L, -7171435L, 613782933L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -460090475L, -7171435L, -9342293L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -13421357L, -7171435L, -7171435L,
    1502778005L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    965907093L, -1785556331L, -879586667L, -158166379L, -695037291L,
    -1584229739L, 1435669141L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 412456341L, -7171435L,
    -7171435L, -11184706L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -15263513L, -7171435L, -7171435L, -1903062635L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 143823509L, -1936551275L, -40725867L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -1299017067L, 412258965L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 1200853907L, -7171435L, -7171435L, -12895025L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16579339L, -7566183L, -7171435L, -1114467692L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1231908203L, -7171435L, -7171435L, -7171435L, -8282719L,
    -9655375L, -8544092L, -7236714L, -7171435L, -7171435L, -577596779L,
    194155157L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    747737495L, -7171435L, -7171435L, -11908411L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -15987217L, -7171435L,
    -7171435L, -1483566443L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 1720881813L, -7171435L,
    -7171435L, -8348254L, -14231324L, -16715521L, -16715521L, -16715521L,
    -15212050L, -9263188L, -7171435L, -7171435L, -1768779115L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 61516458L, -158166379L,
    -7171435L, -10000462L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -14145060L, -7171435L, -7171435L, -91057515L,
    -1315794284L, 1603375510L, 295081622L, 16777215L, 16777215L,
    16777215L, 16777215L, 127046293L, -242052459L, -7171435L, -7629158L,
    -15538958L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16519427L, -8740442L, -7171435L, -23948651L, 747803285L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, -963472747L, -7171435L,
    -8158305L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -12237111L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -74280299L, -1164865131L, 1754502038L, 412456341L,
    16777215L, 915575445L, -7171435L, -7171435L, -12008757L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -13773857L, -7171435L, -7171435L, 1720881813L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, -1819110763L, -7171435L, -7171435L,
    -15263513L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -9868879L, -7171435L, -74280299L, 1368560277L, -1651338603L,
    -325938539L, -7171435L, -7171435L, -7171435L, -40725867L, -1013804395L,
    -1382903147L, -7171435L, -7171435L, -14100510L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16061960L,
    -7171435L, -7171435L, -1668115819L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 1402180499L, -7171435L, -7171435L, -9539923L,
    -16579339L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -13816104L, -7171435L,
    -7171435L, -946695531L, 16777215L, 16777215L, 61516458L, 1116967831L,
    -1802333548L, -460090475L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -14558233L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16388613L, -7302250L, -7171435L,
    -1433234795L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1198353770L, -7171435L, -7171435L, -12500278L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -15987217L, -8092514L, -7171435L, -74280299L, 898666645L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 33554431L,
    949129878L, -1970105706L, -443379050L, -7171435L, -7171435L,
    -12793389L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -14558233L, -7171435L, -7171435L, 1972540053L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    479367826L, -258829675L, -7171435L, -7500391L, -14737438L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16447757L,
    -10263627L, -7171435L, -7171435L, -2070703211L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 294818453L, -23948651L, -7171435L, -8478812L, -16323334L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -9917005L, -7171435L, -7171435L, 1083347605L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 1603375510L,
    -7171435L, -7171435L, -7434600L, -12237111L, -16513548L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -15000603L, -9013337L, -7171435L, -7171435L,
    -778923371L, 109084842L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -1768779115L, -7171435L, -7171435L, -10178634L, -16061960L, -16715521L,
    -16715521L, -16715521L, -16388612L, -11224382L, -7171435L, -7171435L,
    -997027179L, 43160213L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 33554431L, -728591723L, -7171435L,
    -7171435L, -7171435L, -9276502L, -14605855L, -16513549L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -15789843L, -12171320L,
    -7368809L, -7171435L, -7171435L, -376270187L, 781226134L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 194155157L, -577596779L,
    -7171435L, -7171435L, -7890531L, -10636100L, -12335666L, -11028288L,
    -8413533L, -7171435L, -7171435L, -174943595L, 613585557L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 33554431L, 579833750L, 261724569L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    160337550L, -1416457579L, -7171435L, -7171435L, -124611948L,
    -7171435L, -7171435L, -7171435L, -7500391L, -9342293L, -11316288L,
    -12171320L, -10263627L, -8355679L, -7171435L, -7171435L, -7171435L,
    -7171435L, -1416457579L, 344755353L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 647139989L, -913141099L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -476933483L, 1150456469L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 1184010901L, -1131244907L,
    -275606891L, -7171435L, -23948651L, -644705643L, -1768779114L,
    311332508L, 16777215L, 16777215L, 16777215L, 16777215L, 379033495L,
    -929852523L, -7171435L, -23948651L, 2056426132L, 428838809L,
    -1282305642L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -325938539L,
    1485934996L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 59937429L,
    2039648917L, -711814507L, -40725867L, -7171435L, -7171435L, -510487915L,
    -1752001899L, 261264021L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 211129749L, -1701670251L, -7171435L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -426601835L,
    1234408342L, 16777215L, 16777215L, 697274261L, -544042347L, -7171435L,
    -124611947L, 1485934996L, 16777215L, 16777215L, 16777215L, 1167365268L,
    -2137943659L, -1248619627L, -376270187L, -7171435L, -7171435L,
    -91057515L, -846032235L, -1752067691L, 1653772948L, 395350160L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 227709589L, 949129877L, 378704533L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -1550741099L, -7171435L, -7171435L, -8021089L,
    -11616570L, -13446949L, -12662830L, -10178634L, -7171435L, -7171435L,
    -91057515L, 831689367L, 1133613460L, -275606891L, -7171435L,
    -342715755L, 999527315L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 529831060L, 865178006L,
    144678815L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    1368494484L, -7171435L, -7171435L, -9851725L, -15996425L, -16715521L,
    -16715521L, -16715521L, -16715521L, -13904672L, -7563622L, -7171435L,
    -476933483L, -91057514L, -7171435L, -644705643L, 613782933L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, -846032235L,
    -7171435L, -8217439L, -16061704L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -12727853L, -7171435L, -7171435L,
    -7171435L, -1030581611L, 311332508L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 109084842L, -91057515L, -7171435L, -12139828L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16649985L, -7890531L, -7171435L, -695037291L, 109084842L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    831689367L, -7171435L, -7171435L, -13970208L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -9720911L, -7171435L, -1080913258L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 596808338L, -7171435L,
    -7171435L, -13512485L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -9197652L, -7171435L,
    -1299017067L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 33554431L, -258829675L, -7171435L, -11355453L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16192519L, -7498343L, -7171435L, 2089980564L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1265462635L, -7171435L, -7367273L, -14950677L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -10897730L,
    -7171435L, -7171435L, 1049661588L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 630296984L,
    -174943595L, -7171435L, -8086625L, -14100766L, -16715521L, -16715521L,
    -16715521L, -16323077L, -11028288L, -7171435L, -7171435L, -1550741099L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 1821545109L, -7171435L,
    -7171435L, -7236971L, -8740186L, -10439750L, -9655375L, -7825252L,
    -7171435L, -7171435L, -476933483L, 277843855L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 1385337493L, -376270187L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -1332571499L,
    395350160L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 344755353L, 1922142614L, -1533898091L,
    -728591723L, -1080913258L, -1903062635L, 1284805780L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L
  ), .Dim = c(64L, 64L), class = "nativeRaster", channels = 4L)

i.vertex.default <- list(
  color = 1,
  size = 15,
  size2 = 15,
  label = i.get.labels,
  label.degree = -pi / 4,
  label.color = "darkblue",
  label.dist = 0,
  label.family = "serif",
  label.font = 1,
  label.cex = 1,
  frame.color = "black",
  frame.width = 1,
  shape = "circle",
  pie = 1,
  pie.color = list(c(
    "white", "lightblue", "mistyrose",
    "lightcyan", "lavender", "cornsilk"
  )),
  pie.angle = 45,
  pie.density = -1,
  pie.lty = 1,
  raster = .igraph.logo.raster
)

i.edge.default <- list(
  color = "darkgrey",
  label = i.get.edge.labels,
  lty = 1,
  width = 1,
  loop.angle = 0,
  loop.angle2 = 0,
  label.family = "serif",
  label.font = 1,
  label.cex = 1,
  label.color = "darkblue",
  label.x = NULL,
  label.y = NULL,
  arrow.size = 1,
  arrow.mode = i.get.arrow.mode,
  curved = curve_multiple,
  arrow.width = 1
)

i.plot.default <- list(
  palette = categorical_pal(8),
  layout = layout_nicely,
  margin = c(0, 0, 0, 0),
  rescale = TRUE,
  asp = 1,
  frame = FALSE,
  main = i.get.main,
  sub = "",
  xlab = i.get.xlab,
  ylab = ""
)

i.default.values <- new.env()

i.default.values[["vertex"]] <- i.vertex.default
i.default.values[["edge"]] <- i.edge.default
i.default.values[["plot"]] <- i.plot.default

#' Using pie charts as vertices in graph plots
#'
#' More complex vertex images can be used to express addtional information
#' about vertices. E.g. pie charts can be used as vertices, to denote vertex
#' classes, fuzzy classification of vertices, etc.
#'
#' The vertex shape \sQuote{pie} makes igraph draw a pie chart for every
#' vertex. There are some extra graphical vertex parameters that specify how
#' the pie charts will look like: \describe{
#' \item{pie}{Numeric vector, gives
#' the sizes of the pie slices.}
#' \item{pie.color}{A list of color vectors to
#' use for the pies. If it is a list of a single vector, then this is used for
#' all pies. It the color vector is shorter than the number of areas in a pie,
#' then it is recycled.}
#' \item{pie.angle}{The slope of shading lines, given as
#' an angle in degrees (counter-clockwise).}
#' \item{pie.density}{The density of
#' the shading lines, in lines per inch. Non-positive values inhibit the
#' drawing of shading lines.}
#' \item{pie.lty}{The line type of the border of the
#' slices.} }
#'
#' @aliases vertex.shape.pie
#' @author Gabor Csardi \email{csardi.gabor@@gmail.com}
#' @seealso [igraph.plotting()], [plot.igraph()]
#' @keywords graphs
#' @examples
#'
#' g <- make_ring(10)
#' values <- lapply(1:10, function(x) sample(1:10,3))
#' if (interactive()) {
#'   plot(g, vertex.shape="pie", vertex.pie=values,
#'        vertex.pie.color=list(heat.colors(5)),
#'        vertex.size=seq(10,30,length.out=10), vertex.label=NA)
#' }
#' @rdname vertex.shape.pie
#' @name vertex.shape.pie
NULL
