% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map.R
\name{map_if}
\alias{map_if}
\alias{map_at}
\alias{map_depth}
\title{Apply a function to each element of a vector conditionally}
\usage{
map_if(.x, .p, .f, ..., .else = NULL)

map_at(.x, .at, .f, ...)

map_depth(.x, .depth, .f, ..., .ragged = FALSE)
}
\arguments{
\item{.x}{A list or atomic vector.}

\item{.p}{A single predicate function, a formula describing such a
predicate function, or a logical vector of the same length as \code{.x}.
Alternatively, if the elements of \code{.x} are themselves lists of
objects, a string indicating the name of a logical element in the
inner lists. Only those elements where \code{.p} evaluates to
\code{TRUE} will be modified.}

\item{.f}{A function, formula, or vector (not necessarily atomic).

If a \strong{function}, it is used as is.

If a \strong{formula}, e.g. \code{~ .x + 2}, it is converted to a function. There
are three ways to refer to the arguments:
\itemize{
\item For a single argument function, use \code{.}
\item For a two argument function, use \code{.x} and \code{.y}
\item For more arguments, use \code{..1}, \code{..2}, \code{..3} etc
}

This syntax allows you to create very compact anonymous functions.

If \strong{character vector}, \strong{numeric vector}, or \strong{list}, it is
converted to an extractor function. Character vectors index by
name and numeric vectors index by position; use a list to index
by position and name at different levels. If a component is not
present, the value of \code{.default} will be returned.}

\item{...}{Additional arguments passed on to the mapped function.}

\item{.else}{A function applied to elements of \code{.x} for which \code{.p}
returns \code{FALSE}.}

\item{.at}{A character vector of names, positive numeric vector of
positions to include, or a negative numeric vector of positions to
exlude. Only those elements corresponding to \code{.at} will be modified.
If the \code{tidyselect} package is installed, you can use \code{vars()} and
the \code{tidyselect} helpers to select elements.}

\item{.depth}{Level of \code{.x} to map on. Use a negative value to
count up from the lowest level of the list.
\itemize{
\item \code{map_depth(x, 0, fun)} is equivalent to \code{fun(x)}.
\item \code{map_depth(x, 1, fun)} is equivalent to \code{x <- map(x, fun)}
\item \code{map_depth(x, 2, fun)} is equivalent to \code{x <- map(x, ~ map(., fun))}
}}

\item{.ragged}{If \code{TRUE}, will apply to leaves, even if they're not
at depth \code{.depth}. If \code{FALSE}, will throw an error if there are
no elements at depth \code{.depth}.}
}
\description{
The functions \code{map_if()} and \code{map_at()} take \code{.x} as input, apply
the function \code{.f} to some of the elements of \code{.x}, and return a
list of the same length as the input.
\itemize{
\item \code{map_if()} takes a predicate function \code{.p} as input to determine
which elements of \code{.x} are transformed with \code{.f}.
\item \code{map_at()} takes a vector of names or positions \code{.at} to specify
which elements of \code{.x} are transformed with \code{.f}.
}

\itemize{
\item \code{map_depth()} allows to apply \code{.f} to a specific
depth level of a nested vector.
}
}
\examples{
# Use a predicate function to decide whether to map a function:
map_if(iris, is.factor, as.character)

# Specify an alternative with the `.else` argument:
map_if(iris, is.factor, as.character, .else = as.integer)

# Use numeric vector of positions select elements to change:
iris \%>\% map_at(c(4, 5), is.numeric)

# Use vector of names to specify which elements to change:
iris \%>\% map_at("Species", toupper)

# Use `map_depth()` to recursively traverse nested vectors and map
# a function at a certain depth:
x <- list(a = list(foo = 1:2, bar = 3:4), b = list(baz = 5:6))
str(x)
map_depth(x, 2, paste, collapse = "/")

# Equivalent to:
map(x, map, paste, collapse = "/")
}
\seealso{
Other map variants: \code{\link{imap}},
  \code{\link{invoke}}, \code{\link{lmap}},
  \code{\link{map2}}, \code{\link{map}},
  \code{\link{modify}}
}
\concept{map variants}
