% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cnd-abort.R
\name{local_error_call}
\alias{local_error_call}
\title{Set local error call in an execution environment}
\usage{
local_error_call(call, frame = caller_env())
}
\arguments{
\item{call}{This can be:
\itemize{
\item A call to be used as context for an error thrown in that
execution environment.
\item The \code{NULL} value to show no context.
\item An execution environment, e.g. as returned by \code{\link[=caller_env]{caller_env()}}.
The \code{\link[=sys.call]{sys.call()}} for that environment is taken as context.
}}

\item{frame}{The execution environment in which to set the local
error call.}
}
\description{
\code{local_error_call()} is an alternative to explicitly passing a
\code{call} argument to \code{\link[=abort]{abort()}}. It sets the call (or a value that
indicates where to find the call, see below) in a local binding
that is automatically picked up by \code{\link[=abort]{abort()}}.
}
\section{Motivation for setting local error calls}{


By default \code{\link[=abort]{abort()}} uses the function call of its caller as
context in error messages:

\if{html}{\out{<div class="sourceCode">}}\preformatted{foo <- function() abort("Uh oh.")
foo()
#> Error in `foo()`: Uh oh.
}\if{html}{\out{</div>}}

This is not always appropriate. For example a function that checks
an input on the behalf of another function should reference the
latter, not the former:

\if{html}{\out{<div class="sourceCode">}}\preformatted{arg_check <- function(arg,
                      error_arg = as_string(substitute(arg))) \{
  abort(cli::format_error("\{.arg \{error_arg\}\} is failing."))
\}

foo <- function(x) arg_check(x)
foo()
#> Error in `arg_check()`: `x` is failing.
}\if{html}{\out{</div>}}

The mismatch is clear in the example above. \code{arg_check()} does not
have any \code{x} argument and so it is confusing to present
\code{arg_check()} as being the relevant context for the failure of the
\code{x} argument.

One way around this is to take a \code{call} or \code{error_call} argument
and pass it to \code{abort()}. Here we name this argument \code{error_call}
for consistency with \code{error_arg} which is prefixed because there is
an existing \code{arg} argument. In other situations, taking \code{arg} and
\code{call} arguments might be appropriate.

\if{html}{\out{<div class="sourceCode">}}\preformatted{arg_check <- function(arg,
                      error_arg = as_string(substitute(arg)),
                      error_call = caller_env()) \{
  abort(
    cli::format_error("\{.arg \{error_arg\}\} is failing."),
    call = error_call
  )
\}

foo <- function(x) arg_check(x)
foo()
#> Error in `foo()`: `x` is failing.
}\if{html}{\out{</div>}}

This is the generally recommended pattern for argument checking
functions. If you mention an argument in an error message, provide
your callers a way to supply a different argument name and a
different error call. \code{abort()} stores the error call in the \code{call}
condition field which is then used to generate the "in" part of
error messages.

In more complex cases it's often burdensome to pass the relevant
call around, for instance if your checking and throwing code is
structured into many different functions. In this case, use
\code{local_error_call()} to set the call locally or instruct \code{abort()}
to climb the call stack one level to find the relevant call. In the
following example, the complexity is not so important that sparing
the argument passing makes a big difference. However this
illustrates the pattern:

\if{html}{\out{<div class="sourceCode">}}\preformatted{arg_check <- function(arg,
                      error_arg = caller_arg(arg),
                      error_call = caller_env()) \{
  # Set the local error call
  local_error_call(error_call)

  my_classed_stop(
    cli::format_error("\{.arg \{error_arg\}\} is failing.")
  )
\}

my_classed_stop <- function(message) \{
  # Forward the local error call to the caller's
  local_error_call(caller_env())

  abort(message, class = "my_class")
\}

foo <- function(x) arg_check(x)
foo()
#> Error in `foo()`: `x` is failing.
}\if{html}{\out{</div>}}
}

\section{Error call flags in performance-critical functions}{


The \code{call} argument can also be the string \code{"caller"}. This is
equivalent to \code{caller_env()} or \code{parent.frame()} but has a lower
overhead because call stack introspection is only performed when an
error is triggered. Note that eagerly calling \code{caller_env()} is
fast enough in almost all cases.

If your function needs to be really fast, assign the error call
flag directly instead of calling \code{local_error_call()}:

\if{html}{\out{<div class="sourceCode">}}\preformatted{.__error_call__. <- "caller"
}\if{html}{\out{</div>}}
}

\examples{
# Set a context for error messages
function() {
  local_error_call(quote(foo()))
  local_error_call(sys.call())
}

# Disable the context
function() {
  local_error_call(NULL)
}

# Use the caller's context
function() {
  local_error_call(caller_env())
}
}
