/*
*  RAL -- Rubrica Addressbook Library
*  file: card.c
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <time.h>
#include <string.h>
#include <glib.h>
#include <glib-object.h>

#include "card.h"
#include "group_box.h"
#include "group.h"
#include "utils.h"
#include "ref.h"

glong id = 0;



/*  property enumeration
*/
enum {
  CARD_NAME = 1,
  CARD_ID,
  CARD_TYPE,
  CARD_RATE,
  CARD_LOCKED,
  CARD_DELETED,
  CARD_DESTROYED,
  CARD_MARKED,
  CARD_CREATED,
  CARD_CHANGED
};
   

struct _RCardPrivate {
  glong      id;           /* card's unique id                 */

  gchar*     name;         /* card's name                      */
  gchar*     type;         /* card's type                      */
  RRate      rate;         /* card's rate                      */
     
  gboolean   locked;       /* true if card is locked           */
  gboolean   deleted;      /* is card marked as deleted?       */
  gboolean   destroyed;    /* is card marked as destroyed?     */
  gboolean   marked;       /* true if card is marked           */
  
  RGroupBox* box;          /* card's groups set                */
  
  GList*     addresses;
  GList*     nets;         /* web, email and irc uries         */
  GList*     phones;

  GList*     refs;         /* references to other cards        */
  GList*     linked;       /* cards that have a reference to this one */

  time_t     created;      /* card's creation time             */
  time_t     changed;      /* card's last change time          */

  GList*     grp_iter;     /* groups, addresses, telephones,   */
  GList*     addr_iter;    /* nets and refs iterators          */
  GList*     net_iter;
  GList*     phone_iter;
  GList*     refs_iter;
  
  gboolean   dispose_has_run;
};


static void r_card_class_init   (RCardClass* klass);
static void r_card_init         (RCard* self);

static void r_card_dispose      (RCard* self);
static void r_card_finalize     (RCard* self);

static void r_card_set_property (GObject* obj, guint property_id,
				 const GValue* value, GParamSpec* spec);
static void r_card_get_property (GObject* obj, guint property_id,
				 GValue* value, GParamSpec* spec);


static void   r_card_copy_commons           (RCard* new,  RCard* old);
static void   r_card_copy_paste_group       (RCard* card, gpointer data);
static void   r_card_copy_paste_ref         (RCard* card, gpointer data);
static void   r_card_copy_paste_address     (RCard* card, gpointer data);
static void   r_card_copy_paste_net_address (RCard* card, gpointer data);
static void   r_card_copy_paste_telephone   (RCard* card, gpointer data);


/*  Private functions
*/
GType
r_card_get_type()
{
  static GType r_card_type = 0;
  
  if (!r_card_type)
    {
      static const GTypeInfo r_card_info =
	{
	  sizeof(RCardClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) r_card_class_init,
	  NULL,
	  NULL,
	  sizeof(RCard),
	  0,
	  (GInstanceInitFunc) r_card_init
	};

      r_card_type = g_type_register_static (G_TYPE_OBJECT, "RCard",
					    &r_card_info, 0);
    }

  return r_card_type;
}


static void
r_card_class_init(RCardClass* klass)
{
  GObjectClass *class;
  GParamSpec* pspec;
  
  klass->free            = NULL;      /* free method, child will set */
 
  klass->copy            = NULL;      /* copy a card */
  klass->search          = NULL;      /* search a string into the card */

  klass->get_infos       = NULL;      /* return infos on card record   */
  klass->print_card      = NULL;      /* unused */


  class = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) r_card_dispose;
  class->finalize = (GObjectFinalizeFunc) r_card_finalize;

  class->set_property = r_card_set_property;
  class->get_property = r_card_get_property;

  /**
   * RCard:card-id:
   *
   * card's id
   */
  pspec = g_param_spec_long("card-id",
			    "card's id",
			    "the id of the card",
			    G_MINLONG,
			    G_MAXLONG,
			    0,
			    G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_ID, pspec);
  
  
  /**
   * RCard:card-name:
   *
   * card's name
   */
  pspec = g_param_spec_string("card-name",
			      "card's name",
			      "the name of the card",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_NAME, pspec);     
  
  
  /**
   * RCard:card-type:
   *
   * card's type
   */
  pspec = g_param_spec_string("card-type",
			      "card's type",
			      "card's type, must be \"personal\"" 
			      "or \"company\"",
			      NULL,
			      G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_TYPE, pspec);     
  
  /**
   * RCard:rate:
   *
   * card's rate
   */
  pspec = g_param_spec_int("card-rate",
			   "card's rate",
			   "the rate user wants for the card",
			   R_RATE_NONE,
			   R_RATE_VERY_HIGHT,
			   R_RATE_NORMAL,
			   G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_RATE, pspec);     
  
  /**
   * RCard:locked:
   *
   * is the card locked?
   */
  pspec = g_param_spec_boolean("card-locked",
			       "locked",
			       "is the card locked",
			       TRUE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_LOCKED, pspec);     
  
  /**
   * RCard:deleted:
   *
   * is the card deleted?
   */
  pspec = g_param_spec_boolean("card-deleted",
			       "deleted",
			       "is the card deleted",
			       FALSE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_DELETED, pspec);     
  
  /**
   * RCard:destroyed:
   *
   * is the card destroyed?
   */
  pspec = g_param_spec_boolean("card-destroyed",
			       "destroyed",
			       "is the card destroyed",
			       FALSE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_DESTROYED, pspec);     
  

  /**
   * RCard:marked:
   *
   * is the card marked?
   */
  pspec = g_param_spec_boolean("card-marked",
			       "marked",
			       "is the card marked",
			       FALSE,
			       G_PARAM_CONSTRUCT | G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_MARKED, pspec);     
  
  /**
   * RCard:creation:
   *
   * card's creation time
   */
  pspec = g_param_spec_int("card-created",
			   "card's creation time",
			   "the card's creation time",
			   G_MININT,
			   G_MAXINT,
			   -1,
			   G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_CREATED, pspec);  
  
  /**
   * RCard:modification:
   *
   * card's modification time
   */
  pspec = g_param_spec_int("card-changed",
			   "card's modification time",
			   "last time that card was modified",
			   G_MININT,
			   G_MAXINT,
			   -1,
			   G_PARAM_READWRITE);
  g_object_class_install_property(class, CARD_CHANGED, pspec);  
}


static void
r_card_init(RCard* self)
{
  self->priv = g_new(RCardPrivate, 1);
  
  self->priv->id         = ++id;

  self->priv->name       = NULL;
  self->priv->type       = NULL;
  self->priv->rate       = R_RATE_NORMAL;

  self->priv->box        = r_group_box_new();

  self->priv->locked     = FALSE;
  self->priv->deleted    = FALSE;
  self->priv->destroyed  = FALSE;
  self->priv->marked     = FALSE;

  self->priv->linked     = NULL;

  self->priv->addresses  = NULL;
  self->priv->nets       = NULL;
  self->priv->phones     = NULL;

  self->priv->refs       = NULL;

  self->priv->created    = time(NULL);
  self->priv->changed    = time(NULL);

  self->priv->grp_iter   = NULL;
  self->priv->addr_iter  = NULL;
  self->priv->net_iter   = NULL;
  self->priv->phone_iter = NULL;
  self->priv->refs_iter  = NULL;

  self->priv->dispose_has_run = FALSE;
}



static void 
r_card_dispose (RCard* self)
{
  GList* del;  

  g_return_if_fail(IS_R_CARD(self));    

  if (self->priv->dispose_has_run)
    return;

  self->priv->dispose_has_run = TRUE;

  /* free card's children 
  */
  R_CARD_CLASS(R_CARD_GET_CLASS(self))->free(self);

  /* free card's data
   */
  r_utils_free_string(self->priv->name);
  r_utils_free_string(self->priv->type);
  self->priv->name = NULL;
  self->priv->type = NULL;

  /* groups */
  r_group_box_free(R_GROUP_BOX(self->priv->box));

  /* addresses */
  del = self->priv->addresses;
  for (; del; del = del->next)
    {
      RAddress* address = del->data;;

      if(address)
	r_address_free(R_ADDRESS(address));
    }
  self->priv->addresses = NULL;

  /* net addresses */
  del = self->priv->nets;
  for (; del; del = del->next)
    {
      RNetAddress* net = del->data;;

      if(net)
	r_net_address_free(R_NET_ADDRESS(net));
    }
  self->priv->nets = NULL;

  /* telephone numbers */
  del = self->priv->phones;
  for (; del; del = del->next)
    {
      RTelephone* tel = del->data;;

      if(tel)
	r_telephone_free(R_TELEPHONE(tel));
    }
  self->priv->phones = NULL;   
 
  /*  delete cards's references
   */
  for(; self->priv->refs_iter; 
      self->priv->refs_iter = self->priv->refs_iter->next)
    if(self->priv->refs_iter->data)
      {
	r_ref_free(R_REF(self->priv->refs_iter->data));
	// cancellare card del contatto ?
      }

  g_list_free(self->priv->refs);
  g_list_free(self->priv->refs_iter);
  self->priv->refs = self->priv->refs_iter = NULL;

  self->priv->grp_iter   = NULL;
  self->priv->addr_iter  = NULL;
  self->priv->net_iter   = NULL;
  self->priv->phone_iter = NULL;
}


static void 
r_card_finalize (RCard* self)
{
  g_return_if_fail(IS_R_CARD(self));

  g_free(self->priv);
  self->priv = NULL;
}


static void 
r_card_set_property (GObject* obj, guint property_id,
		     const GValue* value, GParamSpec* spec)
{
  RCard* self = (RCard*) obj;

  switch(property_id)
    {
    case CARD_ID:
      self->priv->id = g_value_get_long(value);
      break;

    case CARD_TYPE:
      g_free(self->priv->type);
      self->priv->type = g_value_dup_string(value);
      break;
      
    case CARD_NAME:
      g_free(self->priv->name);
      self->priv->name = g_value_dup_string(value);
      break;

    case CARD_RATE:
      self->priv->rate = g_value_get_int(value);
      break;

    case CARD_LOCKED:
      self->priv->locked = g_value_get_boolean(value);
      break;

    case CARD_DELETED:
      self->priv->deleted = g_value_get_boolean(value);
      break;

    case CARD_DESTROYED:
      self->priv->destroyed = g_value_get_boolean(value);
      break;

    case CARD_MARKED:
      self->priv->marked = g_value_get_boolean(value);
      break;
      
    case CARD_CREATED:
      self->priv->created = g_value_get_int(value);
      break;

    case CARD_CHANGED:
      self->priv->changed = g_value_get_int(value);
      break;

    default:
      break;
    }
}

static void 
r_card_get_property (GObject* obj, guint property_id,
		     GValue* value, GParamSpec* spec)
{
  RCard* self = (RCard*) obj;

  switch(property_id)
    {
    case CARD_ID:
      g_value_set_long(value, self->priv->id);
      break;

    case CARD_NAME:
      g_value_set_string(value, self->priv->name);
      break;

    case CARD_TYPE:
      g_value_set_string(value, self->priv->type);   
      break;

    case CARD_RATE:
      g_value_set_int(value, self->priv->rate);
      break;

    case CARD_LOCKED:
      g_value_set_boolean(value, self->priv->locked);
      break;

    case CARD_DELETED:
      g_value_set_boolean(value, self->priv->deleted);
      break;

    case CARD_DESTROYED:
      g_value_set_boolean(value, self->priv->destroyed);
      break;

    case CARD_MARKED:
      g_value_set_boolean(value, self->priv->marked);
      break;
      
    case CARD_CREATED:
      g_value_set_int(value, self->priv->created);
      break;

    case CARD_CHANGED:
      g_value_set_int(value, self->priv->changed);
      break;
      
    default:
      break;
    }
}



static void 
r_card_copy_paste_group(RCard* card, gpointer data)
{
  RGroup* grp;
      
  grp = r_group_copy(R_GROUP(data));
  if (grp)
    r_card_add_group(R_CARD(card), grp);  
}


static void   
r_card_copy_paste_ref (RCard* card, gpointer data)
{
  RRef* ref;
  
  ref = r_ref_copy(R_REF(data));
  if (ref)
    r_card_add_ref(card, ref);
}


static void   
r_card_copy_paste_address (RCard* card, gpointer data)
{
  RAddress* add;
  
  add = r_address_copy(R_ADDRESS(data));
  if (add)
    r_card_add_address(R_CARD(card), add);
}


static void   
r_card_copy_paste_net_address (RCard* card, gpointer data)
{
  RNetAddress* net;
  
  net = r_net_address_copy(R_NET_ADDRESS(data));
  if (net)
    r_card_add_net_address(R_CARD(card), net);  
}


static void   
r_card_copy_paste_telephone (RCard* card, gpointer data)
{
  RTelephone* tel;

  tel = r_telephone_copy(R_TELEPHONE(data));
  if (tel)
    r_card_add_telephone(R_CARD(card), tel);
}


/*
 * r_card_copy_commons 
 * private method. Copy from the old card to the new one the 
 * RPersonalCard and RCompanycard common values, it est:
 * card infos, groups, addresses, net addresses, telephone numbers
 */
static void
r_card_copy_commons (RCard* new, RCard* old)
{
  gchar *name;
  gint rate;
  gboolean locked, deleted, marked;
  time_t created, changed;

  g_return_if_fail(IS_R_CARD(new));
  g_return_if_fail(IS_R_CARD(old));
  
  g_object_get(G_OBJECT(old), 
	       "card-name", &name, "card-rate", &rate,
	       "card-locked", &locked, "card-deleted", &deleted, 
	       "card-marked", &marked, "card-created", &created, 
	       "card-changed", &changed, NULL);

  g_object_set(G_OBJECT(new), 
	       "card-name", name, "card-rate", rate,
	       "card-locked", locked, "card-deleted", deleted, 
	       "card-marked", marked, "card-created", created, 
	       "card-changed", changed, NULL);

  r_card_foreach_group       (old, (RFunc) r_card_copy_paste_group,       new);
  r_card_foreach_address     (old, (RFunc) r_card_copy_paste_address,     new);
  r_card_foreach_net_address (old, (RFunc) r_card_copy_paste_net_address, new);
  r_card_foreach_telephone   (old, (RFunc) r_card_copy_paste_telephone,   new);
  r_card_foreach_ref         (old, (RFunc) r_card_copy_paste_ref,         new);
}



/* ************************************  Public Functions
 */

/**
 * r_card_new
 *
 * create a new #RCard
 *
 * Returns: a new allocated #RCard*
 */
RCard*  
r_card_new(void)
{
  RCard* card;

  card = g_object_new(r_card_get_type(), NULL);

  return card;
}


/* 
  Returns: #R_CARD_UNDELETED if an error occurred, 
  #R_CARD_HAS_REFS if the given card has got references to 
  other cards (you must delete the referenced card before), or 
  #R_CARD_DELETED if card was successfully deleted
*/


/**
 * r_card_free
 * @card: a #RCard
 * 
 * free the #RCard*
 */
void
r_card_free(RCard* card)
{
  g_return_if_fail(IS_R_CARD(card));
  
  g_object_unref(G_OBJECT(card));
}

/**
 * r_card_id_align
 * @card:
 * @id:
 *
 * align the card's id. User shouldn't use this function.
 */
void
r_card_reassign_id (RCard* self, glong card_id)
{
  g_return_if_fail(IS_R_CARD(self)); 
  g_return_if_fail(card_id > 0L);

  if (card_id > id)
    {
      g_object_set(self, "card-id", card_id, NULL); 
      id = card_id;       
    }
}


gboolean 
r_card_is_personal (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  
  if (g_ascii_strcasecmp(card->priv->type, "personal") == 0)
    return TRUE;
  
  return FALSE;
}


gboolean 
r_card_is_deleted (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  
  return card->priv->deleted;
}



/**
 * r_card_search
 * @card: a #RCard
 * @str: the string that user is looking for
 *
 * search the given string in the card
 *
 * returns: %TRUE if string has been found, %FALSE otherwise.
 * If str is %NULL, %FALSE is returned
 */
gboolean 
r_card_search (RCard* card, const gchar* str)
{
  RCardClass* klass;

  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  g_return_val_if_fail(str != NULL, FALSE);
  
  klass = R_CARD_GET_CLASS(card);
  if (R_CARD_CLASS(klass)->search)
    return R_CARD_CLASS(klass)->search(card, str);

  return FALSE;
}



/**
 * r_card_copy
 * @card: a #RCard
 *
 * copy the given #RCard*
 *
 * Returns: a new allocated copy of the card
 */
RCard*
r_card_copy (RCard* card)
{
  RCardClass* klass;
  RCard* new;
  
  g_return_val_if_fail(IS_R_CARD(card), NULL);   

  klass = R_CARD_GET_CLASS(card);

  if (R_CARD_CLASS(klass)->copy)
    {
      new = R_CARD_CLASS(klass)->copy(R_CARD(card));    
      r_card_copy_commons(R_CARD(new), R_CARD(card));

      return new;
    }
  
  return NULL;
}



/**
 * r_card_get_infos
 * @card: a #RCard
 *
 * get infos about card. 
 *
 * returns: a #RInfos. caller must free the #RInfos, calling
 * the #r_card_free_infos
 */
RInfos* 
r_card_get_infos (RCard* card)
{
  RCardClass* klass;
  RInfos* infos;
  gpointer address, tel;

  g_return_val_if_fail(IS_R_CARD(card), NULL);

  klass = R_CARD_GET_CLASS(card);

  if (R_CARD_CLASS(klass)->get_infos)
    {
      infos = R_CARD_CLASS(klass)->get_infos(R_CARD(card));
      
      g_object_get(R_CARD(card), 
		   "card-id", &infos->id, 
		   "card-name", &infos->label, NULL);

      infos->web   = g_strdup(r_card_get_home_page(R_CARD(card)));
      infos->email = g_strdup(r_card_get_email(R_CARD(card)));
      infos->irc   = g_strdup(r_card_get_irc(R_CARD(card)));
 
      address = r_card_get_address(R_CARD(card));
      tel     = r_card_get_telephone(R_CARD(card));
      
      if (address)
	g_object_get(R_ADDRESS(address), 
		     "city", &infos->city, 
		     "country", &infos->country, NULL); 
      else
	infos->city = infos->country = g_strdup(NULL);
      
      
      if (tel)
	g_object_get(R_TELEPHONE(tel), 
		     "telephone-number", &infos->telephone, NULL);
      else
	infos->telephone = g_strdup(NULL);

      return infos;     
   }
  
  return NULL;
}


/**
 * r_card_free_infos
 * @infos: a #RInfos
 *
 * free the RInfos object
 */
void          
r_card_free_infos (RInfos *infos)
{
  g_return_if_fail(infos != NULL);
  
  g_free(infos->label);
  g_free(infos->first);
  g_free(infos->last);
  g_free(infos->prof);
  g_free(infos->city);
  g_free(infos->country);
  g_free(infos->assignment);
  g_free(infos->web);
  g_free(infos->email);
  g_free(infos->irc);
  g_free(infos->telephone);
 
  g_free(infos);
  infos = NULL;
}


/**
 * r_card_add_group
 * @card: a #RCard
 * @group: a #RGroup
 * 
 * add a group to card (card belongs to added group)
 */
void 
r_card_add_group (RCard* card, RGroup* group)
{
  g_return_if_fail(IS_R_CARD(card));
  g_return_if_fail(IS_R_GROUP(group));

  r_group_box_add_group(card->priv->box, group);
}


/**
 * r_card_delete_group
 * @card: a #RCard
 * @name: group's name
 *
 * delete a group from card's groups list (i.e. remove a card from a group)
 *
 * Returns: TRUE if group was successfully deleted, FALSE otherwise
 */
gboolean      
r_card_delete_group (RCard* card, gchar* name)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  g_return_val_if_fail(name != NULL, FALSE);

  return r_group_box_delete_group_by_name(card->priv->box, name);
}



/**
 * r_card_rename_group
 * @card: a #RCard
 * @oldname: old group's name
 * @newname: new group's name
 *
 * rename the old name group with the new name group
 *
 * Returns: TRUE if replaced, FALSE otherwise
 */
gboolean    
r_card_rename_group (RCard* card, gchar* oldname, gchar* newname)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE); 
  g_return_val_if_fail(oldname != NULL, FALSE);
  g_return_val_if_fail(newname != NULL, FALSE);

  return r_group_box_rename_group(card->priv->box, oldname, newname);
}


gboolean 
r_card_has_groups (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  
  return !r_group_box_is_empty(card->priv->box);
}



/**
 * r_card_find_group
 * @card: a #RCard
 * @name: group's name
 *
 * find a card's group by name. User must cast returned value 
 * to a #RGroup.
 *
 * Returns: a gpointer to the group if group has been found, NULL otherwise.
 */
gpointer 
r_card_find_group (RCard* card, gchar* name)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL); 
  g_return_val_if_fail(name != NULL, NULL);

  return r_group_box_find(card->priv->box, name); 
} 


/** 
 * r_card_belong_to_group
 * @card: a #RCard
 * @group_name: group's name
 * 
 * check if the card belongs to the group 
 *
 * Returns: %TRUE if card belongs to the given group, %FALSE otherwise
 */ 
gboolean  
r_card_belong_to_group (RCard* card, const gchar* group_name)
{ 
  g_return_val_if_fail(IS_R_CARD(card), FALSE);  
  g_return_val_if_fail(group_name != NULL, FALSE); 
  
  return r_group_box_owns_group(card->priv->box, group_name); 
}


/** 
 * r_card_get_groups_owned_by
 * @card: a #RCard
 * @owner: the owner of the groups
 * 
 * look for all groups that belongs to owner
 *
 * Returns: A list of #RGroup, or %NULL
 */ 
GList* 
r_card_get_groups_owned_by (RCard* card, const gchar* owner)
{  
  g_return_val_if_fail(IS_R_CARD(card), NULL);  
  g_return_val_if_fail(owner != NULL, NULL); 
  
  return r_group_box_find_groups_owned_by(card->priv->box, owner);   
}


/**
 * r_card_get_group
 * @card: a #RCard
 *
 * get the first of card's groups. User must cast returned value 
 * to a #RGroup.
 * 
 * Returns: a gpointer.
 */
RGroup* 
r_card_get_group (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);

  return r_group_box_get_group(card->priv->box);
}


/**
 * r_card_get_next_group
 * @card: a #RCard
 *
 * get the next in card's group. User must cast returned value 
 * to a #RGroup.
 * 
 * Returns: a gpointer 
 */
RGroup*
r_card_get_next_group (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card),NULL);

  return r_group_box_get_next_group(card->priv->box);
}


/**
 * r_card_get_prev_group
 * @card: a #RCard
 *
 * get the previous in card's group. User must cast returned value 
 * to a #RGroup.
 * 
 * Returns: a gpointer 
 */
RGroup*
r_card_get_prev_group (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);

  return r_group_box_get_prev_group(card->priv->box);
}



/**
 * r_card_reset_group
 * @card: a #RCard
 *
 * set the private group iterartor to the first one
 */
void 
r_card_reset_group (RCard* card)
{
  g_return_if_fail(IS_R_CARD(card));

  r_group_box_reset(card->priv->box);
}


/**
 * r_card_foreach_group
 * @card: a #RCard
 * @func: the #RFunc that will be called on each group in the card
 * @user_data: user's data to pass to func
 *
 * User's function func, will be called foreach group in card.
 * func must have two parameter: the group (automatically passed) and 
 * the user_data
 */
void          
r_card_foreach_group (RCard* card, RFunc func, gpointer user_data)
{
  gpointer grp;
  
  r_card_reset_group(card);
  grp = r_card_get_group(card);
  for(; grp; grp = r_card_get_next_group(card))
    func(user_data, grp);
}


/**
 * r_card_add_address
 * @card: a #RCard
 * @address: a #RAddress
 * 
 * add an address to card
 */
void 
r_card_add_address (RCard* card, RAddress* address)
{
  g_return_if_fail(IS_R_CARD(card));
  g_return_if_fail(IS_R_ADDRESS(address));

  card->priv->addresses = g_list_append(card->priv->addresses, address);

  /* init address iterator */
  if (!card->priv->addr_iter)
    card->priv->addr_iter = card->priv->addresses;
}



/**
 * r_card_delete_address
 * @card: a #RCard
 * @address: a #RAddress
 * 
 * delete the address from card
 *
 * returns: %TRUE if address has been successfully deleted, %FALSE otherwihe.
 */
gboolean 
r_card_delete_address (RCard* card, RAddress* address)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  g_return_val_if_fail(IS_R_ADDRESS(address), FALSE);
  
  card->priv->addr_iter = card->priv->addresses;
  for (; card->priv->addr_iter; 
       card->priv->addr_iter = card->priv->addr_iter->next)
    {
      if (card->priv->addr_iter && (card->priv->addr_iter->data == address))
	card->priv->addresses = g_list_remove_link(card->priv->addresses,
						   card->priv->addr_iter);

      r_address_free(address);
      g_list_free_1(card->priv->addr_iter);
      card->priv->addr_iter = NULL;

      return TRUE;
    }

  return FALSE;
}


/**
 * r_card_replace_address
 * @card: a #RCard
 * @old: a #RAddress
 * @new: a #RAddress
 * 
 * replace the old address with new one. On success, old address is deleted.
 *
 * returns: %TRUE if address has been successfully replaced, %FALSE otherwihe.
 */
gboolean 
r_card_replace_address (RCard* card, RAddress* old, RAddress* new)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  g_return_val_if_fail(IS_R_ADDRESS(old), FALSE);
  g_return_val_if_fail(IS_R_ADDRESS(new), FALSE);

  card->priv->addr_iter = card->priv->addresses;
  for (; card->priv->addr_iter; 
       card->priv->addr_iter = card->priv->addr_iter->next)
    {
      if (card->priv->addr_iter && (card->priv->addr_iter->data == old))
	{
	  card->priv->addr_iter->data = new;
	  r_address_free(old);
	  card->priv->addr_iter = NULL;
	  
	  return TRUE;
	}
    }

  return FALSE;
}



/**
 * r_card_get_address
 * @card: a #RCard
 *
 * get the first item of card's addresses list. User must cast to
 * a #RAddress.
 *
 * Returns: a gpointer 
 */
gpointer
r_card_get_address (RCard* card)
{
  RAddress* ret = NULL;

  g_return_val_if_fail(IS_R_CARD(card), NULL);

  if (card->priv->addr_iter)
    ret = card->priv->addr_iter->data;
  if (IS_R_ADDRESS(ret))
    return ret;
  
  return NULL;
}


/**
 * r_card_get_next_address
 * @card: a #RCard
 *
 * get the next item in card's addresses list. User must cast to
 * a #RAddress.
 * 
 * Returns: a gpointer 
 */
gpointer 
r_card_get_next_address (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  card->priv->addr_iter = g_list_next(card->priv->addr_iter);
  if(card->priv->addr_iter)
    return card->priv->addr_iter->data;
  else  /* iterator is NULL, reset iterator */
    card->priv->addr_iter = card->priv->addresses;

  return NULL;
}


/**
 * r_card_get_prev_address
 * @card: a #RCard
 *
 * get the previous item in card's addresses list. User must cast to
 * a #RAddress.
 * 
 * Returns: a gpointer 
 */
gpointer
r_card_get_prev_address (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  card->priv->addr_iter = g_list_previous(card->priv->addr_iter);
  if (card->priv->addr_iter)
    return card->priv->addr_iter->data;
  else  /* iterator is NULL, reset iterator */
    card->priv->addr_iter = card->priv->addresses;

  return NULL;
}


/**
 * r_card_reset_address
 * @card: a #RCard
 *
 * set the private address iterartor to the first one
 */
void 
r_card_reset_address (RCard* card)
{
  g_return_if_fail(IS_R_CARD(card));

  card->priv->addr_iter = card->priv->addresses;
}


/**
 * r_card_foreach_address
 * @card: a #RCard
 * @func: the #RFunc that will be called on each group in the card
 * @user_data: user's data to pass to func
 *
 * User's function func, will be called foreach address in card.
 * func must have two parameter: the address (automatically passed) and 
 * the user_data
 */
void
r_card_foreach_address (RCard* card, RFunc func, gpointer user_data)
{
  gpointer address;
  
  r_card_reset_address(card);
  address = r_card_get_address(card);
  for(; address; address = r_card_get_next_address(card))
    func(user_data, address);
}


/**
 * r_card_add_net_address 
 * @card: a #RCard
 * @net: a #RNetAddress
 *
 * add a net address to card 
 */
void 
r_card_add_net_address (RCard* card, RNetAddress* net)
{
  g_return_if_fail(IS_R_CARD(card));
  g_return_if_fail(IS_R_NET_ADDRESS(net));

  card->priv->nets = g_list_append(card->priv->nets, net);

  /* init card iterator */
  if (!card->priv->net_iter)
    card->priv->net_iter = card->priv->nets;
}


/**
 * r_card_delete_net_address 
 * @card: a #RCard
 * @address: a #RNetAddress
 *
 * delete a net address from card
 *
 * returns: %TRUE if address has been successfully deleted, %FALSE otherwihe.
 */
gboolean 
r_card_delete_net_address  (RCard* card, RNetAddress* address)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  g_return_val_if_fail(IS_R_NET_ADDRESS(address), FALSE);
  
  card->priv->net_iter = card->priv->nets;
  for (; card->priv->net_iter; 
       card->priv->net_iter = card->priv->net_iter->next)
    {
      if (card->priv->net_iter && (card->priv->net_iter->data == address))
	card->priv->nets = g_list_remove_link(card->priv->nets,
					      card->priv->net_iter);
      
      r_net_address_free(address);
      g_list_free_1(card->priv->net_iter);
      card->priv->net_iter = NULL;

      return TRUE;
    }

  return FALSE;
}


/**
 * r_card_replace_net_address 
 * @card: a #RCard
 * @old: the old #RNetAddress
 * @new: the new #RNetAddress
 *
 * replace the old net address with new one. 
 * On success, old address is deleted.
 *
 * returns: %TRUE if address has been successfully replaced, %FALSE otherwihe.
 */
gboolean 
r_card_replace_net_address (RCard* card, RNetAddress* old, RNetAddress* new)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  g_return_val_if_fail(IS_R_NET_ADDRESS(old), FALSE);
  g_return_val_if_fail(IS_R_NET_ADDRESS(new), FALSE);

  card->priv->net_iter = card->priv->nets;
  for (; card->priv->net_iter; 
       card->priv->net_iter = card->priv->net_iter->next)
    {
      if (card->priv->net_iter && (card->priv->net_iter->data == old))
	{
	  card->priv->net_iter->data = new;
	  r_net_address_free(old);
	  card->priv->net_iter = NULL;
	  
	  return TRUE;
	}
    }

  return FALSE;
}



/**
 * r_card_get_net_address
 * @card: a #RCard
 *
 * get the first item of card's net addresses list. User must cast to
 * a #RNetAddress.
 * 
 * Returns: a gpointer 
 */
gpointer
r_card_get_net_address (RCard* card)
{
  RNetAddress* ret = NULL;

  g_return_val_if_fail(IS_R_CARD(card), NULL);

  if (card->priv->net_iter)
    ret = card->priv->net_iter->data;
  if (IS_R_NET_ADDRESS(ret))
    return ret;
  
  return NULL;
}


/**
 * r_card_get_next_net_address
 * @card: a #RCard
 *
 * get the next item in card's net addresses list. User must cast to
 * a #RNetAddress.
 * 
 * Returns: a gpointer 
 */
gpointer
r_card_get_next_net_address (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  card->priv->net_iter = g_list_next(card->priv->net_iter);
  if (card->priv->net_iter) 
    return card->priv->net_iter->data;
  else  /* iterator is NULL, reset iterator */
    card->priv->net_iter = card->priv->nets;

  return NULL;  
}


/**
 * r_card_get_prev_net_address
 * @card: a #RCard
 *
 * get the previous item in card's net addresses list. User must cast to
 * a #RNetAddress.
 * 
 * Returns: a gpointer 
 */
gpointer
r_card_get_prev_net_address (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  card->priv->net_iter = g_list_previous(card->priv->net_iter);
  if (card->priv->net_iter)
    return card->priv->net_iter->data;
  else
    card->priv->net_iter = card->priv->nets;

  return NULL;  
}


/**
 * r_card_reset_net_address
 * @card: a #RCard
 *
 * set the private net addresses iterartor to the first one.
 */
void 
r_card_reset_net_address (RCard* card)
{
  g_return_if_fail(IS_R_CARD(card));
  
  card->priv->net_iter = card->priv->nets;
}


/**
 * r_card_foreach_net_address:
 * @card: a #RCard
 * @func: the #RFunc that will be called on each group in the card
 * @user_data: user's data to pass to func
 *
 * User's function func, will be called foreach net address in card.
 * func must have two parameter: the net address (automatically passed) and 
 * the user_data
 */
void
r_card_foreach_net_address (RCard* card, RFunc func, gpointer user_data)
{
  gpointer net;
  
  r_card_reset_net_address(card);
  net = r_card_get_net_address(card);
  for(; net; net = r_card_get_next_net_address(card))
    func(user_data, net);
}


/**
 * r_card_add_telephone
 * @card: a #RCard
 * @tel: a #RTelephone
 * 
 * add a telephone number to card
 */
void 
r_card_add_telephone (RCard* card, RTelephone* tel)
{
  g_return_if_fail(IS_R_CARD(card));
  g_return_if_fail(IS_R_TELEPHONE(tel));

  card->priv->phones = g_list_append(card->priv->phones, tel);

  /* init telephone iterator */
  if (!card->priv->phone_iter)
    card->priv->phone_iter = card->priv->phones;
}



/**
 * r_card_delete_telephone
 * @card: a #RCard
 * @phone: a #RTelephone
 * 
 * delete a telephone number from card
 *
 * returns: %TRUE if telephone has been successfully, %FALSE otherwise. 
 */
gboolean 
r_card_delete_telephone (RCard* card, RTelephone* phone)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  g_return_val_if_fail(IS_R_TELEPHONE(phone), FALSE);
  
  card->priv->phone_iter = card->priv->phones;
  for (; card->priv->phone_iter; 
       card->priv->phone_iter = card->priv->phone_iter->next)
    {
      if (card->priv->phone_iter && 
	  (card->priv->phone_iter->data == phone))
	{
	  card->priv->phones = g_list_remove_link(card->priv->phones,
						  card->priv->phone_iter);
      
	  r_telephone_free(phone);
	  g_list_free_1(card->priv->phone_iter);
	  card->priv->phone_iter = NULL;

	  return TRUE;
	}
    }

  return FALSE;  
}


/**
 * r_card_replace_telephone
 * @card: a #RCard
 * @old: a #RTelephone
 * @new: a #RTelephone
 * 
 * replace the old telephone with new one. 
 * On success, old telephone is deleted.
 *
 * returns: %TRUE if telephone has been successfully replaced , 
 * %FALSE otherwihe. 
 */
gboolean
r_card_replace_telephone (RCard* card, RTelephone* old,	RTelephone* new)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);
  g_return_val_if_fail(IS_R_TELEPHONE(old), FALSE);
  g_return_val_if_fail(IS_R_TELEPHONE(new), FALSE);

  card->priv->phone_iter = card->priv->phones;
  for (; card->priv->phone_iter; 
       card->priv->phone_iter = card->priv->phone_iter->next)
    {
      if (card->priv->phone_iter && 
	  (card->priv->phone_iter->data == old))
	{
	  card->priv->phone_iter->data = new;
	  r_telephone_free(old);
	  card->priv->phone_iter = NULL;
	  
	  return TRUE;
	}
    }

  return FALSE;  
}



/**
 * r_card_get_telephone
 * @card: a #RCard
 *
 * get the first item of card's telephones list. User must cast to
 * a #RTelephone.
 * 
 * Returns: a gpointer 
 */
gpointer
r_card_get_telephone (RCard* card)
{
  RTelephone* ret = NULL;

  g_return_val_if_fail(IS_R_CARD(card), NULL);

  if (card->priv->phone_iter)
    ret = card->priv->phone_iter->data;
  if (IS_R_TELEPHONE(ret))
    return ret;
  
  return NULL;  
}



/**
 * r_card_get_next_telephone
 * @card: a #RCard
 *
 * get the next item in card's telephones list. User must cast to
 * a #RTelephone.
 * 
 * Returns: a gpointer 
 */
gpointer
r_card_get_next_telephone (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  card->priv->phone_iter = g_list_next(card->priv->phone_iter);
  if (card->priv->phone_iter)
    return card->priv->phone_iter->data;
  else  /* iterator is NULL, reset iterator */
    card->priv->phone_iter = card->priv->phones;
  
  return NULL;  
}


/**
 * r_card_get_prev_telephone
 * @card: a #RCard
 *
 * get the previous item in card's telephones list. User must cast to
 * a #RTelephone. 
 * 
 * Returns: a gpointer 
 */
gpointer
r_card_get_prev_telephone (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  card->priv->phone_iter= g_list_previous(card->priv->phone_iter);
  if (card->priv->phone_iter)
    return card->priv->phone_iter->data;
  else
    card->priv->phone_iter = card->priv->phones;

  return NULL;  
}



/**
 * r_card_reset_telephone
 * @card: a #RCard
 *
 * set the private telephones iterartor to the first one
 */
void 
r_card_reset_telephone (RCard* card)
{
  g_return_if_fail(IS_R_CARD(card));
  
  card->priv->phone_iter = card->priv->phones;
}


/**
 * r_card_foreach_telephone
 * @card: a #RCard
 * @func: the #RFunc that will be called on each group in the card
 * @user_data: user's data to pass to func
 *
 * User's function func, will be called foreach telephone in card.
 * func must have two parameter: the telephone (automatically passed) and 
 * the user_data
 */
void
r_card_foreach_telephone (RCard* card, RFunc func, gpointer user_data)
{
  gpointer phone;
  
  r_card_reset_telephone(card);
  phone = r_card_get_telephone(card);
  for(; phone; phone = r_card_get_next_telephone(card))
    func(user_data, phone);
}

/**
 * r_card_add_ref
 * @card: a #RCard
 * @ref: a #RRef
 *
 * add a #RRef to the given #RCard card
 */
void 
r_card_add_ref (RCard* card, RRef* ref)
{
  g_return_if_fail(IS_R_CARD(card));
  g_return_if_fail(IS_R_REF(ref));

  card->priv->refs = g_list_append(card->priv->refs, ref);
  if (!card->priv->refs_iter)
    card->priv->refs_iter = card->priv->refs;
}


/**
 * r_card_del_ref
 * @card: a #RCard
 * @ref: a #RRef
 *
 * remove a #RRef from card 
 * returns: %TRUE if reference is successfully deleted, %FALSE otherwise
 */
gboolean 
r_card_del_ref (RCard* card, RRef* ref)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);  
  g_return_val_if_fail(IS_R_REF(ref), FALSE);
  
  card->priv->refs_iter = card->priv->refs;
  for (; card->priv->refs_iter; 
       card->priv->refs_iter = card->priv->refs_iter->next)
    {
      if (card->priv->refs_iter->data == ref)
	{  
	  card->priv->refs = g_list_remove_link(card->priv->refs,
						card->priv->refs_iter);
	  
	  r_ref_free(ref);
	  g_list_free_1(card->priv->refs_iter);
	  card->priv->refs_iter = NULL;

	  return TRUE;
	}
    }

  return FALSE;
}



/**
 * r_card_has_refs
 * @card: a #RCard
 *
 * Check if card has references to other cards
 *
 * returns: %TRUE if has references, %FALSE otherwise
 */
gboolean 
r_card_has_refs (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), FALSE);

  return (card->priv->refs != NULL); 
}

/**
 * r_card_get_ref
 * @card: a #RCard
 *
 * get the first #RRef
 *
 * returns: a gpointer or NULL if card hasn't refs
 */
gpointer 
r_card_get_ref (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);

  if (card->priv->refs_iter)
    return card->priv->refs_iter->data;
  
  return NULL;
}


/**
 * r_card_get_next_ref
 * @card: a #RCard
 *
 * get the next ref 
 *
 * returns: a gpointer or NULL
 */
gpointer 
r_card_get_next_ref (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);

  card->priv->refs_iter = g_list_next(card->priv->refs_iter);
  if (card->priv->refs_iter)
    return card->priv->refs_iter->data;
  else
    card->priv->refs_iter = card->priv->refs;

  return NULL;
}


/**
 * r_card_get_prev_ref 
 * @card: a #RCard
 *
 * get the previous ref 
 *
 * returns: a gpointer or NULL 
 */
gpointer 
r_card_get_prev_ref (RCard* card)
{
  g_return_val_if_fail(IS_R_CARD(card), NULL);

  card->priv->refs_iter = g_list_previous(card->priv->refs_iter);
  if (card->priv->refs_iter)
    return card->priv->refs_iter->data;
  else
    card->priv->refs_iter = card->priv->refs;

  return NULL;
}


/**
 * r_card_reset_ref
 * @card: a #RCard
 *
 * reset the private refs's iterator to ref head's list
 */
void     
r_card_reset_ref (RCard* card)
{
  g_return_if_fail(IS_R_CARD(card));

  card->priv->refs_iter = card->priv->refs;
}


/**
 * r_card_foreach_ref
 * @card: a #RCard
 * @func: the #RFunc that will be called on each group in the card
 * @user_data: user's data to pass to func
 *
 * User's function func, will be called foreach ref in card.
 * func must have two parameter: the ref (automatically passed) and 
 * the user_data
 */
void
r_card_foreach_ref (RCard* card, RFunc func, gpointer user_data)
{
  gpointer ref;
  
  r_card_reset_ref(card);
  ref = r_card_get_ref(card);
  for(; ref; ref = r_card_get_next_ref(card))
    func(user_data, ref);  
}


/**
 * r_card_get_group_owner
 * @card: a #RCard
 * @group: a #RGroup
 *
 * get the owner of the given group. Caller does not free the returned value
 * 
 * Returns: a gchar*, "r" if given group if a default group, 
 * "user" if given group was build by user
 */
gchar*   
r_card_get_group_owner (RCard* card, RGroup* group)
{
  gchar* owner;

  g_return_val_if_fail(IS_R_CARD(card), NULL);
  g_return_val_if_fail(IS_R_GROUP(card), NULL);
  
  g_object_get(G_OBJECT(group), "group-owner", &owner, NULL);
  
  return owner;
}



/**
 * r_card_get_home_page
 * @card: a #RCard
 *
 * get the first web url in net addresses list. 
 * Caller does not free the returned value
 * 
 * Returns: a gchar*
 */
gchar* 
r_card_get_home_page (RCard* card)
{
  gpointer net;

  g_return_val_if_fail(IS_R_CARD(card), "");
  
  net = r_card_get_net_address(card);
  for (; net; net = r_card_get_next_net_address(card))
    {
      RNetAddressType type = R_NET_ADDRESS_UNKNOWN;
      gchar* url;

      g_object_get(R_NET_ADDRESS(net), "url", &url, "url-type", &type, NULL);
      if (type == R_NET_ADDRESS_WEB)
	{
	  r_card_reset_net_address(card);
		  
	  return url;
	}
    }

  return "";
}


/**
 * r_card_get_email
 * @card: a #RCard
 *
 * get the first email in net addresses list. 
 * Caller does not free the returned value
 * 
 * Returns: a gchar*
 */
gchar* r_card_get_email (RCard* card)
{
  gpointer net;

  g_return_val_if_fail(IS_R_CARD(card), "");
  
  net = r_card_get_net_address(card);
  for (; net; net = r_card_get_next_net_address(card))
    {
      RNetAddressType type = R_NET_ADDRESS_UNKNOWN;
      gchar* url;

      g_object_get(R_NET_ADDRESS(net), "url", &url, "url-type", &type, NULL);
      if (type == R_NET_ADDRESS_EMAIL)
	{
	  r_card_reset_net_address(card);

	  return url;
	}
    }

  return "";
}


/**
 * r_card_get_irc
 * @card: a #RCard
 *
 * get the first irc in net addresses list. 
 * Caller does not free the returned value
 * 
 * Returns: a gchar*
 */
gchar* 
r_card_get_irc (RCard* card)
{
  gpointer net;

  g_return_val_if_fail(IS_R_CARD(card), "");
  
  net = r_card_get_net_address(card);
  for (; net; net = r_card_get_next_net_address(card))
    {
      RNetAddressType type = R_NET_ADDRESS_UNKNOWN;
      gchar* url;

      g_object_get(net, "url", &url, "url-type", &type, NULL);
      if ((type >= R_NET_ADDRESS_IRC) && 
	  (type <= R_NET_ADDRESS_IRC_MSN))
	{
	  r_card_reset_net_address(card);
	  
	  return url;
	}
    }

  return "";  
}


gpointer 
r_card_find_address (RCard* card, RAddressType type)
{
  gpointer address;

  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  address = r_card_get_address(card);
  for (; address; address = r_card_get_next_address(card))
    {
      RAddressType adstype;
      
      g_object_get(R_ADDRESS(address), "address-type", &adstype, NULL);
      if (adstype == type)
	return address;
    }
  
  return NULL;
}


gpointer 
r_card_find_net_address (RCard* card, RNetAddressType type)
{
  gpointer net;

  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  net = r_card_get_net_address(card);
  for (; net; net = r_card_get_next_net_address(card))
    {
      RNetAddressType nettype;
      
      g_object_get(R_NET_ADDRESS(net), "url-type", &nettype, NULL);
      if (nettype == type)
	return net;
    }
  
  return NULL;  
}

gpointer 
r_card_find_telephone (RCard* card, RTelephoneType type)
{
  gpointer telephone;

  g_return_val_if_fail(IS_R_CARD(card), NULL);
  
  telephone = r_card_get_telephone(card);
  for (; telephone; telephone = r_card_get_next_telephone(card))
    {
      RTelephoneType teltype;
      
      g_object_get(R_TELEPHONE(telephone), "telephone-type", &teltype, NULL);
      if (teltype == type)
	return telephone;
    }
  
  return NULL;
}


/* eof */
