use std::path::Path;
use std::path::PathBuf;

use clap::CommandFactory;
use clap_complete::Shell;

pub mod cli {
    include!("src/cli/mod.rs");
}

// To avoid adding a build dependency on sequoia-openpgp, we mock the
// bits of openpgp that the CLI module uses.
pub mod openpgp {
    #[derive(Clone, Debug)]
    pub struct KeyHandle {
    }

    impl From<&str> for KeyHandle {
        fn from(_: &str) -> KeyHandle {
            KeyHandle {}
        }
    }
}

fn main() {
    git_version();
    completions();
    man_pages();
}

fn git_version() {
    // Emit the "cargo:" instructions including
    // "cargo:rustc-env=VERGEN_GIT_DESCRIBE=<DESCRIBE>".
    //
    // If the source directory does not contain a git repository,
    // e.g., because the code was extracted from a tarball, this
    // produces an `Error` result, which we ignore, and
    // `VERGEN_GIT_DESCRIBE` is not set.  That's okay, because we are
    // careful to only use `VERGEN_GIT_DESCRIBE` if it is actually
    // set.
    let _ = vergen::EmitBuilder::builder()
        // VERGEN_GIT_DESCRIBE
        .git_describe(/* dirty */ true, /* tags */ false, None)
        // Don't emit placeholder values for the git version if the
        // git repository is not present.
        .fail_on_error()
        .emit();
}

fn completions() {
    // Generate shell completions
    let outdir = match std::env::var_os("CARGO_TARGET_DIR") {
        None => {
            println!("cargo:warning=Not generating completion files, \
                      environment variable CARGO_TARGET_DIR not set");
            return;
        }
        Some(outdir) => outdir,
    };

    std::fs::create_dir_all(&outdir).unwrap();

    let mut cli = cli::Cli::command();

    for shell in &[Shell::Bash, Shell::Fish, Shell::Zsh,
                   Shell::PowerShell, Shell::Elvish] {
        let path = clap_complete::generate_to(
            *shell, &mut cli, "sq-git", &outdir).unwrap();
        println!("cargo:warning=generated completion file {:?}", path);
    };
}

fn man_pages() {
    // Man page support.
    let outdir = match std::env::var_os("CARGO_TARGET_DIR") {
        None => {
            println!("cargo:warning=Not generating man pages, \
                      environment variable CARGO_TARGET_DIR not set");
            return;
        }
        Some(outdir) => PathBuf::from(outdir),
    };

    std::fs::create_dir_all(&outdir).unwrap();

    let cli = cli::Cli::command();

    let man = clap_mangen::Man::new(cli.clone());
    let mut buffer: Vec<u8> = Default::default();
    man.render(&mut buffer).unwrap();

    let filename = outdir.join("sq-git.1");
    println!("cargo:warning=writing man page to {}", filename.display());
    std::fs::write(filename, buffer).unwrap();

    fn doit(outdir: &Path, prefix: &str, command: &clap::Command) {
        let man = clap_mangen::Man::new(command.clone());
        let mut buffer: Vec<u8> = Default::default();
        man.render(&mut buffer).unwrap();

        let filename = outdir.join(format!("{}-{}.1", prefix, command.get_name()));
        println!("cargo:warning=writing man page to {}", filename.display());
        std::fs::write(filename, buffer).unwrap();

        for sc in command.get_subcommands() {
            doit(outdir,
                 &format!("{}-{}", prefix, command.get_name()),
                 sc);
        }
    }

    for sc in cli.get_subcommands() {
        doit(&outdir, "sq-git", sc);
    }
}
