/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file CSiteSurveyTab.cxx
    \author Romain BONDUE
    \date 28/07/2005 */
#include <sstream>

#include "GtkGUI.h"
#include "Parameters.h" // CCell
#include "CSiteSurveyTab.h"



namespace
{
    std::string MakeTxRateLabel (float TxRate) throw()
    {
        std::ostringstream Os;
        if (TxRate)
            Os << TxRate / 1000.0;
        else
            Os << "N/A";
        return Os.str();

    } // MakeTxRateLabel()

} // anonymous namespace



const char* const nsGUI::CSiteSurveyTab::ScanButtonLabel ("Scan");


const char* nsGUI::CSiteSurveyTab::ColumnTitle [NbColumn] = {0,
                                                 "SSID", "AP Mac Address",
                                                 "Mode", "Link",
                                                 "Signal", "Noise",
                                                 "Channel", "Rate",
                                                 "Cipher","Auth"};

const ::GType nsGUI::CSiteSurveyTab::TabType [NbColumn] = {
                                               G_TYPE_BOOLEAN, G_TYPE_STRING,
                                               G_TYPE_STRING, G_TYPE_STRING,
                                               G_TYPE_INT, G_TYPE_INT,
                                               G_TYPE_INT, G_TYPE_UINT,
                                               G_TYPE_STRING, G_TYPE_STRING,
                                               G_TYPE_STRING};


nsGUI::CSiteSurveyTab::CSiteSurveyTab (CGtkGUI* pUI) throw()
    : CSelectableRowList (NbColumn, TabType, ColumnTitle, pUI,
                          &CGtkGUI::Connect),
      m_pVBox (GTK_BOX (::gtk_vbox_new (false, 8))),
      m_pScanButton (::gtk_button_new_with_label (ScanButtonLabel)),
      m_pAddProfileButton (::gtk_button_new()),
#if GTK_CHECK_VERSION (2, 10, 0)
        /* Should work with gtk+2.6 too, but it doesn't : an ugly "gtk-connect"
         * is displayed instead. Ditto with gtk+2.8 on some distribution... */
      m_pConnectButton (::gtk_button_new_from_stock (GTK_STOCK_CONNECT)),
#else
      m_pConnectButton (::gtk_button_new_with_label ("Connect")),
#endif
      m_IsScanning (false)
{
    ::GtkBox* pHBox (GTK_BOX (::gtk_hbox_new (false, 2)));
    ::gtk_box_pack_start_defaults (pHBox, ::gtk_image_new_from_stock
                                (GTK_STOCK_ADD, GTK_ICON_SIZE_SMALL_TOOLBAR));
    ::gtk_box_pack_start_defaults (pHBox, ::gtk_label_new ("Add profile"));
    ::gtk_container_add (GTK_CONTAINER (m_pAddProfileButton),
                         GTK_WIDGET (pHBox));
    ::gtk_widget_set_sensitive (m_pAddProfileButton, false);
    ::g_signal_connect (m_pAddProfileButton, "clicked",
                        G_CALLBACK (AddProfileButtonCallBack), this);

    ::g_signal_connect (m_pScanButton, "clicked",
                        G_CALLBACK (ScanButtonCallBack), this);
    ::gtk_container_set_border_width (GTK_CONTAINER (m_pVBox), 6);
    ::gtk_box_pack_start_defaults (m_pVBox,
                                CSelectableRowList::operator ::GtkWidget* ());

    ::g_signal_connect (m_pConnectButton, "clicked",
                        G_CALLBACK (ConnectButtonCallBack), this);
    ::gtk_widget_set_sensitive (m_pConnectButton, false);

    pHBox = GTK_BOX (::gtk_hbutton_box_new()); // I reuse the pointer.
    ::gtk_box_set_spacing (pHBox, 4);
    ::gtk_button_box_set_layout (GTK_BUTTON_BOX (pHBox), GTK_BUTTONBOX_END);
    ::gtk_container_add (GTK_CONTAINER (pHBox), m_pAddProfileButton);
    ::gtk_container_add (GTK_CONTAINER (pHBox), m_pConnectButton);
    ::gtk_container_add (GTK_CONTAINER (pHBox), m_pScanButton);

    ::gtk_box_pack_start (m_pVBox, GTK_WIDGET (pHBox), false, false, 0);

} // CSiteSurveyTab()


const nsWireless::CCell* nsGUI::CSiteSurveyTab::AddCells
                   (const CellVec_t& CellVec,
                    const nsWireless::CMacAddress& CurrentAPMacAddr) throw()
{
    const nsWireless::CCell* pCurrentCell (0);
    for (unsigned i (0) ; i < CellVec.size() ; ++i)
    {
        if (!pCurrentCell && CellVec [i].GetAPMacAddress() == CurrentAPMacAddr)
        {
            pCurrentCell = &CellVec [i];
            SetData (i, ButtonPos, true);
        }
        else SetData (i, ButtonPos, false);
        SetData (i, SSIDPos, CellVec [i].GetSSID());
        SetData (i, APMacPos, CellVec [i].GetAPMacAddress().GetStr());
        SetData (i, ModePos, nsWireless::GetModeName (CellVec [i].GetMode()));
        SetData (i, LinkQPos, CellVec [i].GetQuality().GetLinkQuality());
        SetData (i, SignalLPos, CellVec [i].GetQuality().GetSignalLevel());
        SetData (i, NoiseLPos, CellVec [i].GetQuality().GetNoiseLevel());
        SetData (i, ChannelPos, CellVec [i].GetChannel());
        SetData (i, RatePos, MakeTxRateLabel (CellVec [i].GetTxRate()));
        SetData (i, EncryptPos, nsWireless::GetEncryptName
                                (CellVec [i].GetEncryptionD().GetEncrypt()));
        SetData (i, AuthPos, nsWireless::GetAuthName
                                    (CellVec [i].GetEncryptionD().GetAuth()));
    }
    return pCurrentCell;

} // Refresh()


void nsGUI::CSiteSurveyTab::ScanButtonCallBack (GtkButton* pButton,
                                                CSiteSurveyTab* This) throw()
{
    if (!This->m_IsScanning) This->GetUI()->Scan();
    else This->GetUI()->StopScanning();

} // ScanButtonCallBack()


void nsGUI::CSiteSurveyTab::ScanningStarted () throw()
{
    m_IsScanning = true;
    ::gtk_widget_set_sensitive (m_pAddProfileButton, false);
    ::gtk_widget_set_sensitive (m_pConnectButton, false);
    Clear();
    ::gtk_button_set_label (GTK_BUTTON (m_pScanButton), "Stop");

} // ScanningStarted()


void nsGUI::CSiteSurveyTab::RowSelectedCallBack () throw()
{
    ::gtk_widget_set_sensitive (m_pAddProfileButton, true);
    ::gtk_widget_set_sensitive (m_pConnectButton, true);

} // RowSelectedCallBack()


void nsGUI::CSiteSurveyTab::RowUnselectedCallBack () throw()
{
    ::gtk_widget_set_sensitive (m_pAddProfileButton, false);
    ::gtk_widget_set_sensitive (m_pConnectButton, false);

} // RowUnselectedCallBack()


void nsGUI::CSiteSurveyTab::AddProfileButtonCallBack (::GtkButton*,
                                                      CSiteSurveyTab* This)
                                                                        throw()
{
    This->GetUI()->NewProfile (This->GetSelectedRow());

} // AddProfileButtonCallBack()


void nsGUI::CSiteSurveyTab::ConnectButtonCallBack (::GtkButton*,
                                               CSiteSurveyTab* This) throw()
{
    This->RowTriggered();

} // ConnectButtonCallBack()
